/*
 * Decompiled with CFR 0.152.
 */
package fathom.security;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Singleton;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import fathom.Service;
import fathom.authc.AuthenticationException;
import fathom.authc.AuthenticationToken;
import fathom.conf.Settings;
import fathom.exception.FathomException;
import fathom.realm.Account;
import fathom.realm.CachingRealm;
import fathom.realm.Realm;
import fathom.utils.ClassUtil;
import fathom.utils.RequireUtil;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class SecurityManager
implements Service {
    private static final Logger log = LoggerFactory.getLogger(SecurityManager.class);
    @Inject
    private Injector injector;
    @Inject
    private Settings settings;
    private Collection<Realm> allRealms;
    private Cache<AuthenticationToken, Account> accountCache;

    public int getPreferredStartOrder() {
        return 50;
    }

    public void start() {
        Config config;
        this.allRealms = Collections.emptyList();
        URL configFileUrl = this.settings.getFileUrl("security.configurationFile", "classpath:conf/realms.conf");
        if (configFileUrl == null) {
            throw new FathomException("Failed to find Security Realms file '{}'", new Object[]{this.settings.getString("security.configurationFile", "classpath:conf/realms.conf")});
        }
        try (InputStreamReader reader = new InputStreamReader(configFileUrl.openStream());){
            config = ConfigFactory.parseReader((Reader)reader);
            log.info("Configured Security Realms from '{}'", (Object)configFileUrl);
        }
        catch (IOException e) {
            throw new FathomException((Throwable)e, "Failed to parse Security Realms file '{}'", new Object[]{configFileUrl});
        }
        this.allRealms = this.parseDefinedRealms(config);
        int cacheTtl = 0;
        if (config.hasPath("cacheTtl")) {
            cacheTtl = config.getInt("cacheTtl");
        }
        int cacheMax = 100;
        if (config.hasPath("cacheMax")) {
            cacheMax = config.getInt("cacheMax");
        }
        if (cacheTtl > 0 && cacheMax > 0) {
            this.accountCache = CacheBuilder.newBuilder().expireAfterAccess((long)cacheTtl, TimeUnit.MINUTES).maximumSize((long)cacheMax).build();
        }
        String border = Strings.padEnd((String)"", (int)68, (char)'-');
        log.info(border);
        log.info("Starting realms");
        log.info(border);
        for (Realm realm : this.allRealms) {
            log.debug("{} '{}'", (Object)realm.getClass().getName(), (Object)realm.getRealmName());
        }
        for (Realm realm : this.allRealms) {
            try {
                log.info("Starting realm '{}'", (Object)realm.getRealmName());
                realm.start();
            }
            catch (Exception e) {
                log.error("Failed to start realm '{}'", (Object)realm.getRealmName(), (Object)e);
            }
        }
    }

    public void stop() {
        this.clearCache();
        for (Realm realm : this.allRealms) {
            try {
                log.debug("Stopping realm '{}'", (Object)realm.getRealmName());
                realm.stop();
            }
            catch (Exception e) {
                log.error("Failed to stop realm '{}'", (Object)realm.getRealmName(), (Object)e);
            }
        }
    }

    public Account check(AuthenticationToken authenticationToken) {
        Account account = this.authenticate(authenticationToken);
        if (account == null) {
            throw new AuthenticationException("Invalid credentials", new Object[0]);
        }
        return account;
    }

    public Account authenticate(AuthenticationToken authenticationToken) {
        Account account3;
        if (this.accountCache != null && (account3 = (Account)this.accountCache.getIfPresent((Object)authenticationToken)) != null) {
            return account3;
        }
        Account authenticatedAccount = null;
        for (Realm realm2 : this.allRealms) {
            Account account4;
            if (!realm2.canAuthenticate(authenticationToken) || (account4 = realm2.authenticate(authenticationToken)) == null || account4.isDisabled()) continue;
            authenticatedAccount = new Account(account4.getName(), account4.getCredentials().sanitize());
            break;
        }
        if (authenticatedAccount == null) {
            return null;
        }
        Account aggregateAccount = authenticatedAccount;
        this.allRealms.stream().filter(realm -> realm.hasAccount(aggregateAccount.getUsername())).map(realm -> realm.getAccount(aggregateAccount.getUsername())).filter(account -> account.isEnabled()).forEach(account2 -> {
            if (Strings.isNullOrEmpty((String)aggregateAccount.getName())) {
                aggregateAccount.setName(account2.getName());
            }
            aggregateAccount.addEmailAddresses(account2.getEmailAddresses());
            aggregateAccount.addTokens(account2.getTokens());
            aggregateAccount.getAuthorizations().addRoles(account2.getAuthorizations().getRoles()).addPermissions(account2.getAuthorizations().getPermissions());
        });
        if (this.accountCache != null) {
            this.accountCache.put((Object)authenticationToken, (Object)aggregateAccount);
        }
        return aggregateAccount;
    }

    public void clearCache() {
        if (this.accountCache != null) {
            this.accountCache.invalidateAll();
        }
        for (Realm realm : this.allRealms) {
            if (!(realm instanceof CachingRealm)) continue;
            CachingRealm cachingRealm = (CachingRealm)realm;
            cachingRealm.clearCache();
        }
    }

    protected Collection<Realm> parseDefinedRealms(Config config) {
        ArrayList<Realm> realms = new ArrayList<Realm>();
        if (config.hasPath("realms")) {
            log.trace("Parsing Realm definitions");
            for (Config realmConfig : config.getConfigList("realms")) {
                String realmType = Strings.emptyToNull((String)realmConfig.getString("type"));
                Preconditions.checkNotNull((Object)realmType, (Object)"Realm 'type' is null!");
                if (ClassUtil.doesClassExist((String)realmType)) {
                    Class realmClass = ClassUtil.getClass((String)realmType);
                    if (!RequireUtil.allowClass((Settings)this.settings, (Class)realmClass)) continue;
                    Realm realm = (Realm)this.injector.getInstance(realmClass);
                    realm.setup(realmConfig);
                    realms.add(realm);
                    log.debug("Created '{}' named '{}'", (Object)realmType, (Object)realm.getRealmName());
                    continue;
                }
                throw new FathomException("Unknown realm type '{}'!", new Object[]{realmType});
            }
        }
        return Collections.unmodifiableList(realms);
    }
}

