package net.gdface.codegen.thrift;

import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import static com.facebook.swift.codec.metadata.DecoratorThriftServiceMetadata.javadocCommentProviderFactory;
import static com.google.common.base.Preconditions.*;

import com.facebook.swift.codec.metadata.DecoratorThriftEnumMetadata;
import com.facebook.swift.codec.metadata.DecoratorThriftStructMetadata;
import com.facebook.swift.codec.metadata.ErpcProxyReturnCode;
import com.facebook.swift.codec.metadata.ErpcType;
import com.facebook.swift.codec.metadata.ThriftCatalogWithTransformer;
import com.facebook.swift.codec.metadata.DecoratorThriftEnumMetadata.ExtensiveEnumItem;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Ordering;
import com.google.common.collect.Sets;
import com.google.common.reflect.TypeToken;

import net.gdface.codegen.thrift.ThriftServiceDecoratorConfiguration.TaskType;
import net.gdface.thrift.ThriftUtils;
import net.gdface.thrift.TypeTransformer;
import net.gdface.thrift.exception.ServiceRuntimeException;
import net.gdface.utils.BeanRelativeUtilits;
import net.gdface.utils.NameStringUtils;

/**
 * @author guyadong
 *
 */
public class ThriftSchema implements ThriftConstants {
	private final Set<ThriftStructDecorator> thriftStructDecorators = Sets.newLinkedHashSet();
	private final Set<ThriftStructDecorator> thriftStructs = Sets.newLinkedHashSet();
	private final Set<ErpcType> collectionStructs = Sets.newTreeSet();
	private final List<ThriftServiceDecorator<?>> thriftServiceDecorators = Lists.newLinkedList();
	private final Predicate<ThriftStructDecorator> exceptionFilter = new Predicate<ThriftStructDecorator>(){
		@Override
		public boolean apply(ThriftStructDecorator input) {
			return input.isException();
		}};
	private final Predicate<ThriftStructDecorator> enumFilter = new Predicate<ThriftStructDecorator>(){
		@Override
		public boolean apply(ThriftStructDecorator input) {
			return input.isEnum();
		}};
	private final Predicate<ThriftStructDecorator> beanFilter = new Predicate<ThriftStructDecorator>(){
		@Override
		public boolean apply(ThriftStructDecorator input) {
			return input.isBean();
		}};
	@SuppressWarnings("rawtypes")
	private final Predicate<ThriftServiceDecorator> findStringInServices = new Predicate<ThriftServiceDecorator>(){
		@Override
		public boolean apply(ThriftServiceDecorator input) {
			return null!=input && input.isUseStringInService();
		}};
	@SuppressWarnings("rawtypes")
	private final Predicate<ThriftServiceDecorator> findBinaryInServices = new Predicate<ThriftServiceDecorator>(){
		@Override
		public boolean apply(ThriftServiceDecorator input) {
			return null!=input && input.isUseBinaryInService();
		}};
	private final Predicate<ThriftStructDecorator> findString = new Predicate<ThriftStructDecorator>(){
		@Override
		public boolean apply(ThriftStructDecorator input) {
			return null!=input && input.isUseString();
		}};
	private final Predicate<ThriftStructDecorator> findBinary = new Predicate<ThriftStructDecorator>(){
		@Override
		public boolean apply(ThriftStructDecorator input) {
			return null!=input && input.isUseBinary();
		}};
	private final Predicate<ThriftStructDecorator> findMap = new Predicate<ThriftStructDecorator>(){
		@Override
		public boolean apply(ThriftStructDecorator input) {
			return null!=input && input.isUseMap();
		}};
	private final Predicate<ThriftStructDecorator> findSet = new Predicate<ThriftStructDecorator>(){
		@Override
		public boolean apply(ThriftStructDecorator input) {
			return null!=input && input.isUseSet();
		}};
	private final Predicate<ThriftStructDecorator> findVector = new Predicate<ThriftStructDecorator>(){
		@Override
		public boolean apply(ThriftStructDecorator input) {
			return null!=input && input.isUseVector();
		}};
	private final Predicate<ThriftStructDecorator> findException = new Predicate<ThriftStructDecorator>(){
		@Override
		public boolean apply(ThriftStructDecorator input) {
			return null!=input && input.isException();
		}};
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public ThriftSchema(Map<Class<?>, Class<?>> interfaceClasses) {
		for(Entry<Class<?>, Class<?>> entry:interfaceClasses.entrySet()){
			Class<?> interfaceClass = entry.getKey();
			Class<?> refClass = entry.getValue();
			if(refClass == ThriftServiceDecoratorConfiguration.DEF_REF_CLASS){
				refClass = null;
			}
			ThriftServiceDecorator service = new ThriftServiceDecorator(interfaceClass,refClass);
			thriftServiceDecorators.add(service);
		}
		if(!compile()){
			throw new IllegalStateException("compile fail");
		}
	}
	public List<ThriftStructDecorator> getAllStructDecorators() {
		return Lists.newArrayList(thriftStructDecorators);
	}
	public List<ThriftStructDecorator> getExceptionDecorators(){
		return Lists.newArrayList(Sets.filter(thriftStructDecorators, exceptionFilter));
	}
	public List<ThriftStructDecorator> getEnumDecorators(){
		return Lists.newArrayList(Sets.filter(thriftStructDecorators, enumFilter));
	}
	public List<ThriftStructDecorator> getBeanDecorators(){
		return Lists.newArrayList(Sets.filter(thriftStructDecorators, beanFilter));
	}
	public List<ThriftServiceDecorator<?>> getThriftServiceDecorators() {
		return thriftServiceDecorators;
	}
	private boolean compile(){
		for(ThriftServiceDecorator<?> newSourceInfo:getThriftServiceDecorators()){
			if (!newSourceInfo.compile()){
				return false;
			}
			thriftStructDecorators.addAll(newSourceInfo.getDecorateTypes());
			thriftStructs.addAll(newSourceInfo.getThriftTypes());
		}
		for(ThriftStructDecorator newSourceInfo:getAllStructDecorators()){
			if (!newSourceInfo.compile()){
				return false;
			}
			thriftStructDecorators.addAll(newSourceInfo.getDecorateTypes());
			thriftStructs.addAll(newSourceInfo.getThriftTypes());
		}
		return true;
	}
	public boolean isThriftBuildinType(Type type){		
		return ThriftUtils.isThriftBuildinType(type);
	}
	public boolean isCastType(Type type){
		return  ThriftUtils.isCastType(type);
	}
	public boolean isDecoratorType(final Type type) {
		return getDecoratorType(type) != null;
	}
	public ThriftStructDecorator getDecoratorType(final Type type) {
		Optional<ThriftStructDecorator> result = Iterables.tryFind(thriftStructDecorators, new Predicate<ThriftStructDecorator>(){

			@Override
			public boolean apply(ThriftStructDecorator input) {
				return input.getBaseClass().equals(type);
			}});
		return result.isPresent() ? result.get() : null;
	}
	public boolean isThriftStruct(Type type){
		return ThriftUtils.isThriftStruct(type);
	}
	public boolean isPrimitivefloat(Type type){
		return ThriftUtils.isPrimitivefloat(type);
	}
	public boolean isfloat(Type type){
		return ThriftUtils.isfloat(type);
	}
	public boolean isPrimitiveArray(Type type){
		return ThriftUtils.isPrimitiveArray(type);
	}
	public boolean isMap(Type type){
		return TypeToken.of(type).getRawType() == Map.class;
	}
	public boolean isList(Type type){
		return TypeToken.of(type).getRawType() == List.class;
	}
	public boolean isSet(Type type){
		return TypeToken.of(type).getRawType() == Set.class;
	}
	public boolean isCollection(Type type){
		return isList(type) || isSet(type);
	}
	public boolean isArray(Type type){
		return isList(type) || isSet(type);
	}
	public boolean isBinary(Type type){
		return type == byte[].class;
	}
	public final Type wrap(Type type){
		return TypeToken.of(type).wrap().getType();
	}
	/**
	 * 对thrift IDL 关键字(binary,list,map,set,i32...)转义,以确保生成的IDL文件中成员名和参数名与thrift关键字不冲突
	 * @param name
	 */
	public String escapeThrift(String name){
		return NameStringUtils.isThriftReserved(name)? name + "_" : name;
	}
	/**
	 * 字段名转义,避免java,thrift关键字冲突
	 * @param name
	 */
	public String escapeField(String name){
		if(NameStringUtils.isJavaReserved(name)){
			return "_" + name;
		}else {
			return escapeThrift(name);
		}
	}
	public static boolean isIsLocalMethod(Method method){
		return ThriftUtils.isIsLocalMethod(method);
	}
	public static boolean isIsLocalMethod(net.gdface.codegen.Method method){
		return ThriftUtils.isIsLocalMethod(method.delegate());
	}
	public Type[] getActualTypeArguments(Type type){
		if( type instanceof ParameterizedType){
			return ((ParameterizedType)type).getActualTypeArguments();
		}
		return new Type[0];
	}
	public Class<?> getServiceRuntimeExceptionClass(){
		return ServiceRuntimeException.class;
	}
	public Class<?> getTypeTransformerClass(){
		return TypeTransformer.class;
	}
	public String toStubType(Class<?> clazz){
		checkArgument(null !=clazz,"clazz is null");
		if(isThriftStruct(clazz) || Enum.class.isAssignableFrom(clazz) || this.isDecoratorType(clazz)){
			return ThriftServiceDecoratorConfiguration.INSTANCE.getThriftClientPackage() + "." + clazz.getSimpleName();
		}
		throw new IllegalArgumentException(String.format("%s is not thrift stub type", clazz.getName()));
	}
	public String toStubCxxType(Class<?> clazz){
		return CxxHelper.cxxNamespace("::"+toStubType(clazz),true);
	}
	/**
	 * 返回指定包名下的所有{@link ThriftStructDecorator}对象,,按类名自然排序,没有找到则返回空表
	 * @param pkg
	 */
	public List<ThriftStructDecorator> getThriftStructDecorator(final String pkg){
		ArrayList<ThriftStructDecorator> list = Lists.newArrayList(Iterables.filter(this.thriftStructDecorators, new Predicate<ThriftStructDecorator>(){
			@Override
			public boolean apply(ThriftStructDecorator input) {
				return input.getBaseClass().getPackage().getName().equals(pkg);
			}}));
		return Ordering.natural().sortedCopy(list);
	}
	/**
	 * 根据{@link ThriftStructDecorator}对象之间的依赖关系返回{@link ThriftStructDecorator}队列
	 * @param thriftStructIncluded 是否包含thrift struct 类
	 */
	protected List<ThriftStructDecorator> getGenSequence(boolean thriftStructIncluded){
		List<ThriftStructDecorator> types = getAllStructDecorators();
		if(thriftStructIncluded){
			types.addAll(thriftStructs);
			for(Class<?> structType : ThriftServiceDecoratorConfiguration.INSTANCE.getExtStructs()){
				types.add(TypeHelper.makeThriftStructDecorator(structType));
			}
		}
		List<ThriftStructDecorator> list = BeanRelativeUtilits.sortByField(types,"name");
		//List<ThriftStructDecorator> list = getAllStructDecorators();
		ArrayDeque<ThriftStructDecorator> seq = new ArrayDeque<>();
		for(ThriftStructDecorator struct:list){
			traverseMember(struct,seq, thriftStructIncluded);
		}
		return Lists.newArrayList(seq.descendingIterator());
	}
	/**
	 * 生成结构类型序列,确保类型的成员类型如果为结构类型定义在父类型之前
	 * @param parent 父类型
	 * @param seq 生成的结构类型序列
	 * @param thriftStructIncluded
	 */
	private void traverseMember(ThriftStructDecorator parent,Deque<ThriftStructDecorator> seq, boolean thriftStructIncluded){
		if(!seq.contains(parent)){
			List<ThriftStructDecorator> decoratorMembers = BeanRelativeUtilits.sortByField(parent.getDecorateTypes(),"name");
			for(ThriftStructDecorator struct:decoratorMembers){
				traverseMember(struct,seq, thriftStructIncluded);
			}
			if(thriftStructIncluded){
				List<ThriftStructDecorator> thriftMembers = BeanRelativeUtilits.sortByField(parent.getThriftTypes(),"name");
				for(ThriftStructDecorator struct:thriftMembers){
					traverseMember(struct,seq, thriftStructIncluded);
				}
			}
			seq.push(parent);
		}
	}
	/**
	 * 根据{@link ThriftStructDecorator}对象之间的依赖关系返回以package为key的 {@link ListMultimap}映射
	 * @param thriftStructIncluded 是否包含thrift struct 类
	 */
	public ListMultimap<String,Collection<ThriftStructDecorator>> getStructGenSequenceAsMultimap(boolean thriftStructIncluded){
		List<ThriftStructDecorator> seqlist = getGenSequence(thriftStructIncluded);
		seqlist.add(new ThriftStructDecorator(getServiceRuntimeExceptionClass()));
		LinkedListMultimap<String, Collection<ThriftStructDecorator>> seq = LinkedListMultimap.<String, Collection<ThriftStructDecorator>>create();
		String curPkg=null;
		List<ThriftStructDecorator> sub = null;
		for(ThriftStructDecorator struct:seqlist){
			String pkg = struct.getPackage();
			if(!pkg.equals(curPkg)){
				if(null !=curPkg){
					seq.put(curPkg, sub);
				}
				sub = Lists.newLinkedList();
				curPkg = struct.getPackage();
			}
			sub.add(struct);
		}
		if(null !=curPkg){
			seq.put(curPkg, sub);
		}
		return seq;
	}
	public boolean isUseStringInServices(){
		return Iterables.tryFind(thriftServiceDecorators, findStringInServices).isPresent();
	}
	public boolean isUseBinaryInServices(){
		return Iterables.tryFind(thriftServiceDecorators, findBinaryInServices).isPresent();
	}
	public boolean isUseStringInTypes(){
		return Iterables.tryFind(thriftStructDecorators, findString).isPresent();
	}
	public boolean isUseBinaryInTypes(){
		return Iterables.tryFind(thriftStructDecorators, findBinary).isPresent();
	}
	public boolean isUseMapInTypes(){
		return Iterables.tryFind(thriftStructDecorators, findMap).isPresent();
	}
	public boolean isUseSetInTypes(){
		return Iterables.tryFind(thriftStructDecorators, findSet).isPresent();
	}
	public boolean isUseVectorInTypes(){
		return Iterables.tryFind(thriftStructDecorators, findVector).isPresent();
	}
	public boolean isUseExceptionInTypes(){
		return Iterables.tryFind(thriftStructDecorators, findException).isPresent();
	}
	public boolean isCanMove(CxxType cxxType){
		if(null == cxxType){
			return false;			
		}
		CxxTypeMeta uiType = cxxType.getUiType();
		if(uiType.isCanMove() && !uiType.isStruct()){
			return true;
		}
		ThriftStructDecorator memSturct = getDecoratorType(cxxType.getJavaType());
		return null !=memSturct && memSturct.isHasCanMoveField();
	}
	public boolean isCommonType(final Type type){
		if(type instanceof Class<?>){
			Set<String> commonTypes = ThriftServiceDecoratorConfiguration.INSTANCE.getCommonTypes();
			return Iterables.tryFind(commonTypes, new Predicate<String>() {

				@Override
				public boolean apply(String input) {
					Class<?> clazz = (Class<?>)type;
					return clazz.getName().equals(input) 
							|| clazz.getSimpleName().equals(input)
							|| clazz.getPackage().getName().startsWith(input);
				}
			}).isPresent();
		}
		return false;
	}
	public String getProgramName(){
		return ThriftServiceDecoratorConfiguration.INSTANCE.getProgramName();
	}
	public String getPortPrefix(){
		return ThriftServiceDecoratorConfiguration.INSTANCE.getPortPrefix();
	}
	/**
	 * @param thriftServiceClasses 要设置的 thriftServiceClasses
	 * @return 当前对象
	 */
	public ThriftSchema setThriftServiceClasses(Map<Class<?>, Class<?>> thriftServiceClasses) {
		if(thriftServiceClasses != null){
			for(ThriftServiceDecorator<?> decorator:thriftServiceDecorators){
				decorator.setThriftServiceClass(thriftServiceClasses.get(decorator.getInterfaceClass())).erpcProxyInit();
			}
		}
		return this;
	}
	public DecoratorThriftEnumMetadata<?> getThriftEnumMetadata(Type type){
		if(type instanceof Class<?>){
			return (DecoratorThriftEnumMetadata<?>) ThriftCatalogWithTransformer.CATALOG.getThriftEnumMetadata((Class<?>) type);
		}
		return null;
	}
	public DecoratorThriftEnumMetadata<?> getThriftEnumMetadata(String className) throws ClassNotFoundException{
		return getThriftEnumMetadata(Class.forName(className));
	}
	public DecoratorThriftStructMetadata getThriftStructMetadata(Type type){
		return (DecoratorThriftStructMetadata) ThriftCatalogWithTransformer.CATALOG.getThriftStructMetadata(type);
	}
	public DecoratorThriftStructMetadata getThriftStructMetadata(String className) throws ClassNotFoundException{
		return getThriftStructMetadata(Class.forName(className));
	}
	/**
	 * @return collectionStructs
	 */
	public List<ErpcType> getCollectionStructs() {
		return Lists.newArrayList(collectionStructs);
	}
	/**
	 * @return collectionStructs
	 */
	public List<ErpcType> getCollectionStructs(final boolean structElement) {
		Iterable<ErpcType> filtered = Iterables.filter(collectionStructs, new Predicate<ErpcType>() {

			@Override
			public boolean apply(ErpcType input) {
				boolean isStruct = isThriftStruct(input.getValueType().getJavaType());
				return  structElement ? isStruct : !isStruct;
			}
		});
		ArrayList<ErpcType> types = Lists.newArrayList(filtered);		
		return types;
	}
	
	public int getExistsListString(){
		return Iterables.tryFind(collectionStructs, new Predicate<ErpcType>(){

			@Override
			public boolean apply(ErpcType input) {
				return input.getValueType().getProtocolType().isString();
			}}).isPresent() ? 1 : 0;
	}
	public int getExistsListBinary(){
		return Iterables.tryFind(collectionStructs, new Predicate<ErpcType>(){

			@Override
			public boolean apply(ErpcType input) {
				return input.getValueType().getProtocolType().isBinary();
			}}).isPresent() ? 1 : 0;
	}
	public ThriftSchema erpcProxyInit(){
		if(TaskType.ERPC_PROXY.equals(ThriftServiceDecoratorConfiguration.INSTANCE.getTaskType())){
			for(ThriftServiceDecorator<?> newSourceInfo:getThriftServiceDecorators()){
				collectionStructs.addAll(newSourceInfo.getCollectionTypes());
			}
			DecoratorThriftEnumMetadata.beforeExtensiveEnumItems = createBeforeExtensiveEnumItem();
			// 添加到列表头部
			thriftStructs.add(new ThriftStructDecorator(ErpcProxyReturnCode.class));
		}
		return this;
	}
	private ImmutableMap<Class<? extends Enum<?>>, List<ExtensiveEnumItem>> createBeforeExtensiveEnumItem(){
		ArrayList<ExtensiveEnumItem> items = Lists.newArrayList();
		for(ThriftStructDecorator type:thriftStructDecorators){
			Class<?> javaType = type.getBaseClass();
			if(javaType instanceof Class<?> && Exception.class.isAssignableFrom(javaType)){
				 ImmutableList<String> comment = null;
				if(javadocCommentProviderFactory != null){
					comment =  javadocCommentProviderFactory.apply(javaType).commentOfClass();
				}
				items.add(new ExtensiveEnumItem(javaType.getSimpleName(),null,comment));
			}
		}
		return ImmutableMap.<Class<? extends Enum<?>>, List<ExtensiveEnumItem>>of(ErpcProxyReturnCode.class,items);
	}
	public int getErpcForwardPort(){
		return ThriftServiceDecoratorConfiguration.INSTANCE.getErpcForwardPort();
	}
	public int getErpcProxyPort(){
		return ThriftServiceDecoratorConfiguration.INSTANCE.getErpcProxyPort();
	}
	public int getDefaultMaxLength(){
		return ThriftServiceDecoratorConfiguration.INSTANCE.getDefaultMaxLength();
	}
	public int getErrmsgMaxLength(){
		return ThriftServiceDecoratorConfiguration.INSTANCE.getErrmsgMaxLength();
	}
	public int getBinaryOutputSize(){
		return ThriftServiceDecoratorConfiguration.INSTANCE.getBinaryOutputSize();
	}
}