package net.gdface.facedb;

import static com.google.common.base.Preconditions.*;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.lang3.tuple.Pair;

import com.google.common.base.MoreObjects;
import com.google.common.base.Predicates;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import gu.sql2java.exception.RuntimeDaoException;
import net.gdface.facedb.db.FaceBean;
import net.gdface.facedb.db.FeatureBean;
import net.gdface.facedb.db.ImageBean;
import net.gdface.facedb.db.StoreBean;
import net.gdface.image.ImageErrorException;
import net.gdface.image.LazyImage;
import net.gdface.image.NotImageException;
import net.gdface.image.UnsupportedFormatException;
import net.gdface.sdk.CodeInfo;
import net.gdface.utils.FaceUtilits;
import net.gdface.utils.Judge;

/**
 * 数据库操作扩展
 * @author guyadong
 *
 */
class Dao extends BaseDao {
	final FacedbTypeTransformer trans = new FacedbTypeTransformer();
	protected static StoreBean makeStoreBean(ByteBuffer imageBytes,String md5,String encodeing){
		if(Judge.isEmpty(imageBytes)){
			return null;
		}
		if(null == md5){
			md5 = FaceUtilits.getMD5String(imageBytes);
		}
		StoreBean storeBean = new StoreBean();
		storeBean.setData(imageBytes);
		storeBean.setMd5(md5);
		if(!Strings.isNullOrEmpty(encodeing)){
			storeBean.setEncoding(encodeing);
		}
		return storeBean;
	}
	/**
	 * 创建{@link ImageBean}对象,填充图像基本信息,同时创建对应的{@link StoreBean}对象
	 * @param imageBytes
	 * @return 返回 {@link ImageBean}和{@link StoreBean}对象
	 * @throws NotImageException
	 * @throws UnsupportedFormatException
	 */
	protected static Pair<ImageBean, StoreBean> makeImageBean(ByteBuffer imageBytes) throws NotImageException, UnsupportedFormatException{
		if(Judge.isEmpty(imageBytes)){
			return null;
		}
		LazyImage image = LazyImage.create(imageBytes);
		String md5 = FaceUtilits.getMD5String(imageBytes);
		ImageBean imageBean = new ImageBean();
		imageBean.setMd5(md5);
		imageBean.setWidth(image.getWidth());
		imageBean.setHeight(image.getHeight());
		imageBean.setFormat(image.getSuffix());
		StoreBean storeBean = makeStoreBean(imageBytes, md5, null);
		return Pair.of(imageBean, storeBean);
	}
	protected static FeatureBean makeFeatureBean(ByteBuffer feature){
		return FacedbTypeTransformer.FEATUREBEAN_FUN.apply(feature);
	}
	/**
	 * 删除IMAGE表同时删除STORE表记录
	 */
	@Override
	protected int daoDeleteImage(String md5) throws RuntimeDaoException {
		daoDeleteStore(md5);
		return super.daoDeleteImage(md5);
	}
	protected int daoDeleteImage(String md5, boolean cascade) {
		List<FeatureBean> featureBeans = this.daoGetFeaturesByImageMd5(md5);
		int result = daoDeleteImage(md5);
		if(result==1){
			if(cascade){
				this.daoDeleteFeatures(featureBeans);
			}
		}
		return result;
	}
	protected int daoDeleteImages(Collection<String> imgMd5List, boolean cascade){
		int count = 0;
		imgMd5List = MoreObjects.firstNonNull(imgMd5List, Collections.<String>emptyList());
		for(String md5:imgMd5List){
			count += daoDeleteImage(md5, cascade);
		}
		return count;
	}
	protected int daoDeleteFeature(String featureId, boolean cascade) {
		List<ImageBean> imageBeans = this.daoGetImagesByFeatureId(featureId);
		int result = daoDeleteFeature(featureId);
		if(result == 1){
			if(cascade){
				this.daoDeleteImages(imageBeans);
			}
		}
		return result;
	}
	protected int daoDeleteFeatures(Collection<String> featureIdList, boolean cascade){
		int count = 0;
		featureIdList = MoreObjects.firstNonNull(featureIdList, Collections.<String>emptyList());
		for(String featureId:featureIdList){
			count += daoDeleteFeature(featureId, cascade);
		}
		return count;
	}
	protected List<ImageBean> daoGetImagesByFeatureId(String featureId) {
		List<FaceBean> faceBeans = daoGetFaceBeansByFeatureMd5OnFeature(featureId);
		List<ImageBean> imageBeans = Lists.transform(faceBeans, daoCastFaceToReferencedImageBean);
		// 去除重复元素和null
		HashSet<ImageBean> imageSet = Sets.newHashSet(Iterables.filter(imageBeans, Predicates.notNull()));
		return Lists.newArrayList(imageSet);
	}
	/**
	 * 根据提供的主键ID,返回图像数据记录
	 * @param primaryKey 主键
	 * @param refType 主键引用类型
	 * @return 图像数据记录
	 */
	protected ImageBean daoGetImage(String primaryKey,RefSrcType refType) {
		checkArgument(refType!=null,"refType is null");
		if(null == primaryKey){
			return null;
		}
		
		switch (MoreObjects.firstNonNull(refType,RefSrcType.DEFAULT)) {
		case FACE:
		{
			List<ImageBean> beans = daoGetImagesByFeatureId(primaryKey);
			return beans.size() == 1 ? beans.get(0) : null;
		}
		case FEATURE:
		{
			FaceBean bean = daoGetFace(Integer.valueOf(primaryKey));
			return null == bean ? null : daoGetImage(bean.getImageMd5());
		}
		case IMAGE:
		case DEFAULT:
			return daoGetImage(primaryKey);
		default:
			return null;
		}		
	}
	protected List<FeatureBean> daoGetFeaturesByImageMd5(String imageMd5) {
		List<FaceBean> faceBeans = daoGetFaceBeansByImageMd5OnImage(imageMd5);
		List<FeatureBean> featureBeans = Lists.transform(faceBeans, daoCastFaceToReferencedFeatureBean);
		// 去除重复元素和null
		HashSet<FeatureBean> featureSet = Sets.newHashSet(Iterables.filter(featureBeans, Predicates.notNull()));
		return Lists.newArrayList(featureSet);
	}
	protected ImageBean daoAddImage(ByteBuffer imageBytes, Collection<FaceBean> impFlFacebyImgMd5) throws DuplicateRecordException{
		if(Judge.isEmpty(imageBytes)){
			return null;
		}
		Pair<ImageBean, StoreBean> pair;
		try {
			pair = makeImageBean(imageBytes);
		} catch (ImageErrorException e) {
			throw new RuntimeException(e);
		}
		daoAddStore(pair.getRight());
		return daoAddImage(pair.getLeft(),  impFlFacebyImgMd5);
	}
	protected ImageBean daoAddImage(ByteBuffer imgData, List<CodeInfo> features) throws DuplicateRecordException {
		checkArgument(null !=imgData ,"imgData is null");
		checkArgument(null != features && !features.isEmpty(),"features is nul or empty" );
		checkArgument(!Iterables.tryFind(features, Predicates.isNull()).isPresent(),"exist null element in features");
		// 这里不能用transform对象，因为transform对象迭代时不可修改
		List<FaceBean> faceBeans = Lists.newArrayList(trans.to(features,CodeInfo.class,FaceBean.class));
		List<FeatureBean> featureBeans =  Lists.newArrayList(trans.to(features,CodeInfo.class,FeatureBean.class));
		checkState(!Iterables.tryFind(featureBeans, Predicates.isNull()).isPresent(),"exist null code field in features");
		daoAddFeatures(featureBeans);
		return daoAddImage(imgData, faceBeans);
	}
	protected FeatureBean daoAddFeature(ByteBuffer feature, Map<ByteBuffer, CodeInfo> faces) throws DuplicateRecordException {
		checkArgument(null != faces && !faces.isEmpty(),"faces is null or empty");
		checkArgument(null != feature,"feature is null");
		// 这里不能用transform对象，因为transform对象迭代时不可修改
		Map<ByteBuffer, FaceBean> faceBeanMap = Maps.newHashMap(trans.to(faces, ByteBuffer.class, CodeInfo.class,ByteBuffer.class,FaceBean.class));
		FeatureBean featureBean = makeFeatureBean(feature);
		featureBean = this.daoAddFeature(featureBean,faceBeanMap.values());
		for(Entry<ByteBuffer, FaceBean> entry:faceBeanMap.entrySet()){
			daoAddImage(entry.getKey(),Arrays.asList(entry.getValue()));
		}
		return featureBean;
	}
}
