package net.gdface.facedb;

import net.gdface.exception.NotFoundBeanException;
import net.gdface.facedb.db.FeatureBean;
import net.gdface.facedb.db.ImageBean;
import net.gdface.image.ImageErrorException;
import net.gdface.sdk.BaseFaceApi;
import net.gdface.sdk.CodeInfo;
import net.gdface.sdk.CompareResult;
import net.gdface.sdk.FaceApi;
import net.gdface.sdk.NotFaceDetectedException;
import net.gdface.sdk.fse.FeatureSe;
import net.gdface.utils.FaceUtilits;
import static net.gdface.utils.SimpleLog.log;

import static com.google.common.base.Preconditions.*;

import java.util.Arrays;
import java.util.List;

import com.google.common.base.MoreObjects;

/**
 * 实现{@link FaceDb}人脸识别算法相关方法
 * @author guyadong
 *
 */
public class FacedbFull extends  FacedbLocal {
	private final FaceApi faceApi;
	private FeatureDbSearchEngine fdbse;
	private boolean multiFaceFeature;
	public FacedbFull(FaceApi faceApi) {
		super();
		this.faceApi = checkNotNull(faceApi,"faceApi is null");
		this.multiFaceFeature = Boolean.valueOf(MoreObjects.firstNonNull(this.faceApi.sdkCapacity().get("MULTI_FACE_FEATURE"),"false"));
		FeatureSe fse = BaseFaceApi.getFeatureSeInstance(faceApi);
		if(null != fse){
			fdbse= new FeatureDbSearchEngine(fse);
			fdbse.init(dao);
			capacity.put(C_SUPPORT_SEARCH, Boolean.TRUE.toString());
		}else{
			log("WARN:DISABLE feature search engine");
		}
		// 装入 FaceApi能力描述
		capacity.putAll(faceApi.sdkCapacity());
	}

	public FaceApi getFaceApi() {
		return faceApi;
	}

	@Override
	protected void checkMulti() {
		if(!multiFaceFeature){
			throw new UnsupportedOperationException();
		}
	}

	@Override
	protected void checkNotMulti() {
		if(multiFaceFeature){
			throw new UnsupportedOperationException();
		}
	}

	@Override
	public ImageBean addImageIfAbsent(byte[] imgData, CodeInfo code, double similarty)
			throws NotFaceDetectedException, ImageErrorException {
		checkNotMulti();
		if(null == code){
			code = getFaceApi().detectCenterFace(imgData);
		}
		if(null == code.getCode()){
			code = getFaceApi().getCodeInfo(imgData, 1, new CodeInfo[]{code})[0];
		}
		SearchResult result = searchFeatures(code.getCode(), similarty, 1, null);
		if (result.getFeatureIds().length > 0) {
			// 返回相似度最高的记录
			return getImageByFeatureId(result.getFeatureIds()[0]);
		}else{
			try {
				// 没有相似记录时添加到数据库
				return addImage( imgData, Arrays.asList(code));
			} catch (DuplicateRecordException e) {
				return getImage(FaceUtilits.getMD5String(imgData));
			}
		}
	}
	@Override
	public double compareFeature(String featureId, byte[] feature) throws NotFoundBeanException {
		byte[] f1 = FaceUtilits.getBytesInBuffer(getFeatureChecked(featureId).getFeature());
		return getFaceApi().compareCode(f1,feature);
	}

	@Override
	public double compareFeature(String featureId1, String featureId2) throws NotFoundBeanException {
		byte[] f1 = FaceUtilits.getBytesInBuffer(getFeatureChecked(featureId1).getFeature());
		byte[] f2 = FaceUtilits.getBytesInBuffer(getFeatureChecked(featureId2).getFeature());
		return getFaceApi().compareCode(f1,f2);
	}

	@Override
	public double[] compareFeatures(String featureId, CodeInfo[] features) throws NotFoundBeanException {
		byte[] f1 = FaceUtilits.getBytesInBuffer(getFeatureChecked(featureId).getFeature());
		return getFaceApi().compareCodes(f1, features);
	}

	@Override
	public double[] compareFaces(String featureId, byte[] imgData, CodeInfo[] facePos)
			throws NotFoundBeanException, NotFaceDetectedException {
		checkArgument(null !=facePos,"facePos is null");
		if(facePos.length == 0){
			return new double[0];
		}
		facePos = getFaceApi().getCodeInfo(imgData, facePos.length, facePos);
		return compareFeatures(featureId,facePos);
	}

	@Override
	public CompareResult detectAndCompareFaces(String featureId, byte[] imgData, int faceNum)
			throws NotFoundBeanException, ImageErrorException, NotFaceDetectedException {
		FeatureBean featureBean = getFeatureChecked(featureId);
		return getFaceApi().compareFaces(FaceUtilits.getBytesInBuffer(featureBean.getFeature()), imgData, faceNum);
	}

	@Override
	public SearchResult detectAndSearchFaces(byte[] imgData, double similarty, int rows, String where) throws ImageErrorException, NotFaceDetectedException {
		CodeInfo code = getFaceApi().detectAndGetCodeInfo(imgData, 1)[0];
		return searchFeatures(code.getCode(), similarty, rows, where);
	}

	@Override
	public ImageBean detectAndAddFeatures(byte[] imgData, int faceNum)
			throws DuplicateRecordException, ImageErrorException, NotFaceDetectedException {
		checkNotMulti();
		List<CodeInfo> codes = Arrays.asList(getFaceApi().detectAndGetCodeInfo(imgData, faceNum));
		return addImage(imgData,codes);
	}

	@Override
	public SearchResult searchFaces(byte[] imgData, CodeInfo facePos, double similarty, int rows, String where)
			throws NotFaceDetectedException, ImageErrorException {
		checkArgument(null != facePos,"facePos is null");
		if(null == facePos.getCode()){
			facePos = getFaceApi().getCodeInfo(imgData, 1,new CodeInfo[] {facePos})[0];
		}
		return this.searchFeatures(facePos.getCode(), similarty, rows, where);
	}

	@Override
	public SearchResult searchFeatures(byte[] feature, double similarty, int rows, String where) {
		if(null != fdbse){
			return fdbse.searchFeatures(feature, similarty, rows);
		}else{
			return super.searchFeatures(feature, similarty, rows, where);
		}
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("FacedbFull [faceApi=");
		builder.append(faceApi);
		builder.append(", fse=");
		builder.append(fdbse);
		builder.append("]");
		return builder.toString();
	}
}
