package net.gdface.facedb;

import net.gdface.facedb.db.FeatureBean;
import net.gdface.facedb.db.IFeatureManager;
import gu.sql2java.Managers;
import gu.sql2java.TableListener;
import gu.sql2java.TableManager.Action;
import gu.sql2java.exception.RuntimeDaoException;
import net.gdface.sdk.fse.CodeBean;
import net.gdface.sdk.fse.FeatureSe;
import net.gdface.sdk.fse.FeatureSeDecorator;
import net.gdface.utils.FaceUtilits;
import net.gdface.utils.SampleLog;

import java.util.concurrent.atomic.AtomicInteger;

import static com.google.common.base.Preconditions.*;
/**
 * 基于数据库的特征搜索引擎
 * @author guyadong
 *
 */
public class FeatureDbSearchEngine {
	private final FeatureSeDecorator fse;
	/**
	 * 
	 * @param fse {@link FeatureSe}引擎实例
	 */
	public FeatureDbSearchEngine(FeatureSe fse) {
		this.fse= FeatureSeDecorator.makeDecorator(checkNotNull(fse,"fse is null"));
	}

	/**
	 * 加载所有FEATURE表特征到feature_se
	 */
	private void loaddb(Dao dao){
		SampleLog.log("Loading FD_FEATURE Table from Database into memory...");
		final AtomicInteger rowCount = new AtomicInteger(0);
		checkNotNull(dao,"dao is null").daoLoadFeatureByWhereForAction(null, new Action<FeatureBean>(){
			@Override
			public void call(FeatureBean bean) throws RuntimeDaoException {
				addFeatureBean(bean);	
				showProgress(rowCount.incrementAndGet());
			}
		});
	}
	/**
	 * 搜索引擎初始化
	 * @param dao Dao对象
	 */
	public void init(Dao dao){
		loaddb(dao);
		Managers.instanceOf(IFeatureManager.class).registerListener(featureTableListener);
	}
	/** 显示完成进度 */
	private void showProgress(int c){
		// 显示完成进度
		if(0==c%100000){
			System.out.printf("*");
			if(0==(c%1000000)){							
				if(0==(c%=10000000)){
					System.out.println();
				}else{
					System.out.print(" ");
				}
			}
		}
	}
	private final TableListener<FeatureBean> featureTableListener = 
		new TableListener.Adapter<FeatureBean>(){	
			@Override
			public void afterInsert(FeatureBean bean) throws RuntimeDaoException {								
				addFeatureBean(bean);
			}
			@Override
			public void afterDelete(FeatureBean bean) throws RuntimeDaoException {
				fse.removeFeature(bean.getMd5());
			}		
		};
	private void addFeatureBean(FeatureBean bean){
		checkArgument(null != bean,"bean is null");
		fse.addFeature(
				bean.getMd5(),
				FaceUtilits.getBytesInBuffer(bean.getFeature()), null);
	}
	/**
	 * @param feature  要比对的特征码
	 * @param similarty 相似度阀值
	 * @param rows 最多返回的记录数目
	 * @return 返回包含相似度(降序)的结果数组,如果没有查到匹配的记录则返回空数组
	 * @see FeatureSe#searchCode(byte[], double, int)
	 */
	public SearchResult searchFeatures(byte[] feature, double similarty, int rows) {
		
		CodeBean[] codeBeans = fse.searchCode(checkNotNull(feature,"feature is null"),similarty,rows);
		String[] featureIds = new String[codeBeans.length];
		Double[] similartys = new Double[codeBeans.length];
		for(int i=0;i<codeBeans.length;++i){
			featureIds[i]=FaceUtilits.getMD5String(codeBeans[i].id);
			similartys[i] = codeBeans[i].similarity;
		}
		SearchResult result = new SearchResult();
		result.setFeatureIds(featureIds);
		result.setSimilartys(similartys);
		return result;
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("FeatureDbSearchEngine [fseClassName=");
		builder.append(fse);
		builder.append("]");
		return builder.toString();
	}
}
