package net.gdface.mtfsdk;

import static net.gdface.mtfsdk.MtfAndroidConfigProvider.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;

import android.graphics.Bitmap;
import net.gdface.image.ImageErrorException;
import net.gdface.image.LazyImage;
import net.gdface.license.LicenseManager;
import net.gdface.mtfsdk.MtfAndroidArmBridge.RuntimeParam;
import net.gdface.sdk.BaseFaceApiLocal;
import net.gdface.sdk.CodeInfo;
import net.gdface.sdk.FaceApiGenericDecorator;
import net.gdface.sdk.NotFaceDetectedException;
import net.gdface.sdk.fse.FeatureSe;
import net.gdface.utils.Assert;
import net.gdface.utils.ShareLock;

/**
 * MTFSDK android/arm平台 {@link net.gdface.sdk.FaceApi}接口实现类(线程安全)<br>
 * 基于commons-pools将非线程安全的{@link MtfAndroidArmBridge}实例用资源池{@link GenericObjectPool}管理,
 * 每次调用前从资源池申请一个{@link MtfAndroidArmBridge}实例来使用，用完后归还资源池，
 * 以实现对{@link MtfAndroidArmBridge}实例线程安全的调用
 * @author guyadong
 *
 */
public class FaceApiMtfAndroid extends BaseFaceApiLocal {
	/**
	 * 用于人脸检测的数据缓冲区
	 */
	private static final ThreadLocal<double[]> faceBuffer = new ThreadLocal<double[]>(){

		@Override
		protected double[] initialValue() {
			return new double[FDDATA_LEN*MAX_FACE_COUNT];
		}		
	};
	private static final FaceApiMtfAndroid INSTANCE = new FaceApiMtfAndroid();
	private static final FaceApiGenericDecorator GENERIC_INSTANCE = new FaceApiGenericDecorator(INSTANCE);
	protected FaceApiMtfAndroid() {
	}
	/**
	 * 返回单实例
	 * @return the instance
	 */
	public static FaceApiMtfAndroid getInstance() {
		return INSTANCE;
	}

	/**
	 * 返回支持泛型封闭的单实例
	 * @return
	 */
	public static FaceApiGenericDecorator getGenericInstance(){
		return GENERIC_INSTANCE;
	}
	/**
	 * @return 返回SDK的授权信息管理实例
	 */
	public static LicenseManager licenseManager(){
		return MtfAndroidArmBridge.LICENSE_MANAGER;
	}
	@Override
	public final String sdkVersion() {
		return MtfAndroidArmBridge.SDK_VERSION;
	}
	@Override
	protected double nativeCompareCode(byte[] code1, byte[] code2) {
		return MtfAndroidArmBridge.FFSimilarityByte(code1,code2);
	}

	/**
	 * 人脸检测，最多返回三张人脸
	 */
	@Override
	public void nativeDetectFace(byte[] imgMatrix, int width, int height, List<CodeInfo> faceInfo) {
		nativeDetectFace(imgMatrix, width, height, faceInfo, 0);
	}
	@Override
	public byte[] nativeGetFaceFeature(byte[] imgMatrix, int width, int height, CodeInfo faceInfo) {
		NativeFaceInfo nativeFaceInfo = NativeFaceInfo.toNative(faceInfo);
		MtfAndroidArmBridge instance = applyInstance();
		try{
			setInstanceParam(instance,RuntimeParam.featureThreadNumber);
			 return instance.feaExtractByte(imgMatrix, width, height, nativeFaceInfo.nativeData);
		}finally{
			returnInstance(instance);
		}
	}

	/**
	 * 人脸检测，最多返回三张人脸
	 * @see MtfAndroidArmBridge#detect(byte[], int, int, int, double[])
	 */
	public void nativeDetectFace(byte[] imgMatrix, int width, int height, List<CodeInfo> faceInfo,int detectMod) {
		double[] buffer = faceBuffer.get();
		MtfAndroidArmBridge instance = applyInstance();
		try{
			setInstanceParam(instance,RuntimeParam.detectThreadNumber);
			setInstanceParam(instance,RuntimeParam.minFaceSize);
			int res = instance.detect(imgMatrix, width, height, detectMod, buffer);
			for(int i=0; i < res; ++i){
				faceInfo.add(new NativeFaceInfo(buffer, i*FDDATA_LEN));
			}
		}finally {
			returnInstance(instance);
		}
	}
	/**
	 * @deprecated replaced by {@link #nativeDetectMaxFace(byte[], int, int, int)}
	 */
	public NativeFaceInfo nativeDetectMaxFace(byte[] imgMatrix, int width, int height) {
		return nativeDetectMaxFace(imgMatrix, width, height, 0);	
	}
	/**
	 * 执行单人脸检测，返回最大人脸位置信息
	 * @param imgMatrix
	 * @param width
	 * @param height
	 * @param detectMod  0:用于检测图片,1:用于视频的循环检测
	 * @return
	 */
	public NativeFaceInfo nativeDetectMaxFace(byte[] imgMatrix, int width, int height, int detectMod) {
		double[] buffer = faceBuffer.get();
		MtfAndroidArmBridge instance = applyInstance();
		try{
			setInstanceParam(instance,RuntimeParam.detectThreadNumber);
			setInstanceParam(instance,RuntimeParam.minFaceSize);
			int res = instance.detectMaxFace(imgMatrix, width, height, detectMod, buffer);
			return res ==0 ? null : new NativeFaceInfo(buffer, 0);
		}finally {
			returnInstance(instance);
		}		
	}
	/**
	 * @deprecated replaced by {@link #detectMaxFace(Bitmap, int)}
	 */
	public NativeFaceInfo detectMaxFace(Bitmap bitmap) {
		return detectMaxFace(bitmap,0);
	}
	/**
	 * @see #nativeDetectMaxFace(byte[], int, int, int)
	 */
	public NativeFaceInfo detectMaxFace(Bitmap bitmap, int detectMod) {
		try {
			byte[] matrix = getMatrixData(LazyImage.create(bitmap));
			return nativeDetectMaxFace(matrix,bitmap.getWidth(),bitmap.getHeight(), detectMod);
		} catch (ImageErrorException e) {
			throw new RuntimeException(e);
		}
	}
	/**
     * 位置关联
     * 在人脸检测函数后用到，无论有无检测到人脸，均需要调用，从属于检测系列
     * @param faceNums     人脸个数
     * @param rectRate 
     * @param faceRect     人脸信息(人脸检测数据rect) 21个元素
     * @param faceRectInfo [out]返回的标记，最后一个元素为是否成功比对的标志，其他为人脸信息
     * @return 位置关联总个数
     */
	public int facePosInterrelate(int faceNums, double rectRate, double[] faceRect, double[] faceRectInfo){
		MtfAndroidArmBridge instance = applyInstance();
		try{
			return instance.facePosInterrelate(faceNums, rectRate, faceRect, faceRectInfo);
		}finally {
			returnInstance(instance);
		}
	}
	/**
	 * 从检测原始数据获取人脸角度
	 * @param BGR 
	 * @param width
	 * @param height
	 * @param rect 人脸检测原始数据
	 * @param rectinfo [out] leng>=3,不可为null,返回3个元素 1: roll 2: yaw 3: pitch
	 * @return
	 * @deprecated 人脸检测原始数据中已经包含了人脸角度,不再需要此函数
	 */
	public int getDetectInfo(byte[] BGR, int width, int height, double[] rect, double[] rectinfo){
		MtfAndroidArmBridge instance = applyInstance();
		try{
			return instance.getDetectInfo(BGR, width, height, rect, rectinfo);
		}finally {
			returnInstance(instance);
		}
	}
	public int setFacePosStatus(int facePos, int flag){
		MtfAndroidArmBridge instance = applyInstance();
		try{
			return instance.setFacePosStatus(facePos, flag);
		}finally {
			returnInstance(instance);
		}
	}
	/**
	 * 执行活体检测
	 * @param BGR 输入的BGR格式图像
	 * @param width 图像宽度
	 * @param height 图像高度
	 * @param rect {@link #nativeDetectFace(byte[], int, int, List)}方法返回的人脸检测信息原始数据，
	 * 							参见{@link NativeFaceInfo#getNativeData()}
	 * @return false--非活体,true--活体
	 */
	public boolean nativeLiveProcess(byte[] BGR, int width, int height, double[] rect){
		MtfAndroidArmBridge instance = applyInstance();
		try{
			return instance.liveProcess(BGR, width, height, rect);
		}finally {
			returnInstance(instance);
		}
	}
	/**
	 * @see #nativeLiveProcess(byte[], int, int, double[])
	 */
	public boolean liveProcess(Bitmap bitmap,NativeFaceInfo faceinfo) {
		try {
			byte[] matrix = getMatrixData(LazyImage.create(bitmap));
			return nativeLiveProcess(matrix,bitmap.getWidth(),bitmap.getHeight(),faceinfo.nativeData);
		} catch (ImageErrorException e) {
			throw new RuntimeException(e);
		}
	}
	/**
	 * 检测人脸
	 * @param bitmap
	 * @return
	 * @see #detectFace(net.gdface.image.BaseLazyImage)
	 */
	public List<CodeInfo> detectFace(Bitmap bitmap){
		try {
			return detectFace(LazyImage.create(bitmap));
		} catch (ImageErrorException e) {
			throw new RuntimeException(e);
		}
	}
	/**
	 * 提取特征
	 * @param bitmap
	 * @param faceNum
	 * @param facePos
	 * @return
	 * @throws NotFaceDetectedException
	 * @see #getCodeInfo(net.gdface.image.BaseLazyImage, int, List)
	 */
	public List<CodeInfo> getCodeInfo(Bitmap bitmap, int faceNum, List<CodeInfo> facePos) 
			throws NotFaceDetectedException{
		return getCodeInfo(LazyImage.create(bitmap), faceNum, facePos);
	}
	/**
	 * 提取单个人脸特征 
	 * @param bitmap
	 * @param facePos
	 * @return
	 */
	public CodeInfo getCodeInfo(Bitmap bitmap, CodeInfo facePos){
		return getCodeInfo(LazyImage.create(bitmap), facePos);
	}
	/**
	 * 检测并提取人脸特征
	 * @param bitmap
	 * @param faceNum
	 * @return
	 * @throws NotFaceDetectedException
	 * @see {@link BaseFaceApiLocal#detectAndGetCodeInfo(net.gdface.image.BaseLazyImage, int)}
	 */
	public CodeInfo[] detectAndGetCodeInfo(Bitmap bitmap, int faceNum) 
			throws NotFaceDetectedException {
		try {
			return detectAndGetCodeInfo(LazyImage.create(bitmap).open(), faceNum);
		} catch (ImageErrorException e) {
			throw new RuntimeException(e);
		}
	}
	/**
	 * 全局SDK运行参数
	 */
	private final HashMap<RuntimeParam, Object> globalRuntimeParams = new HashMap<RuntimeParam, Object>();	
	private final ReentrantReadWriteLock globalParamLock = new ReentrantReadWriteLock();
	/**
	 * 当前线程的SDK运行参数
	 */
	private final static ThreadLocal<Map<RuntimeParam, Object>> localRuntimeParams = new ThreadLocal<Map<RuntimeParam, Object>>();
	/**
	 * 设置SDK运行时参数,对所有线程和实例有效<br>
	 * 参数名为null时抛出{@link IllegalArgumentException}异常<br>
	 * 参数值为null时设置为默认的参数
	 * @param name 参数名
	 * @param value 参数值,数据类型必须参数名的要求
	 * @return 当前对象
	 */
	public FaceApiMtfAndroid setGlobalRuntimeParam(RuntimeParam name,Object value){
		Assert.notNull(name, "name");
		globalParamLock.writeLock().lock();
		try {
			globalRuntimeParams.put(name, value);	
		} finally {
			globalParamLock.writeLock().unlock();
		}		
		return this;
	}
	/**
	 * 设置SDK运行时参数,对当前线程的实例有效<br>
	 * 参数名为null时抛出{@link IllegalArgumentException}异常<br>
	 * 参数值为null时设置为默认的参数
	 * @param name 参数名
	 * @param value 参数值,数据类型必须参数名的要求
	 * @return 当前对象
	 */
	public FaceApiMtfAndroid setLocalRuntimeParam(RuntimeParam name,Object value){
		Assert.notNull(name, "name");
		Map<RuntimeParam, Object> m = localRuntimeParams.get();
		if(m == null){
			m = new HashMap<>();
			localRuntimeParams.set(m);
		}
		m.put(name, value);
		return this;
	}
	/**
	 * 删除保存SDK运行参数的线程局部变量<br>
	 * {@link #setLocalRuntimeParam(RuntimeParam, Object)}方法设置的参数保存在线程局部变量中(thread local variable),
	 * 如果使用该方法设置了运行时参数，则在线程调用结束时需要调用本方法删除线程局部变量，否则可能会引起资源泄露
	 * @return 当前对象
	 */
	public FaceApiMtfAndroid removeLocalRunTimeParam(){
		localRuntimeParams.remove();
		return this;
	}
	/**
	 * 调用{@link MtfAndroidArmBridge#setRuntimeParam(RuntimeParam, Object)}方法设置SDK运行参数,
	 * 优先使用为当前线程的设置参数({@link #localRuntimeParams}),然后是全局参数({@link #globalRuntimeParams}),
	 * 如果两个变量中都没定义name指定的参数,则忽略.
	 * @param instance SDK实例
	 * @param name 参数名
	 */
	private void setInstanceParam(MtfAndroidArmBridge instance,RuntimeParam name){
		Map<RuntimeParam, Object> m = localRuntimeParams.get();
		if(null != m && m.containsKey(name)){
			instance.setRuntimeParam(name, m.get(name));
		} else { 
			globalParamLock.readLock().lock();
			try {
				if(globalRuntimeParams.containsKey(name)){
					instance.setRuntimeParam(name, globalRuntimeParams.get(name));
				}
			} finally {
				globalParamLock.readLock().unlock();
			}
		}
	}
	@Override
	public FeatureSe getFeatureSe() {
		return FseMtfAndroidBridge.getFeatureSe();
	}
	/**
	 * {@link MtfAndroidArmBridge}实例资源池<br>
	 * 每一次执行SDK调用时，要从资源池从申请一个{@link MtfAndroidArmBridge}实例，调用结束后归还到资源池，
	 * 实例的初始化和销毁都由资源负责，调用者不须负责{@link MtfAndroidArmBridge}实例的初始化与释放
	 */
	private static final GenericObjectPool<MtfAndroidArmBridge> pool;
	static {
		try {
			// MTFSDK JNI接口静态区初始化
			Class.forName(net.gdface.mtfsdk.MtfAndroidArmBridge.class.getName());
		} catch (ClassNotFoundException e) {
			throw new ExceptionInInitializerError(e);
		}

		// 资源池初始化
		GenericObjectPoolConfig config = new GenericObjectPoolConfig();
		// android平台不支持JMX,所以这里必须设置为false,否则抛出异常
		config.setJmxEnabled(false);
		pool = new GenericObjectPool<MtfAndroidArmBridge>(new BridgeInstanceFactory(),config);
		
		// 根据配置参数设置并发线程数
		int concurrency = MtfAndroidArmBridge.CONFIG.getConcurrency();
		if(concurrency > 0){
			concurrentLock = new ShareLock(concurrency);
		}
	}
	/**
	 * 从资源池申请一个实例
	 * @return
	 */
	private static MtfAndroidArmBridge applyInstance() {
		try {
			return pool.borrowObject();
		} catch(RuntimeException e){
			throw e;
		}catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
	/**
	 * 归还实例到资源池
	 * @param obj
	 */
	private static void returnInstance(MtfAndroidArmBridge obj) {
		pool.returnObject(obj);
	}
}
