package net.gdface.mtfsdk;

import net.gdface.license.GfLicenseProvider;
import net.gdface.license.LicenseManager;
import net.gdface.license.LicenseUtils;
import net.gdface.utils.Assert;
import net.gdface.utils.Judge;
import net.gdface.utils.SimpleLog;

import static net.gdface.mtfsdk.MtfAndroidConfigProvider.*;

import java.util.Iterator;
import java.util.ServiceLoader;

/**
 * MTFSDK for android/arm JNI 接口<br>
 * 此类为非线程安全的，不可多线程共享调用，需要在外部进一步封装实现线程安全
 * @author guyadong
 *
 */
public class MtfAndroidArmBridge {
	/**
	 * 应用层提供的{@link MtfAndroidConfigProvider}接口实例,未提供则初始化为{@link DefaultMtfAndroidConfig}实例
	 */
	public static final MtfAndroidConfigProvider CONFIG = getConfigProvider();
	/** 授权管理器接口实例 */ 
	public static final LicenseManager LICENSE_MANAGER  = LicenseUtils.getLicenseManager();
	public static final String SDK_VERSION = "MTFSDK512V2";
	final long[] detectHandle = new long[1];
	final long[] featureHandle = new long[1];
	final long[] liveHandle = new long[1];

	/**
	 * 初始化状态
	 */
	private SdkStatus status=SdkStatus.SDK_UNINITIALIZED;

	static {
		try {
			// 加载算法动态库
			System.loadLibrary("FS_AndroidFaceSDK");
		} catch (Exception e) {
			SimpleLog.log(e.getMessage());
			throw new ExceptionInInitializerError(e);
		}
		SimpleLog.log("MtfAndroid Config INSTANCE:{}",DefaultMtfAndroidConfig.toString(CONFIG));
		// 要在动态库加载后调用，因为方法中可能会调用动态库中的函数
		LICENSE_MANAGER.installLicenseIfSPIAvailable();

	}

	MtfAndroidArmBridge() {
	}
	/**
	 * SPI(Service Provider Interface)机制加载 {@link GfLicenseProvider}实例,没有找到返回{@code null}
	 * @return
	 */
	private static GfLicenseProvider getLicenseProvider() {		
		ServiceLoader<GfLicenseProvider> providers = ServiceLoader.load(GfLicenseProvider.class);
		Iterator<GfLicenseProvider> itor = providers.iterator();
		if(!itor.hasNext()){
			return null;
		}
		return itor.next();
	}

	/**
	 * SPI(Service Provider Interface)机制加载 {@link GfLicenseProvider}实例,
	 * 没有找到返回{@link DefaultMtfAndroidConfig}实例
	 * @return
	 */
	private static MtfAndroidConfigProvider getConfigProvider() {		
		ServiceLoader<MtfAndroidConfigProvider> providers = ServiceLoader.load(MtfAndroidConfigProvider.class);
		Iterator<MtfAndroidConfigProvider> itor = providers.iterator();
		if(!itor.hasNext()){
			return new DefaultMtfAndroidConfig();
		}
		return itor.next();
	}
	/**
	 * 返回当前SDK初始化状态
	 * @return
	 */
	SdkStatus getStatus() {
		return status;
	}

	/**
     * 如果input非0结尾则添加'\0'返回,否则返回input
     * @param input
     * @return 
     */
    private static String zeroEnd(String input){
    	if(input == null){
    		return null;
    	}
		if(input.endsWith("\0")){
			return input;
		}    		
    	return input+"\0";
    }
	/**
	 * 执行SDK(检测模块)初始化
	 * @param licenseKey 授权关键字
	 * @param licenseCode 授权码
	 * @param instance
	 * @return 初始化状态
	 */
	static SdkStatus fdInit(String licenseKey, String licenseCode,MtfAndroidArmBridge instance) {
		if(SdkStatus.SDK_INIT_OK  == instance.status){
			return instance.status;
		}
		licenseCode = zeroEnd(licenseCode);
		String szFilePath = "\0";
		String szCompanyNames = zeroEnd(licenseKey);
		// 检测模块初始化
		int fdFlag = FDInit(licenseCode.getBytes(), 
				szCompanyNames.getBytes(), 
				szFilePath.getBytes(), 
				szFilePath,
				instance.detectHandle);
		return SdkStatus.jniCode(fdFlag);
	}
	/**
	 * 执行SDK(识别模块)初始化
	 * @param licenseKey 授权关键字
	 * @param licenseCode 授权码
	 * @return 初始化状态
	 */
	private static SdkStatus ffInit(String licenseKey, String licenseCode,MtfAndroidArmBridge instance) {
		if(SdkStatus.SDK_INIT_OK  == instance.status){
			return instance.status;
		}
		licenseCode = zeroEnd(licenseCode);
		String szFilePath = "\0";
		String szCompanyNames = zeroEnd(licenseKey);
		int ffFlag = FFInit(licenseCode.getBytes(), 
				szCompanyNames.getBytes(), 
				szFilePath.getBytes(), 
				szFilePath,
				instance.featureHandle);
		return SdkStatus.jniCode(ffFlag);
	}
	/**
	 * 执行SDK(活体检测)初始化
	 * @param licenseKey 授权关键字
	 * @param licenseCode 授权码
	 * @return 初始化状态
	 */
	private static SdkStatus flInit(String licenseKey, String licenseCode,MtfAndroidArmBridge instance) {
		if(SdkStatus.SDK_INIT_OK  == instance.status){
			return instance.status;
		}
		licenseCode = zeroEnd(licenseCode);
		String szFilePath = "\0";
		String szCompanyNames = zeroEnd(licenseKey);
		int ffFlag = FLInit(licenseCode.getBytes(), 
				szCompanyNames.getBytes(), 
				szFilePath.getBytes(), 
				instance.liveHandle);
		return SdkStatus.jniCode(ffFlag);
	}
	/**
	 * 执行SDK(检测模块,识别模块,活体检测模块)初始化,初始化失败则抛出异常
	 * @param licenseKey 授权关键字
	 * @param licenseCode 授权码
	 * @throws SdkInitException 
	 */
	private void sdkInit(String licenseKey, String licenseCode) throws SdkInitException {
		if(SdkStatus.SDK_INIT_OK  == status){
			return;
		}
		SdkStatus fdStatus = SdkStatus.SDK_UNINITIALIZED;
		SdkStatus ffStatus  = SdkStatus.SDK_UNINITIALIZED;
		SdkStatus flStatus  = SdkStatus.SDK_UNINITIALIZED;
		try{
			fdStatus  = fdInit(licenseKey, licenseCode, this);
			if(fdStatus !=SdkStatus.SDK_INIT_OK){
				status = fdStatus;
				SimpleLog.log("detect module: {}", status.msg);
				throw new SdkInitException(status);
			}

			ffStatus  = ffInit(licenseKey, licenseCode, this);
			if(ffStatus !=SdkStatus.SDK_INIT_OK){
				status = ffStatus;
				SimpleLog.log("feature module: {}", status.msg);
				throw new SdkInitException(status);
			}

			if(CONFIG.needLive()){
				flStatus  = flInit(licenseKey, licenseCode, this);
				if(flStatus !=SdkStatus.SDK_INIT_OK){
					status = flStatus;
					SimpleLog.log("live module: {}", status.msg);
					throw new SdkInitException(status);
				}
			}else{
				SimpleLog.log("SKIP initialization for live module(跳过活体检测模块初始化)");
			}
			// 检测模块和识别模块都初始化成功则置状态为成功
			status = SdkStatus.SDK_INIT_OK;	
		}finally {
			// 如果没有完全初始化成功，则destroy已经初始化模块
			if(status != SdkStatus.SDK_INIT_OK){
				if(fdStatus !=SdkStatus.SDK_INIT_OK){
					FDDestroy(detectHandle[0]);
				}
				if(ffStatus !=SdkStatus.SDK_INIT_OK){
					FFDestroy(featureHandle[0]);
				}
				if(flStatus !=SdkStatus.SDK_INIT_OK){
					FLDestroy(liveHandle[0]);
				}
			}
		}
	}
	/**
	 * SDK初始化<br>
	 * 初始化前{@link #licenseKey}和{@link licenseCode}必须已经初始化
	 * @return
	 * @throws SdkInitException
	 */
	MtfAndroidArmBridge init() throws SdkInitException{
		String licenseKey = LICENSE_MANAGER.getLicenseKey();
		String licenseCode = LICENSE_MANAGER.getLicenseCode();

		if(Judge.isEmpty(licenseKey)){
			throw new SdkInitException("EMPTY licenseKey,must call setLicenseKey() firstly");
		}
		if(Judge.isEmpty(licenseCode)){
			throw new SdkInitException("EMPTY licenseCode,must call setLicenseCode() firstly");
		}
		sdkInit(licenseKey, licenseCode);
		return this;
	}
    /**
	 * 人脸识别SDK资源释放
	 * @see #FDDestroy(long)
	 * @see #FFDestroy(long)
	 * @see #FLDestroy(long)
	 */
	void destroy() {	
		if(SdkStatus.SDK_INIT_OK  == status){
			status = SdkStatus.SDK_UNINITIALIZED;
	        FDDestroy(detectHandle[0]);
	        FFDestroy(featureHandle[0]);
	        FLDestroy(liveHandle[0]);
		}
    }
	////////////////////////////////////
	/**
     * 对输入图像进行人脸检测,检测结果由rect返回,
     * 检测到人脸返回>0的人脸个数,
     * 返回的是21个元素的脸部信息 可配合FDGetDetectInfo拿到人脸角度   人脸数量<=3
     * @param BGR 待检测图像BGR格式
     * @param width 图像宽度
     * @param height 图像高度
     * @param detectMod 0:用于检测图片,1:用于视频的循环检测
     * @param rect 用于返回人脸位置及关键点信息
     * @return 检测到的人脸个数(>0)或者错误代码(< 0),see also {@link SdkStatus}
     * @see #FDDetect(long, byte[], int, int, int, double[])

	 */
	public int detect(byte[] BGR, int width, int height, int detectMod, double[] rect){
		int ret = FDDetect(detectHandle[0], BGR, width, height, detectMod, rect);
		if(ret<0){
			throw new SdkRuntimeException(SdkStatus.jniCode(ret));
		}
		return ret;
	}
	
	/**
     * 对输入图像进行人脸检测返回最大人脸,检测结果由rect返回,
     * @param BGR 待检测图像BGR格式
     * @param width 图像宽度
     * @param height 图像高度
     * @param detectMod 0:用于检测图片,1:用于视频的循环检测
     * @param rect 用于返回人脸位置及关键点信息
     * @return 检测到的人脸个数1,或者错误代码(< 0),see also {@link SdkStatus}
     * @see #FDDetectMaxFace(long, byte[], int, int, int, double[])
	 */
	public  int detectMaxFace(byte[] BGR, int width, int height, int detectMod, double[] rect){
		int ret = FDDetectMaxFace(detectHandle[0], BGR, width, height, detectMod, rect);
		if(ret<0){
			throw new SdkRuntimeException(SdkStatus.jniCode(ret));
		}
		return ret;
	}
	/**
	 * @deprecated 人脸检测原始数据中已经包含了人脸角度,不再需要此函数
	 * @see #FDGetDetectInfo(long, byte[], int, int, double[], double[])
	 */
	public int getDetectInfo(byte[] BGR, int width, int height, double[] rect, double[] rectinfo){
		int ret = FDGetDetectInfo(detectHandle[0], BGR, width, height, rect, rectinfo);
		if(ret<0){
			throw new SdkRuntimeException(SdkStatus.jniCode(ret));
		}
		return ret;
	}
	/**
	 * @param BGR
	 * @param width
	 * @param height
	 * @param rect
	 * @return
	 * @deprecated replaced by {@link #feaExtractByte(byte[], int, int, double[])}
	 */
	public double[] feaExtract(byte[] BGR, int width, int height, double[] rect){
		double[] buffer = new double[FEATURE_LEN];
		int ret = FFFeaExtract(featureHandle[0], BGR, width, height, buffer, rect);
		if(ret<0){
			throw new SdkRuntimeException(SdkStatus.jniCode(ret));
		}
		return buffer;
	}
	public byte[] feaExtractByte(byte[] BGR, int width, int height, double[] rect){
		byte[] buffer = new byte[FEATURE_BYTES];
		int ret = FFFeaExtractByte(featureHandle[0], BGR, width, height, buffer, rect);
		if(ret<0){
			throw new SdkRuntimeException(SdkStatus.jniCode(ret));
		}
		return buffer;
	}
	public int facePosInterrelate(int faceNums, double rectRate, double[] faceRect, double[] faceRectInfo){
		int ret = FSFacePosInterrelate(detectHandle[0], faceNums, rectRate,faceRect, faceRectInfo);
		if(ret<0){
			throw new SdkRuntimeException(SdkStatus.jniCode(ret));
		}
		return ret;
	}
	public int setFacePosStatus(int facePos, int flag){
		int ret = FSSetFacePosStatus(detectHandle[0], facePos, flag);
		if(ret<0){
			throw new SdkRuntimeException(SdkStatus.jniCode(ret));
		}
		return ret;
	}
	public boolean liveProcess(byte[] BGR, int width, int height, double[] rect){
		double[] socre = new double[1];		
		int ret = FLProcess(liveHandle[0], BGR, width, height, rect, socre);
		if(ret<0){
			throw new SdkRuntimeException(SdkStatus.jniCode(ret));
		}
		return ret == 1;
	}
	/**
	 * 运行时参数名
	 * @author guyadong
	 */
	public enum RuntimeParam{
		/** 检测模块并发线程数,数据类型: int */detectThreadNumber,
		/** 最小人脸检测(宽/高)尺寸(像素),数据类型: int */minFaceSize,
		/** 特征提取模块并发线程数,数据类型: int */featureThreadNumber
	}
	/**
	 * 设置SDK运行时参数<br>
	 * 参数名为null时抛出{@link IllegalArgumentException}异常<br>
	 * 参数值为null时使用默认值
	 * @param name 参数名
	 * @param value 参数值,数据类型必须参数名的要求
	 * @return 当前对象
	 * @throws SdkRuntimeException 参数设置错误
	 */
	public MtfAndroidArmBridge setRuntimeParam(RuntimeParam name,Object value) throws SdkRuntimeException{
		Assert.notNull(name, "name");
		switch (name) {
		case detectThreadNumber:
			int detectThreadNumber = value != null ? (Integer) value : MtfAndroidConfigProvider.DEFAULT_DETECT_THREAD_NUMBER;
			if(detectThreadNumber > 0){
				int code = FDSetThreadsNumber(detectHandle[0], detectThreadNumber);
				if(code != 0){
					status = SdkStatus.jniCode(code);
					throw new SdkRuntimeException(status);
				}
			}
			break;
		case minFaceSize:
			int minFaceSize = value != null ? (Integer) value : MtfAndroidConfigProvider.DEFAULT_DETECT_MIN_FACE_SIZE;
			if(minFaceSize >= 0){
				int code = FDSetMinFaceSize(detectHandle[0], minFaceSize);
				if(code != 0){
					status = SdkStatus.jniCode(code);
					throw new SdkRuntimeException(status);	
				}
			}
			break;
		case featureThreadNumber:
			int featureThreadNumber = value != null ? (Integer) value : MtfAndroidConfigProvider.DEFAULT_FEATURE_THREAD_NUMBER;
			if(featureThreadNumber > 0){
				int code = FFSetThreadsNumber(featureHandle[0], featureThreadNumber);
				if(code != 0){
					status = SdkStatus.jniCode(code);
					throw new SdkRuntimeException(status);	
				}
			}
			break;
		default:
			throw new IllegalArgumentException(SimpleLog.logString("INVALID parameter name: {}",name));
		}	
		return this;
	}
	///////////////////////////////////
	/**
     * 获取设备加密信息
     * @param encryptionSerial [out]获取的设备加密信息
     * @param licenseKey 授权关键字
     * @return 成功(>0 加密信息长度)或错误代码(< 0)
     */
    public static native int FSGetDevicesSerial(byte[] encryptionSerial, byte[] licenseKey);

    /**
	 * 人脸检测初始化函数
	 * @param licenseCode 授权码
	 * @param licenseKey 授权关键字
	 * @param path 授权码文件本地路径
	 * @param faceDetectionModelPath_ 模型路径
	 * @param handle  [out]结构体
	 * @return 成功(0)或错误代码(< 0),see also {@link SdkStatus}
	 */
    private static native int FDInit(byte[] licenseCode, byte[] licenseKey, byte[] path, String faceDetectionModelPath_, long[] handle);
	/** 
     * 人脸检测模块资源释放函数
     * @param handle
     */
    static native void FDDestroy(long handle);

    /**
     * 对输入图像进行人脸检测,检测结果由rect返回,
     * 检测到人脸返回>0的人脸个数,
     * 返回的是21个元素的脸部信息 可配合FDGetDetectInfo拿到人脸角度   人脸数量<=3
     * @param handle
     * @param BGR 待检测图像BGR格式
     * @param width 图像宽度
     * @param height 图像调试
     * @param detectMod 0:用于检测图片,1:用于视频的循环检测
     * @param rect 用于返回人脸位置及关键点信 息,大小在外部申请 (61 * MAX_FACE_COUNT)<br>
     * 							rect格式：61 个数据为一组，依次为 人脸坐标(left, top, width, height) 
     * 							人脸质量 人脸姿态(roll yaw pitch) 人脸清晰度 人脸亮度 人脸关键点（25*2）
     * 							人脸检测框置信度
     * @return 检测到的人脸个数(>0)或者错误代码(< 0),see also {@link SdkStatus}
     */
    private static native int FDDetect(long handle, byte[] BGR, int width, int height, int detectMod, double[] rect);

    /**
     * 对输入图像进行人脸检测返回最大人脸,检测结果由rect返回,
	 * 返回的是21个元素的脸部信息  可配合FDGetDetectInfo拿到人脸角度  人脸数量 = 1
     * @param handle
     * @param BGR 待检测图像BGR格式
     * @param width 图像宽度
     * @param height 图像高度
     * @param detectMod 0:用于检测图片,1:用于视频的循环检测
     * @param rect 用于返回人脸位置及关键点信 息,大小在外部申请 (61 * MAX_FACE_COUNT)<br>
     * 							rect格式：61 个数据为一组，依次为 人脸坐标(left, top, width, height) 
     * 							人脸质量 人脸姿态(roll yaw pitch) 人脸清晰度 人脸亮度 人脸关键点（25*2）
     * 							人脸检测框置信度
     * @return 检测到的人脸个数1,或者错误代码(< 0),see also {@link SdkStatus}
	 */
    private static native int FDDetectMaxFace(long handle, byte[] BGR, int width, int height, int detectMod, double[] rect);
	/**
	 * 获取人脸角度，在注册用到，从属于检测系列
	 * double [] rect 输入人脸区域  double [] rectinfo 返回3个元素   1: roll       2:  yaw    3: pitch
	 * @deprecated 人脸检测原始数据中已经包含了人脸角度,不再需要此函数
	 */
	private static native int FDGetDetectInfo(long handle, byte[] BGR, int width, int height, double[] rect, double[] rectinfo);
	//    public static native int FDDetectMaxFaceTrack(long handle, byte[] BGR, int width, int height, double[] rect);    //直接返回61个元素的人脸信息
	
	private static native int FDSetMinFaceSize(long handle, int minSize);
	private static native int FDSetThreadsNumber(long handle, int threadsNumber);
	private static native int FFSetThreadsNumber(long handle, int threadsNumbers);
	/**
     * @return 人脸检测模块版本
     */
    public static native String FDgetVersion();


    ////////////////////////////////////////特征提取系列///////////////////////////////////////////
	/**
	 * 人脸识别初始化函数
	 * @param licenseCode 授权码
	 * @param licenseKey 授权关键字
	 * @param path 授权码文件本地路径
	 * @param faceDetectionModelPath 模型路径
	 * @param handle [out]句柄
	 * @return 成功(0)或错误代码(< 0),see also {@link SdkStatus}
	 */
	private static native int FFInit(byte[] licenseCode, byte[] licenseKey, byte[] path, String faceDetectionModelPath, long[] handle);
	/**
     * 人脸识别模块资源释放函数
     * @param handle 句柄
     */
    private static native void FFDestroy(long handle);

    /**
     * 对输入图像中检测到的人脸提取特征
     * @param handle 操作句柄
     * @param BGR 待检测图像BGR格式
     * @param width 图像宽度
     * @param height 图像高度
     * @param fea [out] 人脸特征 空间在外部申请，长度512
     * @param rect 人脸位置及关键点信息 由人脸检测SDK 产生,参见 {@link #FDDetect(long, byte[], int, int, int, double[])}
     * @return 成功(0)或错误代码(< 0),see also {@link SdkStatus}
     */
    private static native int FFFeaExtract(long handle, byte[] BGR, int width, int height, double[] fea, double[] rect);

    /**
     * 人脸相似度比对函数<br>
     * 对人脸特征feaA和feaB进行相似度比对，返回相似度结果
     * @param feaA 128 double
     * @param feaB 128 double
     * @return 相似度(0.0~1.0)
     * @deprecated replaced by {@link #FFSimilarityByte(byte[], byte[])}
     */
    public static native double FFSimilarity(double[] feaA, double[] feaB);
    /**
     * 对输入图像中检测到的人脸提取特征
     * @param handle 操作句柄
     * @param BGR 待检测图像BGR格式
     * @param width 图像宽度
     * @param height 图像高度
     * @param fea [out] 人脸特征 空间在外部申请，长度4096
     * @param rect 人脸位置及关键点信息 由人脸检测SDK 产生,参见 {@link #FDDetect(long, byte[], int, int, int, double[])}
     * @return 成功(0)或错误代码(< 0),see also {@link SdkStatus}
     */
    private static native int FFFeaExtractByte(long handle, byte[] BGR, int width, int height, byte[] fea, double[] rect);

    /**
     * 人脸相似度比对函数<br>
     * 对人脸特征feaA和feaB进行相似度比对，返回相似度结果
     * @param feaA 1024 bytes
     * @param feaB 1024 bytes
     * @return 相似度(0.0~1.0)
     */
    public static native double FFSimilarityByte(byte[] feaA, byte[] feaB);

    /**
     * @return 人脸识别模块版本
     */
    public static native String FFgetVersion();

    /**
     * 位置关联
     * 在人脸检测函数后用到，无论有无检测到人脸，均需要调用，从属于检测系列
     * @param handle
     * @param faceNums     人脸个数
     * @param rectRate 
     * @param faceRect     人脸信息(人脸检测数据rect) 21个元素
     * @param faceRectInfo 返回的标记，最后一个元素为是否成功比对的标志，其他为人脸信息
     * @return 位置关联总个数
     */
    private static native int FSFacePosInterrelate(long handle, int faceNums, double rectRate ,double[] faceRect, double[] faceRectInfo);
    /**
     * 人脸标记(成功后)
     *人脸识别成功后调用，告知该人脸已经识别成功，不用重复提取特征进行特征提取
     * @param facePos 人脸数组位置下班
     * @param flag    标记成功 1
     * @return 0成功  不为0失败
     */
    private static native int FSSetFacePosStatus(long handle, int facePos, int flag);

    


//    public static native int FDDetectMaxFaceTrack(long handle, byte[] BGR, int width, int height, double[] rect);    //直接返回61个元素的人脸信息

    ////////////////////////////////////////活体检测系列///////////////////////////////////////////
    /**
     * 活体检测模块初始化函数
     * @param licenseCode 授权码
     * @param licenseKey 授权关键字
     * @param path 授权码文件本地路径
     * @param handle [out]结构体
     * @return
     */
    private static native int FLInit(byte[] licenseCode, byte[] licenseKey, byte[] path, long[] handle);

    private static native void FLDestroy(long handle);

    /**
     * @param handle
     * @param BGR
     * @param width
     * @param height
     * @param rect
     * @param socre
     * @return 0--非活体,1--活体,<0 错误代码 
     */
    private static native int FLProcess(long handle, byte[] BGR, int width, int height, double[] rect, double[] socre);

}
