package net.facelib.mtfsdk;

import net.facelib.jni.BaseJniBridge;
import net.facelib.jni.SdkInitException;
import net.facelib.jni.SdkRuntime.RuntimeParam;
import net.facelib.jni.SdkRuntimeException;
import net.facelib.jni.SdkStatus;
import net.gdface.utils.Assert;
import net.gdface.utils.SimpleLog;

import static net.facelib.mtfsdk.MtfsdkConstant.*;

/**
 * MTFSDK for android/arm JNI 接口<br>
 * 此类为非线程安全的，不可多线程共享调用，需要在外部进一步封装实现线程安全
 * @author guyadong
 *
 */
public class V1AndroidBridge extends BaseJniBridge{
	public static final String SDK_VERSION = "MTFSDKARM512";
	final long[] featureHandle = new long[1];
	/**
	 * 初始化状态
	 */
	private SdkStatus status=SdkStatus.SDK_UNINITIALIZED;

	static {
		try {
			// 加载算法动态库
			System.loadLibrary("FS_AndroidFeaV1SDK");
		} catch (Exception e) {
			SimpleLog.log(e.getMessage());
			throw new ExceptionInInitializerError(e);
		}
	}

	public V1AndroidBridge() {
	}

	/**
	 * 返回当前SDK初始化状态
	 * @return
	 */
	protected SdkStatus getStatus() {
		return status;
	}
	/**
	 * 执行SDK(识别模块)初始化
	 * @param licenseKey 授权关键字
	 * @param licenseCode 授权码
	 * @return 初始化状态
	 */
	static SdkStatus ffInit(String licenseKey, String licenseCode,V1AndroidBridge instance) {
		if(SdkStatus.SDK_INIT_OK  == instance.status){
			return instance.status;
		}
		licenseCode = zeroEnd(licenseCode);
		String szFilePath = "\0";
		String szCompanyNames = zeroEnd(licenseKey);
		int ffFlag = FFInit(licenseCode.getBytes(), 
				szCompanyNames.getBytes(), 
				szFilePath.getBytes(), 
				szFilePath,
				instance.featureHandle);
		return SdkStatus.jniCode(ffFlag);
	}
	/**
	 * 执行SDK(检测模块,识别模块,活体检测模块)初始化,初始化失败则抛出异常
	 * @param licenseKey 授权关键字
	 * @param licenseCode 授权码
	 * @throws SdkInitException 
	 */
	@Override
	protected void sdkInit(String licenseKey, String licenseCode) throws SdkInitException {
		if(SdkStatus.SDK_INIT_OK  == status){
			return;
		}
		SdkStatus ffStatus  = SdkStatus.SDK_UNINITIALIZED;
		try{

			ffStatus  = ffInit(licenseKey, licenseCode, this);
			if(ffStatus !=SdkStatus.SDK_INIT_OK){
				status = ffStatus;
				SimpleLog.log("feature module: {}", status.msg);
				throw new SdkInitException(status);
			}

			// 检测模块和识别模块都初始化成功则置状态为成功
			status = SdkStatus.SDK_INIT_OK;	
		}finally {
			// 如果没有完全初始化成功，则destroy已经初始化模块
			if(status != SdkStatus.SDK_INIT_OK){
				if(ffStatus !=SdkStatus.SDK_INIT_OK){
					FFDestroy(featureHandle[0]);
				}
			}
		}
	}
    /**
	 * 人脸识别SDK资源释放
	 * @see #FDDestroy(long)
	 * @see #FFDestroy(long)
	 * @see #FLDestroy(long)
	 */
	@Override
	protected void destroy() {	
		if(SdkStatus.SDK_INIT_OK  == status){
			status = SdkStatus.SDK_UNINITIALIZED;
	        FFDestroy(featureHandle[0]);
		}
    }
	public byte[] feaExtractByte(byte[] BGR, int width, int height, double[] rect){
		byte[] buffer = new byte[FEATURE_BYTES];
		int ret = FFFeaExtractByte(featureHandle[0], BGR, width, height, buffer, rect);
		if(ret<0){
			throw new SdkRuntimeException(SdkStatus.jniCode(ret));
		}
		return buffer;
	}
	@Override
	public V1AndroidBridge setRuntimeParam(RuntimeParam name,Object value) throws SdkRuntimeException{
		Assert.notNull(name, "name");
		switch (name) {
		case featureThreadNumber:
			int featureThreadNumber = value != null ? (Integer) value : MtfAndroidConfigProvider.DEFAULT_FEATURE_THREAD_NUMBER;
			if(featureThreadNumber > 0){
				int code = FFSetThreadsNumber(featureHandle[0], featureThreadNumber);
				if(code != 0){
					status = SdkStatus.jniCode(code);
					throw new SdkRuntimeException(status);	
				}
			}
			break;
		default:
			throw new IllegalArgumentException(SimpleLog.logString("INVALID parameter name: {}",name));
		}	
		return this;
	}
	////////////////////////////////////////特征提取系列///////////////////////////////////////////
	/**
	 * 人脸识别初始化函数
	 * @param licenseCode 授权码
	 * @param licenseKey 授权关键字
	 * @param path 授权码文件本地路径
	 * @param faceDetectionModelPath 模型路径
	 * @param handle [out]句柄
	 * @return 成功(0)或错误代码(< 0),see also {@link SdkStatus}
	 */
	private static native int FFInit(byte[] licenseCode, byte[] licenseKey, byte[] path, String faceDetectionModelPath, long[] handle);
	/**
     * 人脸识别模块资源释放函数
     * @param handle 句柄
     */
    static native void FFDestroy(long handle);

    /**
     * 对输入图像中检测到的人脸提取特征
     * @param handle 操作句柄
     * @param BGR 待检测图像BGR格式
     * @param width 图像宽度
     * @param height 图像高度
     * @param fea [out] 人脸特征 空间在外部申请，长度4096
     * @param rect 人脸位置及关键点信息 由人脸检测SDK 产生,参见 {@link #FDDetect(long, byte[], int, int, int, double[])}
     * @return 成功(0)或错误代码(< 0),see also {@link SdkStatus}
     */
    private static native int FFFeaExtractByte(long handle, byte[] BGR, int width, int height, byte[] fea, double[] rect);

    /**
     * 人脸相似度比对函数<br>
     * 对人脸特征feaA和feaB进行相似度比对，返回相似度结果
     * @param feaA 1024 bytes
     * @param feaB 1024 bytes
     * @return 相似度(0.0~1.0)
     */
    public static native double FFSimilarityByte(byte[] feaA, byte[] feaB);

    /**
     * @return 人脸识别模块版本
     */
    public static native String FFgetVersion();
	private static native int FFSetThreadsNumber(long handle, int threadsNumbers);

    


//    public static native int FDDetectMaxFaceTrack(long handle, byte[] BGR, int width, int height, double[] rect);    //直接返回61个元素的人脸信息

}
