package gu.simplemq.activemq;

import java.util.HashMap;
import java.util.Iterator;

import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.Session;

import org.apache.activemq.advisory.AdvisorySupport;
import org.apache.activemq.command.ActiveMQDestination;
import org.apache.activemq.command.ActiveMQQueue;
import org.apache.activemq.command.ActiveMQTopic;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Maps;

import gu.simplemq.IConsumerAdvisor;

import static com.google.common.base.Preconditions.*;
/**
 * threadSafe
 * @author guyadong
 *
 */
public class AdvisoryMessageManager implements AutoCloseable,IConsumerAdvisor,ActivemqConstants{
	private final HashMap<String, MessageConsumer> advisoryConsumers = Maps.newHashMap();
	private static final LoadingCache<ActivemqPoolLazy,AdvisoryMessageManager> CACHE = CacheBuilder.newBuilder().build(new CacheLoader<ActivemqPoolLazy, AdvisoryMessageManager>(){

		@Override
		public AdvisoryMessageManager load(ActivemqPoolLazy key) throws Exception {
			return new AdvisoryMessageManager(key);
		}});
	private volatile Connection advisoryConnection;
	private volatile Session advisorySession;
	private final ActivemqPoolLazy poolLazy;
	private AdvisoryMessageManager(ActivemqPoolLazy poolLazy) {
		this.poolLazy = checkNotNull(poolLazy, "poolLazy is null");
	}
	private void init() throws JMSException{
		if(advisoryConnection == null){
			advisoryConnection = this.poolLazy.borrow();
			advisoryConnection.start();
			advisorySession = advisoryConnection.createSession(Boolean.FALSE, Session.AUTO_ACKNOWLEDGE);
		}
	}
	
	void addAdvisoryConsumerIfAbsent(ActiveMQDestination destination) throws JMSException {
		String name = destination.getPhysicalName();
		synchronized (this) {
			if(!advisoryConsumers.containsKey(name)){
				init();
				ActiveMQTopic advisoryTopic = AdvisorySupport.getConsumerAdvisoryTopic(destination);
				MessageConsumer advisoryConsumer = advisorySession.createConsumer(advisoryTopic);
				advisoryConsumer.setMessageListener(new AdvisoryListener());		
				advisoryConsumers.put(name, advisoryConsumer);
			}
		}
	}
	@Override
	public void addAdvisoryTopicIfAbsent(String name)  {
		try {
			addAdvisoryConsumerIfAbsent(new ActiveMQTopic(name));
		} catch (JMSException e) {
			throw new RuntimeException(e);
		}
	}
	@Override
	public void addAdvisoryQueueIfAbsent(String name)  {
		try {
			addAdvisoryConsumerIfAbsent(new ActiveMQQueue(name));
		} catch (JMSException e) {
			throw new RuntimeException(e);
		}
	}
	/**
	 * 返回指定频道的 advisory message<br>
	 * @param channelName
	 * @return advisory message,没有收到消息返回{@code null}
	 * @throws JMSException
	 */
	Message advisoryMessageOf(String channelName) throws JMSException{
		synchronized (this) {
			if(advisoryConsumers.containsKey(channelName)){
				MessageConsumer advisoryConsumer = advisoryConsumers.get(channelName);
				AdvisoryListener advisoryListener = (AdvisoryListener) advisoryConsumer.getMessageListener();
				if(advisoryListener.message == null){
					synchronized (advisoryListener) {
						try {
							advisoryListener.wait(5000);
						} catch (InterruptedException e) {
						}
					}
				}
				return advisoryListener.message;	
			}
			return null;
		}
	}
	@Override
	public void close() {
		synchronized (this) {
			// 关闭所有 MessageConsumer 实例
			for(Iterator<MessageConsumer> itor = advisoryConsumers.values().iterator();itor.hasNext();){
				try {
					itor.next().close();
				} catch (JMSException e) {
					e.printStackTrace();
				}
				itor.remove();
			}
			try {
				if(null != advisorySession){
					advisorySession.close();
					advisorySession = null;
				}
			} catch (JMSException e) {
				e.printStackTrace();
			}
			if(null != advisoryConnection){
				this.poolLazy.release(advisoryConnection);
				advisoryConnection = null;
			}
		}
	}
	
	@Override
	public int consumerCountOf(String channelName) {
		try {
			Message message = advisoryMessageOf(channelName);
			return message == null ? 0 : message.getIntProperty(PROP_CONSUMER_COUNT);
		} catch (JMSException e) {
			throw new RuntimeException(e);
		}
	}
	
	public static AdvisoryMessageManager instanceOf(ActivemqPoolLazy pool){
		return CACHE.getUnchecked(pool);
	}
	/**
	 * 关闭并删除所有资源池中的{@link AdvisoryMessageManager}实例
	 */
	public synchronized static void closeAll(){
		for(Iterator<AdvisoryMessageManager> itor = CACHE.asMap().values().iterator();itor.hasNext();){
			AdvisoryMessageManager p = itor.next();
			itor.remove();
			p.close();
		}
	}
	private static class AdvisoryListener implements MessageListener{
		/** 最近一次收到的 advisory 消息 */
		private Message message ;

		@Override
		public void onMessage(Message message) {
			this.message = message;
			synchronized (this) {
				this.notifyAll();
			}
		}
	}
}
