package gu.simplemq.activemq;

import java.util.Hashtable;

import javax.jms.BytesMessage;
import javax.jms.Connection;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.Session;
import javax.jms.TextMessage;

import org.apache.activemq.command.ActiveMQDestination;

import gu.simplemq.BaseMQDispatcher;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Activemq消息分发器抽象实现(线程安全)<br>
 * @author guyadong
 *
 */
abstract class BaseActivemqDispatcher extends BaseMQDispatcher<Connection>{
	private final Hashtable<String, MessageConsumer> consumers = new Hashtable<>();
	private volatile Session session;
	private final AdvisoryMessageManager advisoryMessageManager;
	
	/**
	 * 创建消息接收处理对象
	 * @param session
	 * @param name name of queue or topic
	 * @return 消息消费对象
	 * @throws JMSException
	 */
	protected abstract Destination makeDestination(String name) throws JMSException;
	BaseActivemqDispatcher(ActivemqPoolLazy pool) {
		super(pool);
		this.advisoryMessageManager = AdvisoryMessageManager.instanceOf(pool);
	}

	protected Session getSession() {
		return checkNotNull(session,"session is uninitialized");
	}
	@Override
	protected void doInit() throws JMSException{
		Connection c = getConnection();
		c.start();
		session = c.createSession(Boolean.FALSE, Session.AUTO_ACKNOWLEDGE); 
	}
	@Override
	protected void doUninit() {
		if(session != null){
			try {
				session.close();
			} catch (JMSException e) {
				e.printStackTrace();
			}
			session = null;
		}
	}
	@Override
	protected void doSub(String channel) throws JMSException {
		synchronized (consumers) {
			MessageConsumer messageConsumer = consumers.get(channel);
			if(null == messageConsumer){
				Destination destination = makeDestination(channel);
				messageConsumer = getSession().createConsumer(destination);
				messageConsumer.setMessageListener(new ActivemqListener(channel));
				consumers.put(channel, messageConsumer);
				advisoryMessageManager.addAdvisoryConsumerIfAbsent((ActiveMQDestination) destination);
			}			
		}
	}

	@Override
	protected void doUnsub(String channel) throws JMSException {
		synchronized (consumers) {
			MessageConsumer messageConsumer = consumers.remove(channel);
			if(null != messageConsumer){
				messageConsumer.close();
			}
		}
	}
	private class ActivemqListener implements MessageListener{
		private final String channel;
		ActivemqListener(String channel) {
			this.channel = channel;
		}
		
		private String textOf(Message message) throws JMSException{
			if(message instanceof TextMessage){
				return ((TextMessage) message).getText();
			}
			if(message instanceof BytesMessage){
				BytesMessage bytesMessage = (BytesMessage)message;
				byte[] buf = new byte[(int) bytesMessage.getBodyLength()];
				bytesMessage.readBytes(buf);
				return new String(buf);
			}
			throw new IllegalArgumentException(String.format("INVALID message type,%s,%s required",
					TextMessage.class.getName(),
					BytesMessage.class.getName()));
		}
		@Override
		public void onMessage(Message message) {
			try {
				// 调用 IMessageDispatcher 接口分发消息
				dispatch(channel, textOf(message));
			} catch (JMSException e) {
				e.printStackTrace();
			}
		}
		
	}
}
