package gu.simplemq.activemq;

import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.TextMessage;
import org.apache.activemq.command.ActiveMQQueue;
import org.apache.activemq.command.ActiveMQTopic;

import static com.google.common.base.Preconditions.checkNotNull;  
/**
 * ACTIVEMQ 消息发送基类
 * @author guyadong
 *
 */
abstract class BaseSender  {
	protected final AdvisoryMessageManager advisoryMessageManager;

	public BaseSender(AdvisoryMessageManager advisoryMessageManager) {
		super();
		this.advisoryMessageManager = checkNotNull(advisoryMessageManager,"advisoryMessageManager is null");
	}

	/**
	 * 创建消息发送对象
	 * @param session
	 * @param name name of queue or topic
	 * @return 消息消费对象
	 * @throws JMSException
	 */
	protected abstract MessageProducer makeSender(Session session,String name) throws JMSException;

	void doSend(Connection connection, String channel, Iterable<String> messages) throws JMSException {
		// 启动连接
		connection.start();
		// 创建会话
		Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
		 MessageProducer producer = null;
		// 创建一个生产者
		try {
	        producer = makeSender(session,channel);
	        for(String message : messages){
		        TextMessage textMessage = session.createTextMessage(message);
		        // 发布消息
	            producer.send(textMessage);
	        }
		} finally {
			if(null != producer){
				producer.close();
			}
			session.close();
		}
	}
	static class PublishSender extends BaseSender{


		public PublishSender(AdvisoryMessageManager advisoryMessageManager) {
			super(advisoryMessageManager);
		}

		@Override
		protected MessageProducer makeSender(Session session, String name) throws JMSException {
			// 创建Topic
//			Topic topic = session.createTopic(name);
			ActiveMQTopic topic = new ActiveMQTopic(name);
			advisoryMessageManager.addAdvisoryConsumerIfAbsent(topic);
			// 创建消息生产者
			return session.createProducer(topic);
		}
	}
	static class ProducerSender extends BaseSender{
		public ProducerSender(AdvisoryMessageManager advisoryMessageManager) {
			super(advisoryMessageManager);
		}
		@Override
		protected MessageProducer makeSender(Session session, String name) throws JMSException {
			// 创建队列
			ActiveMQQueue queue = new ActiveMQQueue(name);
			advisoryMessageManager.addAdvisoryConsumerIfAbsent(queue);
			// 创建消息生产者
			return session.createProducer(queue);
		}
	}
}
