/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014-2017 abel533@gmail.com
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package gu.sql2java.pagehelper;

import java.util.List;
import java.util.Properties;

import com.google.common.base.Function;
import com.google.common.collect.Lists;

/**
 * 基础分页方法
 *
 * @author liuzh
 */
@SuppressWarnings("rawtypes")
public abstract class PageHelper {
	protected static final ThreadLocal<Page> LOCAL_PAGE = new ThreadLocal<Page>();
    protected static boolean DEFAULT_COUNT = true;

    /**
     * 设置 Page 参数
     *
     * @param page
     */
    protected static void setLocalPage(Page page) {
        LOCAL_PAGE.set(page);
    }

    /**
     * 获取 Page 参数
     *
     * @return
     */
    @SuppressWarnings("unchecked")
	public static <T> Page<T> getLocalPage() {
        return LOCAL_PAGE.get();
    }

    /**
     * 移除本地变量
     */
    public static void clearPage() {
        LOCAL_PAGE.remove();
    }

    /**
     * 开始分页
     *
     * @param pageNum  页码
     * @param pageSize 每页显示数量
     */
    public static <E> Page<E> startPage(int pageNum, int pageSize) {
        return startPage(pageNum, pageSize, DEFAULT_COUNT);
    }

    /**
     * 开始分页
     *
     * @param pageNum  页码
     * @param pageSize 每页显示数量
     * @param count    是否进行count查询
     */
    public static <E> Page<E> startPage(int pageNum, int pageSize, boolean count) {
        return startPage(pageNum, pageSize, count, null, null);
    }

    /**
     * 开始分页
     *
     * @param pageNum  页码
     * @param pageSize 每页显示数量
     * @param orderBy  排序
     */
    public static <E> Page<E> startPage(int pageNum, int pageSize, String orderBy) {
        Page<E> page = startPage(pageNum, pageSize);
        return page;
    }

    /**
     * 开始分页
     *
     * @param pageNum      页码
     * @param pageSize     每页显示数量
     * @param count        是否进行count查询
     * @param reasonable   分页合理化,null时用默认配置
     * @param pageSizeZero true且pageSize=0时返回全部结果，false时分页,null时用默认配置
     */
    public static <E> Page<E> startPage(int pageNum, int pageSize, boolean count, Boolean reasonable, Boolean pageSizeZero) {
        Page<E> page = new Page<E>(pageNum, pageSize, count);
        page.setReasonable(reasonable);
        page.setPageSizeZero(pageSizeZero);
        setLocalPage(page);
        return page;
    }

    /**
     * 保存WhereHelper生成的WHERE SQL语句
     * @param where
     */
    public static void setWhere(String where){
    	Page<?> page = getLocalPage();
        if (page != null) {
            page.setWhere(where);
        } else {
            page = new Page();
            page.setWhere(where);
            setLocalPage(page);
        }
    }
    /**
     * 返回WhereHelper生成的WHERE SQL语句
     */
    public static String getWhere(){
    	Page<?> page = getLocalPage();
        if (page != null) {
            return page.getWhere();
        }
        return null;
    }
    /**
     * 设置当前操作是否启用分页查询<br>
     * 在启用了分页查询的服务方法中,{@link Page}的enable字段默认为{@code true}，
     * 这时如果要执行非分页查询的数据库查询，需要调用此方法设置为{@code false},
     * 在调用真正需要分页查询的数据库查询前，需要恢复设置为{@code true},
     * 对没有启用分页查询的服务调用,此设置无效。
     * @param enable
     */
    public static void setEnable(boolean enable){
    	Page<?> page = getLocalPage();
    	if (null == page) {
            setLocalPage(page = new Page());
		}
    	page.setEnable(enable);
    }
    /**
     * 设置参数
     *
     * @param properties 插件属性
     */
    protected static void setStaticProperties(Properties properties){
        //defaultCount，这是一个全局生效的参数，多数据源时也是统一的行为
        if(properties != null){
            DEFAULT_COUNT = Boolean.valueOf(properties.getProperty("defaultCount", "true"));
        }
    }

	/**
	 * 对Page对象元素执行指定的转换
	 * @param page
	 * @param trans
	 * @return always input
	 */
	@SuppressWarnings({ "unchecked" })
	public static List transform(List input,Function trans){
		if(null != input && null != trans){
			if(input instanceof Page){
				Page page = (Page)input;
				for(int i=0 ; i<page.size();++i){
					page.set(i, trans.apply(page.get(i)));
				}
				return page;
			}else {
				return Lists.transform(input, trans);
			}
		}
		return input;
	}

}
