package gu.sql2java.store;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.net.URLStreamHandlerFactory;
import java.nio.ByteBuffer;

/**
 * 二进制数据存储接口
 * @author guyadong
 *
 */
public interface URLStore {
	/**
	 * 判断指定的URL是否当前实例的协议一致
	 * @param url 为{@code null}返回{@code false}
	 * @return 协议一致返回{@code true},否则返回{@code false}
	 */
	boolean isStored(URL url);
	/**
	 * 判断URL指定的数据是否存在<br>
	 * @param url 为{@code null}返回{@code false}
	 * @return 数据存在返回{@code true},否则返回{@code false}
	 */
	boolean exists(URL url);
	/**
	 * 保存二进制数据<br>
	 * @param <T> 数据类型可以是byte[],{@link InputStream},{@link ByteBuffer},{@link String}(base64编码),
	 * {@link File},{@link URL},{@link URI} 否则抛出{@link IllegalArgumentException}<br>
	 * @param input 二进制数据
	 * @param md5 二进制数据的MD5校验码,为{@code null}时会自动计算 
	 * @param extension 数据类型(文件后缀),可为{@code null}
	 * @param overwrite 是否覆盖已存在数据
	 * @param makeURLOnly 为{@code true}时不存储数据只返回存储URL
	 * @return 保存的URL
	 * @throws IOException
	 */
	<T> URL store(T input, String md5, String extension, boolean overwrite, boolean makeURLOnly) throws IOException;
	/**
	 * 删除{@code md5}指定的存储数据
	 * @param md5
	 * @return {@code true} if deleted, otherwise {@code false}
	 * @throws IOException
	 */
	boolean delete(String md5) throws IOException;
	/**
	 * 保存二进制数据<br>
	 * 如果数据已经被保存则不做任何事
	 * @param url
	 * @param overwrite 是否覆盖已存在数据
	 * @param makeURLOnly 为{@code true}时不存储数据只返回存储URL
	 * @return 保存的URL
	 * @throws IOException
	 */
	URL store(URL url, boolean overwrite, boolean makeURLOnly) throws IOException;
	/**
	 * 删除指定的存储数据
	 * @param url 存储的URL
	 * @return {@code true} if deleted, otherwise {@code false}
	 * @throws IOException
	 */
	boolean delete(URL url) throws IOException;
	/**
	 * 返回当前存储实现的协议名
	 */
	String getProtocol();
	
	/**
	 * 设置当前线程有效的附加参数 
	 * @param name 参数名
	 * @param value 参数值
	 * @return 当前对象
	 */
	BaseURLStore setAdditionalParam(String name, Object value);
	/**
	 * 将当前{@link URLStreamHandlerFactory}实例安装到JVM
	 * @return 当前对象
	 * @throws Exception
	 */
	BaseURLStore intall() throws Exception;
}