package org.sean.framework.es;

import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.sean.framework.code.StatusInfo;
import org.sean.framework.exception.StatusException;
import org.sean.framework.logging.Logger;
import org.sean.framework.util.GSONUtil;
import org.sean.framework.util.ObjectUtil;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * 查询项
 */
public class QueryItem {
    /**
     * 操作类型
     */
    public QueryOption opt;
    //
    public float boost = 1.0f;
    /**
     * 操作字段
     */
    public QueryField field;
    //term查询
    public Integer term;
    //term查询
    public String term4txt;
    // terms查询
    public List<Integer> terms = new ArrayList<>();
    // terms查询
    public List<String> terms4txt = new ArrayList<>();
    // 正则查询
    public String regexp;
    // 通配符查询
    public String wildcard;
    // 全文搜索
    public String stringQuery;
    // 全文搜索
    public String simpleStringQuery;
    // exist
    public Boolean exist;
    // Range
    public Range range;
    private Logger logger = Logger.newInstance(getClass());

    public QueryItem(QueryOption opt, QueryField field) {
        this.opt = opt;
        this.field = field;
    }

    /**
     * exist
     *
     * @param opt   QueryOption
     * @param field QueryField
     * @param exist is exist
     * @return QueryItem
     */
    public static QueryItem create(QueryOption opt, QueryField field, Boolean exist) {
        QueryItem item = new QueryItem(opt, field);
        item.exist = exist;
        return item;
    }

    /**
     * term
     *
     * @param opt   QueryOption
     * @param field QueryField
     * @param term  Term
     * @return QueryItem
     */
    public static QueryItem create(QueryOption opt, QueryField field, Integer term) {
        QueryItem item = new QueryItem(opt, field);
        item.term = term;
        return item;
    }

    /**
     * term
     *
     * @param opt   QueryOption
     * @param field QueryField
     * @param term  Term
     * @return QueryItem
     */
    public static QueryItem create(QueryOption opt, QueryField field, String term) {
        QueryItem item = new QueryItem(opt, field);
        item.term4txt = term;
        return item;
    }

    /**
     * terms
     *
     * @param opt   QueryOption
     * @param field QueryField
     * @param terms Terms
     * @return QueryItem
     */
    public static QueryItem create(QueryOption opt, QueryField field, Collection<Integer> terms) {
        QueryItem item = new QueryItem(opt, field);
        item.terms.addAll(terms);
        return item;
    }

    /**
     * terms
     *
     * @param opt   QueryOption
     * @param field QueryField
     * @param terms Terms
     * @return QueryItem
     */
    public static QueryItem create4Txt(QueryOption opt, QueryField field, Collection<String> terms) {
        QueryItem item = new QueryItem(opt, field);
        item.terms4txt.addAll(terms);
        return item;
    }

    /**
     * Regexp
     *
     * @param opt    QueryOption
     * @param field  QueryField
     * @param regexp Regexp
     * @return QueryItem
     */
    public static QueryItem create4Regexp(QueryOption opt, QueryField field, String regexp) {
        QueryItem item = new QueryItem(opt, field);
        item.regexp = regexp;
        return item;
    }

    /**
     * Wildcard
     *
     * @param opt      QueryOption
     * @param field    QueryField
     * @param wildcard Wildcard String
     * @return QueryItem
     */
    public static QueryItem create4Wildcard(QueryOption opt, QueryField field, String wildcard) {
        QueryItem item = new QueryItem(opt, field);
        item.wildcard = wildcard;
        return item;
    }

    /**
     * String Query
     *
     * @param opt   QueryOption
     * @param value String
     * @return QueryItem
     */
    public static QueryItem create4StringQuery(QueryOption opt, String value) {
        QueryItem item = new QueryItem(opt, null);
        item.stringQuery = value;
        return item;
    }

    /**
     * Simple String Query
     *
     * @param opt   QueryOption
     * @param value String
     * @return QueryItem
     */
    public static QueryItem create4SimpleStringQuery(QueryOption opt, String value) {
        QueryItem item = new QueryItem(opt, null);
        item.simpleStringQuery = value;
        return item;
    }

    /**
     * Range
     *
     * @param field QueryField
     * @param opt   QueryOption
     * @param range Range
     * @return QueryItem
     */
    public static QueryItem create(QueryOption opt, QueryField field, Range range) {
        QueryItem item = new QueryItem(opt, field);
        item.range = range;
        return item;
    }

    /**
     * Boost
     *
     * @param boost boost
     * @return QueryItem
     */
    public QueryItem setBoost(float boost) {
        this.boost = boost;
        return this;
    }

    /**
     * 创建查询条件
     *
     * @return QueryBuilder
     */
    public QueryBuilder toQueryBuilder() {
        String name = field.getName();
        if (term != null) {
            return QueryBuilders.termQuery(name, term);
        } else if (ObjectUtil.isNotEmpty(terms)) {
            return QueryBuilders.termsQuery(name, terms);
        } else if (term4txt != null) {
            return QueryBuilders.termQuery(name, term4txt);
        } else if (ObjectUtil.isNotEmpty(terms4txt)) {
            return QueryBuilders.termsQuery(name, terms4txt);
        } else if (Boolean.TRUE.equals(exist)) {
            return QueryBuilders.existsQuery(name);
        } else if (regexp != null) {
            return QueryBuilders.regexpQuery(name, regexp);
        } else if (wildcard != null) {
            return QueryBuilders.wildcardQuery(name, wildcard);
        } else if (stringQuery != null) {
            return QueryBuilders.queryStringQuery(stringQuery);
        } else if (simpleStringQuery != null) {
            return QueryBuilders.simpleQueryStringQuery(simpleStringQuery);
        } else if (range != null) {
            if (range.gt != null) {
                return QueryBuilders.rangeQuery(name).gt(range.gt);
            }
            if (range.gte != null) {
                return QueryBuilders.rangeQuery(name).gte(range.gte);
            }
            if (range.lt != null) {
                return QueryBuilders.rangeQuery(name).lt(range.lt);
            }
            if (range.lte != null) {
                return QueryBuilders.rangeQuery(name).lte(range.lte);
            }
            if (range.to != null && range.from != null) {
                return QueryBuilders.rangeQuery(name).from(range.from)
                        .to(range.to)
                        .includeLower(range.includeLower)
                        .includeUpper(range.includeUpper);
            }
            throw new StatusException(StatusInfo.paramsInvalidError().setAnyMessage("Range条件异常"));
        } else {
            logger.warn("不支持的查询条件:{}", GSONUtil.obj2Json(this));
            throw new StatusException(StatusInfo.paramsInvalidError().setAnyMessage("不支持的查询条件"));
        }
    }

    /**
     * 范围查询
     */
    public static class Range {
        public Object from;
        public Object to;
        public Object gt;
        public Object gte;
        public Object lt;
        public Object lte;
        public boolean includeLower = true;
        public boolean includeUpper = true;
    }
}
