/*
 * Decompiled with CFR 0.152.
 */
package com.github.chouheiwa.wallet.socket.bitlib.crypto;

import com.github.chouheiwa.wallet.socket.bitlib.bitcoinj.Base58;
import com.github.chouheiwa.wallet.socket.bitlib.crypto.Hmac;
import com.github.chouheiwa.wallet.socket.bitlib.crypto.InMemoryPrivateKey;
import com.github.chouheiwa.wallet.socket.bitlib.crypto.PublicKey;
import com.github.chouheiwa.wallet.socket.bitlib.crypto.ec.Parameters;
import com.github.chouheiwa.wallet.socket.bitlib.crypto.ec.Point;
import com.github.chouheiwa.wallet.socket.bitlib.model.NetworkParameters;
import com.github.chouheiwa.wallet.socket.bitlib.model.hdpath.HdKeyPath;
import com.github.chouheiwa.wallet.socket.bitlib.util.BitUtils;
import com.github.chouheiwa.wallet.socket.bitlib.util.ByteReader;
import com.github.chouheiwa.wallet.socket.bitlib.util.ByteWriter;
import com.google.common.base.Preconditions;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.util.List;
import java.util.UUID;

public class HdKeyNode
implements Serializable {
    public static final int HARDENED_MARKER = Integer.MIN_VALUE;
    private static final String BITCOIN_SEED = "Bitcoin seed";
    private static final int CHAIN_CODE_SIZE = 32;
    private final InMemoryPrivateKey _privateKey;
    private final PublicKey _publicKey;
    private final byte[] _chainCode;
    private final int _depth;
    private final int _parentFingerprint;
    private final int _index;
    private static final byte[] PRODNET_PUBLIC = new byte[]{4, -120, -78, 30};
    private static final byte[] TESTNET_PUBLIC = new byte[]{4, 53, -121, -49};
    private static final byte[] PRODNET_PRIVATE = new byte[]{4, -120, -83, -28};
    private static final byte[] TESTNET_PRIVATE = new byte[]{4, 53, -125, -108};

    public void toCustomByteFormat(ByteWriter writer) {
        if (this.isPrivateHdKeyNode()) {
            writer.put((byte)1);
            Preconditions.checkArgument((this._privateKey.getPrivateKeyBytes().length == 32 ? 1 : 0) != 0);
            writer.putBytes(this._privateKey.getPrivateKeyBytes());
        } else {
            writer.put((byte)0);
        }
        Preconditions.checkArgument((this._publicKey.getPublicKeyBytes().length == 33 ? 1 : 0) != 0);
        writer.putBytes(this._publicKey.getPublicKeyBytes());
        writer.putBytes(this._chainCode);
        writer.putIntLE(this._depth);
        writer.putIntLE(this._parentFingerprint);
        writer.putIntLE(this._index);
    }

    public byte[] toCustomByteFormat() {
        ByteWriter writer = new ByteWriter(1024);
        this.toCustomByteFormat(writer);
        return writer.toBytes();
    }

    public static HdKeyNode fromCustomByteformat(byte[] bytes) throws ByteReader.InsufficientBytesException {
        return HdKeyNode.fromCustomByteformat(new ByteReader(bytes));
    }

    public static HdKeyNode fromCustomByteformat(ByteReader reader) throws ByteReader.InsufficientBytesException {
        boolean hasPrivateKey;
        boolean bl = hasPrivateKey = reader.get() == 1;
        if (hasPrivateKey) {
            InMemoryPrivateKey privateKey = new InMemoryPrivateKey(reader.getBytes(32), reader.getBytes(33));
            return new HdKeyNode(privateKey, reader.getBytes(32), reader.getIntLE(), reader.getIntLE(), reader.getIntLE());
        }
        return new HdKeyNode(new PublicKey(reader.getBytes(33)), reader.getBytes(32), reader.getIntLE(), reader.getIntLE(), reader.getIntLE());
    }

    HdKeyNode(InMemoryPrivateKey privateKey, byte[] chainCode, int depth, int parentFingerprint, int index) {
        this._privateKey = privateKey;
        this._publicKey = this._privateKey.getPublicKey();
        this._chainCode = chainCode;
        this._depth = depth;
        this._parentFingerprint = parentFingerprint;
        this._index = index;
    }

    public HdKeyNode(PublicKey publicKey, byte[] chainCode, int depth, int parentFingerprint, int index) {
        this._privateKey = null;
        this._publicKey = publicKey;
        this._chainCode = chainCode;
        this._depth = depth;
        this._parentFingerprint = parentFingerprint;
        this._index = index;
    }

    public static HdKeyNode fromSeed(byte[] seed) throws KeyGenerationException {
        Preconditions.checkArgument((seed.length * 8 >= 128 ? 1 : 0) != 0, (Object)"seed must be larger than 128");
        Preconditions.checkArgument((seed.length * 8 <= 512 ? 1 : 0) != 0, (Object)"seed must be smaller than 512");
        byte[] I = Hmac.hmacSha512(HdKeyNode.asciiStringToBytes(BITCOIN_SEED), seed);
        byte[] IL = BitUtils.copyOfRange(I, 0, 32);
        BigInteger k = new BigInteger(1, IL);
        if (k.compareTo(Parameters.n) >= 0) {
            throw new KeyGenerationException("An unlikely thing happened: The derived key is larger than the N modulus of the curve");
        }
        if (k.equals(BigInteger.ZERO)) {
            throw new KeyGenerationException("An unlikely thing happened: The derived key is zero");
        }
        InMemoryPrivateKey privateKey = new InMemoryPrivateKey(IL, true);
        byte[] IR = BitUtils.copyOfRange(I, 32, 64);
        return new HdKeyNode(privateKey, IR, 0, 0, 0);
    }

    public boolean isPrivateHdKeyNode() {
        return this._privateKey != null;
    }

    public HdKeyNode getPublicNode() {
        return new HdKeyNode(this._publicKey, this._chainCode, this._depth, this._parentFingerprint, this._index);
    }

    public InMemoryPrivateKey createChildPrivateKey(int index) throws KeyGenerationException {
        if (!this.isPrivateHdKeyNode()) {
            throw new KeyGenerationException("Not a private HD key node");
        }
        return this.createChildNode((int)index)._privateKey;
    }

    public PublicKey createChildPublicKey(int index) throws KeyGenerationException {
        return this.createChildNode((int)index)._publicKey;
    }

    public HdKeyNode createChildNode(HdKeyPath keyPath) {
        List<Integer> addrN = keyPath.getAddressN();
        HdKeyNode ak = this;
        for (Integer i : addrN) {
            ak = ak.createChildNode(i);
        }
        return ak;
    }

    public HdKeyNode createHardenedChildNode(int index) throws KeyGenerationException {
        return this.createChildNode(index | Integer.MIN_VALUE);
    }

    public HdKeyNode createChildNode(int index) throws KeyGenerationException {
        byte[] data;
        ByteWriter writer;
        byte[] publicKeyBytes = this._publicKey.getPublicKeyBytes();
        if (0 == (index & Integer.MIN_VALUE)) {
            writer = new ByteWriter(publicKeyBytes.length + 4);
            writer.putBytes(publicKeyBytes);
            writer.putIntBE(index);
            data = writer.toBytes();
        } else {
            if (!this.isPrivateHdKeyNode()) {
                throw new KeyGenerationException("Cannot generate hardened HD key node from pubic HD key node");
            }
            writer = new ByteWriter(37);
            writer.put((byte)0);
            writer.putBytes(this._privateKey.getPrivateKeyBytes());
            writer.putIntBE(index);
            data = writer.toBytes();
        }
        byte[] l = Hmac.hmacSha512(this._chainCode, data);
        byte[] lL = BitUtils.copyOfRange(l, 0, 32);
        byte[] lR = BitUtils.copyOfRange(l, 32, 64);
        BigInteger m = new BigInteger(1, lL);
        if (m.compareTo(Parameters.n) >= 0) {
            throw new KeyGenerationException("An unlikely thing happened: A key derivation parameter is larger than the N modulus of the curve");
        }
        if (this.isPrivateHdKeyNode()) {
            BigInteger kpar = new BigInteger(1, this._privateKey.getPrivateKeyBytes());
            BigInteger k = m.add(kpar).mod(Parameters.n);
            if (k.equals(BigInteger.ZERO)) {
                throw new KeyGenerationException("An unlikely thing happened: The derived key is zero");
            }
            byte[] privateKeyBytes = this.bigIntegerTo32Bytes(k);
            InMemoryPrivateKey key = new InMemoryPrivateKey(privateKeyBytes, true);
            return new HdKeyNode(key, lR, this._depth + 1, this.getFingerprint(), index);
        }
        Point q = Parameters.G.multiply(m).add(Parameters.curve.decodePoint(this._publicKey.getPublicKeyBytes()));
        if (q.isInfinity()) {
            throw new KeyGenerationException("An unlikely thing happened: Invalid key point at infinity");
        }
        PublicKey newPublicKey = new PublicKey(new Point(Parameters.curve, q.getX(), q.getY(), true).getEncoded());
        return new HdKeyNode(newPublicKey, lR, this._depth + 1, this.getFingerprint(), index);
    }

    private byte[] bigIntegerTo32Bytes(BigInteger b) {
        byte[] bytes = b.toByteArray();
        Preconditions.checkArgument((bytes.length <= 33 ? 1 : 0) != 0);
        if (bytes.length == 33) {
            Preconditions.checkArgument((bytes[0] == 0 ? 1 : 0) != 0);
            return BitUtils.copyOfRange(bytes, 1, 33);
        }
        byte[] result = new byte[32];
        System.arraycopy(bytes, 0, result, result.length - bytes.length, bytes.length);
        return result;
    }

    public int getFingerprint() {
        byte[] hash = this._publicKey.getPublicKeyHash();
        int fingerprint = (hash[0] & 0xFF) << 24;
        fingerprint += (hash[1] & 0xFF) << 16;
        fingerprint += (hash[2] & 0xFF) << 8;
        return fingerprint += hash[3] & 0xFF;
    }

    public InMemoryPrivateKey getPrivateKey() throws KeyGenerationException {
        if (!this.isPrivateHdKeyNode()) {
            throw new KeyGenerationException("Not a private HD key node");
        }
        return this._privateKey;
    }

    public PublicKey getPublicKey() {
        return this._publicKey;
    }

    public String serialize(NetworkParameters network) throws KeyGenerationException {
        ByteWriter writer = new ByteWriter(77);
        if (network.isProdnet()) {
            writer.putBytes(this.isPrivateHdKeyNode() ? PRODNET_PRIVATE : PRODNET_PUBLIC);
        } else {
            writer.putBytes(this.isPrivateHdKeyNode() ? TESTNET_PRIVATE : TESTNET_PUBLIC);
        }
        writer.put((byte)(this._depth & 0xFF));
        writer.putIntBE(this._parentFingerprint);
        writer.putIntBE(this._index);
        writer.putBytes(this._chainCode);
        if (this.isPrivateHdKeyNode()) {
            writer.put((byte)0);
            writer.putBytes(this._privateKey.getPrivateKeyBytes());
        } else {
            writer.putBytes(this._publicKey.getPublicKeyBytes());
        }
        return Base58.encodeWithChecksum(writer.toBytes());
    }

    public static HdKeyNode parse(String string, NetworkParameters network) throws KeyGenerationException {
        try {
            boolean isPrivate;
            byte[] bytes = Base58.decodeChecked(string);
            if (bytes == null) {
                throw new KeyGenerationException("Invalid checksum");
            }
            if (bytes.length != 78) {
                throw new KeyGenerationException("Invalid size");
            }
            ByteReader reader = new ByteReader(bytes);
            byte[] magic = reader.getBytes(4);
            if (BitUtils.areEqual(magic, PRODNET_PRIVATE)) {
                if (!network.isProdnet()) {
                    throw new KeyGenerationException("Invalid network");
                }
                isPrivate = true;
            } else if (BitUtils.areEqual(magic, PRODNET_PUBLIC)) {
                if (!network.isProdnet()) {
                    throw new KeyGenerationException("Invalid network");
                }
                isPrivate = false;
            } else if (BitUtils.areEqual(magic, TESTNET_PRIVATE)) {
                if (network.isProdnet()) {
                    throw new KeyGenerationException("Invalid network");
                }
                isPrivate = true;
            } else if (BitUtils.areEqual(magic, TESTNET_PUBLIC)) {
                if (network.isProdnet()) {
                    throw new KeyGenerationException("Invalid network");
                }
                isPrivate = false;
            } else {
                throw new KeyGenerationException("Invalid magic header for HD key node");
            }
            int depth = reader.get() & 0xFF;
            int parentFingerprint = reader.getIntBE();
            int index = reader.getIntBE();
            byte[] chainCode = reader.getBytes(32);
            if (isPrivate) {
                if (reader.get() != 0) {
                    throw new KeyGenerationException("Invalid private key");
                }
                InMemoryPrivateKey privateKey = new InMemoryPrivateKey(reader.getBytes(32), true);
                return new HdKeyNode(privateKey, chainCode, depth, parentFingerprint, index);
            }
            PublicKey publicKey = new PublicKey(reader.getBytes(33));
            return new HdKeyNode(publicKey, chainCode, depth, parentFingerprint, index);
        }
        catch (ByteReader.InsufficientBytesException e) {
            throw new KeyGenerationException("Insufficient bytes in serialization");
        }
    }

    public String toString() {
        return "Fingerprint: " + Integer.toString(this.getFingerprint());
    }

    private static byte[] asciiStringToBytes(String string) {
        try {
            return string.getBytes("US-ASCII");
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException();
        }
    }

    public int hashCode() {
        return this._publicKey.hashCode();
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof HdKeyNode)) {
            return false;
        }
        HdKeyNode other = (HdKeyNode)obj;
        if (!this._publicKey.equals(other._publicKey)) {
            return false;
        }
        if (this._depth != other._depth) {
            return false;
        }
        if (this._parentFingerprint != other._parentFingerprint) {
            return false;
        }
        if (this._index != other._index) {
            return false;
        }
        if (!BitUtils.areEqual(this._chainCode, other._chainCode)) {
            return false;
        }
        return this.isPrivateHdKeyNode() == other.isPrivateHdKeyNode();
    }

    public int getIndex() {
        return this._index;
    }

    public int getParentFingerprint() {
        return this._parentFingerprint;
    }

    public int getDepth() {
        return this._depth;
    }

    public UUID getUuid() {
        byte[] publicKeyBytes = this.getPublicKey().getPublicKeyBytes();
        return new UUID(BitUtils.uint64ToLong(publicKeyBytes, 8), BitUtils.uint64ToLong(publicKeyBytes, 16));
    }

    public static class KeyGenerationException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        public KeyGenerationException(String message) {
            super(message);
        }
    }
}

