/*
This file is part of the BrowserMob Proxy Client project by Ivan De Marino (http://ivandemarino.me).

Copyright (c) 2014, Ivan De Marino (http://ivandemarino.me)
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package com.github.detro.browsermobproxyclient;

import com.github.detro.browsermobproxyclient.exceptions.*;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import org.apache.http.*;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.*;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.openqa.selenium.Proxy;

import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

/**
 * Client API for controlling a Proxy created via BrowserMob Proxy REST API.
 *
 * The target use is to generate a JSON in HAR format, based on the traffic
 * sent through the proxy up to that point (i.e. traffic usually generated by
 * an HTTP User Agent / Browser).
 * </p>
 *
 * The HAR produced implements the HAR (HTTP ARchive) format 1.2, defined
 * <a href="http://www.softwareishard.com/blog/har-12-spec/"
 *  target="_blank">here</a>.
 * </p>
 *
 * It uses the REST API provided by BrowserMob Proxy to create a new HTTP Proxy.
 * Maps all the REST API to a usable client object that internally only
 * deals with HTTP REST-ful calls.
 * </p>
 *
 * IMPORTANT: the REST API runs on a specific port (provided at Construction time), but
 * all the instances of this class will then be assigned a second Port towards which
 * a Driver/Browser will be pointed.
 * This means that a single BrowserMob Proxy instance listens to N+1 ports:
 * <ul>
 *     <li>1 port for the REST API</li>
 *     <li>N for N Proxy</li>
 * </ul>
 * This is so that every Driver can have an isolated Proxy on it's own.
 * </p>
 *
 * Starting and stopping BrowserMob Proxy REST API needs to be done in another place:
 * this class assumes that such REST API is up and running and consumes it.
 */
public class BMPCProxy {

    private static final Gson GSON = new GsonBuilder()
            .serializeNulls()
            .create();

    private final CloseableHttpClient HTTPclient = HttpClients.createSystem();

    private final String APIHost;
    private final int APIPort;
    private final int proxyPort;


    /**
     * Create a BrowserMob Proxy Instance
     *
     * TODO Support for 'port' and 'bindAddress' parameters when creating a new Proxy Client
     *
     * @param apiHost Host were BrowserMob Proxy is running
     * @param apiPort Port were BrowserMob Proxy REST API is listening
     */
    public BMPCProxy(String apiHost, int apiPort) {
        this.APIHost = apiHost;
        this.APIPort = apiPort;

        // Store newly created Proxy Port
        this.proxyPort = requestNewProxyPort(null);
    }

    /**
     * Create a BrowserMob Proxy Instance
     *
     * TODO Support for 'port' and 'bindAddress' parameters when creating a new Proxy Client
     *
     * @param apiHost Host were BrowserMob Proxy is running
     * @param apiPort Port were BrowserMob Proxy REST API is listening
     * @param upstreamProxyHostAndPort New proxy will use this HTTP Proxy instead of
     *                          directly connecting to the target address.
     *                          Ex. you need BrowserMob Proxy to still pass
     *                          a company proxy to reach the outside network.
     *                          IMPORTANT: format must be "HOST:PORT".
     */
    public BMPCProxy(String apiHost, int apiPort, String upstreamProxyHostAndPort) {
        this.APIHost = apiHost;
        this.APIPort = apiPort;

        // Store newly created Proxy Port
        this.proxyPort = requestNewProxyPort(upstreamProxyHostAndPort);
    }

    /**
     * Create a BrowserMob Proxy Instance.
     *
     * This constructor will not actually request a NEW Proxy: it will
     * just assume the one provided (via the <code>proxyPort</code> parameter)
     * exists. If it doesn't, behaviour is undefined.
     *
     * TODO Support for 'port' and 'bindAddress' parameters when creating a new Proxy Client
     *
     * @param apiHost Host were BrowserMob Proxy is running
     * @param apiPort Port were BrowserMob Proxy REST API is listening
     * @param proxyPort Existing Proxy Port to connect to
     */
    public BMPCProxy(String apiHost, int apiPort, int proxyPort) {
        this.APIHost = apiHost;
        this.APIPort = apiPort;
        this.proxyPort = proxyPort;
    }

    private int requestNewProxyPort(String upstreamHttpProxy) {
        try {
            // Request BMP to create a new Proxy
            HttpPost request = new HttpPost(requestURIBuilder()
                    .setPath("/proxy")
                    .build());

            // Add form parameters to the request
            applyFormParamsToHttpRequest(request,
                    new BasicNameValuePair("httpProxy", upstreamHttpProxy));

            // Execute request
            CloseableHttpResponse response = HTTPclient.execute(request);

            // Parse response into JSON
            JsonObject createProxyResponseJson = httpResponseToJsonObject(response);
            response.close();
            if (null == createProxyResponseJson || !createProxyResponseJson.isJsonObject()) {
                throw new RuntimeException("Unexpected Response JSON: " + createProxyResponseJson);
            }

            return  createProxyResponseJson.getAsJsonPrimitive("port").getAsInt();
        } catch (Exception e) {
            throw new BMPCUnableToConnectException(String.format(
                    "Unable to connect to BMP Proxy at '%s:%s'",
                    APIHost,
                    APIPort
            ), e);
        }
    }

    @Override
    protected void finalize() throws Throwable {
        close();
        super.finalize();
    }

    /**
     * Returns the Proxy this client wraps, in form of a Selenium Proxy configuration object.
     *
     * @return Selenium Proxy configuration object
     */
    public Proxy asSeleniumProxy() {
        Proxy seleniumProxyConfig = new Proxy();

        seleniumProxyConfig.setHttpProxy(asHttpProxy());

        return seleniumProxyConfig;
    }

    /**
     * Returns HTTP URL to this Proxy
     *
     * @return HTTP URL to this Proxy
     */
    public String asHttpProxy() {
        return String.format("http://%s", asHostAndPort());
    }

    /**
     * Returns "[HOST]:[PORT]" of this Proxy
     *
     * @return "[HOST]:[PORT]" of this Proxy
     */
    public String asHostAndPort() {
        return String.format("%s:%s", APIHost, proxyPort);
    }

    /**
     * Host on which BrowserMob Proxy REST API are listening.
     *
     * @return Host on which the REST API are listening.
     */
    public String getAPIHost() {
        return APIHost;
    }

    /**
     * Port on which BrowserMob Proxy REST API are listening.
     * This is NOT the port of the proxy itself.
     *
     * @return Port on which the REST API are listening.
     */
    public int getAPIPort() {
        return APIPort;
    }

    /**
     * Port on which the Proxy is listening.
     * This is where the driver will connect.
     *
     * @return Port on which the proxy is listening
     */
    public int getProxyPort() {
        return proxyPort;
    }

    /**
     * @see BMPCProxy#newHar(String, boolean, boolean, boolean)
     */
    public JsonObject newHar() {
        return newHar(null, false, false, false);
    }

    /**
     * @see BMPCProxy#newHar(String, boolean, boolean, boolean)
     */
    public JsonObject newHar(String initialPageRef) {
        return newHar(initialPageRef, false, false, false);
    }

    /**
     * @see BMPCProxy#newHar(String, boolean, boolean, boolean)
     */
    public JsonObject newHar(String initialPageRef,
                             boolean captureHeaders) {
         return newHar(initialPageRef, captureHeaders, false, false);
    }

    /**
     * Creates a new HAR attached to the proxy.
     *
     * @param initialPageRef Name of the first pageRef that should be used by
     *                       the HAR. If "null", default to "Page 1"
     * @param captureHeaders Enables capturing of HTTP Headers
     * @param captureContent Enables capturing of HTTP Response Content (body)
     * @param captureBinaryContent Enabled capturing of HTTP Response
     *                             Binary Content (in bse64 encoding)
     * @return JsonObject HAR response if this proxy was previously collecting
     *         another HAR, effectively considering that concluded.
     *         "null" otherwise.
     */
    public JsonObject newHar(String initialPageRef,
                             boolean captureHeaders,
                             boolean captureContent,
                             boolean captureBinaryContent) {
        try {
            // Request BMP to create a new HAR for this Proxy
            HttpPut request = new HttpPut(requestURIBuilder()
                    .setPath(proxyURIPath() + "/har")
                    .build());

            // Add form parameters to the request
            applyFormParamsToHttpRequest(request,
                    new BasicNameValuePair("initialPageRef", initialPageRef),
                    new BasicNameValuePair("captureHeaders", Boolean.toString(captureHeaders)),
                    new BasicNameValuePair("captureContent", Boolean.toString(captureContent)),
                    new BasicNameValuePair("captureBinaryContent", Boolean.toString(captureBinaryContent)));

            // Execute request
            CloseableHttpResponse response = HTTPclient.execute(request);

            // Parse response into JSON
            JsonObject previousHar = httpResponseToJsonObject(response);
            // Close HTTP Response
            response.close();

            return previousHar;
        } catch (URISyntaxException|IOException e) {
            throw new BMPCUnableToCreateHarException(e);
        }
    }

    /**
     * @see BMPCProxy#newPage(String)
     */
    public void newPage() {
        newPage(null);
    }

    /**
     * Starts a new page on the existing HAR.
     * All the traffic recorded in the HAR from this point on will be
     * considered part of this new Page.
     *
     * @param pageRef Name of this new pageRef that should be used by the HAR.
     *                If "null" defaults to "Page N", where "N" is the number
     *                of pages so far.
     */
    public void newPage(String pageRef) {
        try {
            // Request BMP to create a new HAR for this Proxy
            HttpPut request = new HttpPut(requestURIBuilder()
                    .setPath(proxyURIPath() + "/har/pageRef")
                    .build());

            // Add form parameters to the request
            applyFormParamsToHttpRequest(request,
                    new BasicNameValuePair("pageRef", pageRef));

            // Execute request
            CloseableHttpResponse response = HTTPclient.execute(request);

            // Check request was successful
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != 200) {
                throw new BMPCUnableToCreatePageException(
                        "Invalid HTTP Response when attempting to create"
                                + "new Page in HAR: "
                                + statusCode
                );
            }

            // Close HTTP Response
            response.close();

        } catch (URISyntaxException|IOException e) {
            throw new BMPCUnableToCreateHarException(e);
        }
    }

    /**
     * Produces the HAR so far, based on the traffic generated so far.
     *
     * Contains Headers, Content and so forth, based on the parameters
     * used passed to {@link BMPCProxy#newHar(String, boolean, boolean, boolean)}.
     *
     * @return JsonObject in HAR format.
     */
    public JsonObject har() {
        try {
            // Request BMP to create a new HAR for this Proxy
            HttpGet request = new HttpGet(requestURIBuilder()
                    .setPath(proxyURIPath() + "/har")
                    .build());

            // Execute request
            CloseableHttpResponse response = HTTPclient.execute(request);

            // Parse response into JSON
            JsonObject har = httpResponseToJsonObject(response);
            // Close HTTP Response
            response.close();

            return har;
        } catch (URISyntaxException|IOException e) {
            throw new BMPCUnableToCreateHarException(e);
        }
    }

    /**
     * No traffic has passed through this proxy yet.
     *
     * This means that no HAR has been created yet and no traffic has
     * been recoreded yet either.
     *
     * @return "true" if it has not been used yet.
     */
    public boolean notUsedYet() {
        return null == har();
    }

    /**
     * Closes the Proxy and releases the Proxy Client resources (ex. HTTPClient).
     *
     * After this call the Proxy Client is rendered unusable and references
     * to it should be discarded.
     */
    public void close() {
        try {
            // Request BMP to create a new HAR for this Proxy
            HttpDelete shutdownProxyDELETE = new HttpDelete(requestURIBuilder()
                    .setPath(proxyURIPath())
                    .build());

            // Execute request
            CloseableHttpResponse response = HTTPclient.execute(shutdownProxyDELETE);

            // Check request was successful
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != 200) {
                throw new BMPCUnableToCloseProxyException(String.format(
                        "Invalid HTTP Response when attempting to close " +
                                "Proxy '%d'. Status code: %d",
                        proxyPort, statusCode));
            }
            // Close HTTP Response
            response.close();

            // Close HTTP Client
            HTTPclient.close();
        } catch (URISyntaxException|IOException e) {
            throw new BMPCUnableToCloseProxyException(e);
        }
    }

    private JsonObject httpResponseToJsonObject(HttpResponse response) {
        int statusCode = response.getStatusLine().getStatusCode();
        HttpEntity entity = response.getEntity();

        // Workout if we got back a good response
        if (statusCode < 200 || statusCode >= 300) {
            throw new RuntimeException(String.format(
                    "Unexpected HTTP Status Code %d. Response: %s",
                    statusCode,
                    response
            ));
        }

        if (statusCode == 204) {
            // Request successful but the response has No Content
            return null;
        } else {
            try {
                // Workout the charset
                Charset charset = ContentType.getOrDefault(entity).getCharset();

                // De-serialize
                return GSON.fromJson(
                        new InputStreamReader(entity.getContent(), null != charset ? charset : Consts.UTF_8),
                        JsonObject.class);
            } catch (IOException e) {
                throw new BMPCUnableToParseJsonResponseException(e);
            }
        }
    }

    private void applyFormParamsToHttpRequest(HttpEntityEnclosingRequestBase httpReq, NameValuePair ... pairs) {
        // Filter out null-value Pairs
        List<NameValuePair> formParams = new ArrayList<NameValuePair>();
        for (NameValuePair pair : pairs) {
            if (pair.getValue() != null) {
                formParams.add(pair);
            }
        }

        // Encode as entity and set
        UrlEncodedFormEntity entity = new UrlEncodedFormEntity(formParams, Consts.UTF_8);
        httpReq.setEntity(entity);
    }

    private URIBuilder requestURIBuilder() {
        return new URIBuilder()
                .setScheme("http")
                .setHost(APIHost)
                .setPort(APIPort);
    }

    private String proxyURIPath() {
        return String.format("/proxy/%d", proxyPort);
    }

    // TODO Implement more API based on what's documented at:
    //   https://github.com/lightbody/browsermob-proxy/blob/master/README.md#rest-api
    // - whitelist PUT/DELETE
    // - blacklist PUT/DELETE
    // - bandwidth limit PUT
    // - HTTP headers POST
    // - host/DNS override POST
    // - basic auth POST
    // - wait PUT
    // - timeouts PUT
    // - url rewrite PUT/DELETE
    // - retry count PUT
    // - DNS cache flush DELETE
}
