/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.BlockStorageLocation;
import org.apache.hadoop.fs.HdfsVolumeId;
import org.apache.hadoop.fs.VolumeId;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.hdfs.protocol.ClientDatanodeProtocol;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.HdfsBlocksMetadata;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenIdentifier;
import org.apache.hadoop.hdfs.security.token.block.InvalidBlockTokenException;
import org.apache.hadoop.ipc.RPC;
import org.apache.hadoop.security.token.Token;
import org.apache.htrace.core.SpanId;
import org.apache.htrace.core.TraceScope;
import org.apache.htrace.core.Tracer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
class BlockStorageLocationUtil {
    static final Logger LOG = LoggerFactory.getLogger(BlockStorageLocationUtil.class);

    BlockStorageLocationUtil() {
    }

    private static List<VolumeBlockLocationCallable> createVolumeBlockLocationCallables(Configuration conf, Map<DatanodeInfo, List<LocatedBlock>> datanodeBlocks, int timeout, boolean connectToDnViaHostname, Tracer tracer, SpanId parentSpanId) {
        if (datanodeBlocks.isEmpty()) {
            return Lists.newArrayList();
        }
        ArrayList<VolumeBlockLocationCallable> callables = new ArrayList<VolumeBlockLocationCallable>();
        for (Map.Entry<DatanodeInfo, List<LocatedBlock>> entry : datanodeBlocks.entrySet()) {
            DatanodeInfo datanode = entry.getKey();
            List<LocatedBlock> locatedBlocks = entry.getValue();
            if (locatedBlocks.isEmpty()) continue;
            String poolId = locatedBlocks.get(0).getBlock().getBlockPoolId();
            for (LocatedBlock lb : locatedBlocks) {
                if (poolId.equals(lb.getBlock().getBlockPoolId())) continue;
                throw new IllegalArgumentException("All blocks to be queried must be in the same block pool: " + locatedBlocks.get(0).getBlock() + " and " + lb + " are from different pools.");
            }
            long[] blockIds = new long[locatedBlocks.size()];
            int i = 0;
            ArrayList<Token<BlockTokenIdentifier>> dnTokens = new ArrayList<Token<BlockTokenIdentifier>>(locatedBlocks.size());
            for (LocatedBlock b : locatedBlocks) {
                blockIds[i++] = b.getBlock().getBlockId();
                dnTokens.add(b.getBlockToken());
            }
            VolumeBlockLocationCallable callable = new VolumeBlockLocationCallable(conf, datanode, poolId, blockIds, dnTokens, timeout, connectToDnViaHostname, tracer, parentSpanId);
            callables.add(callable);
        }
        return callables;
    }

    static Map<DatanodeInfo, HdfsBlocksMetadata> queryDatanodesForHdfsBlocksMetadata(Configuration conf, Map<DatanodeInfo, List<LocatedBlock>> datanodeBlocks, int poolsize, int timeoutMs, boolean connectToDnViaHostname, Tracer tracer, SpanId parentSpanId) throws InvalidBlockTokenException {
        List<VolumeBlockLocationCallable> callables = BlockStorageLocationUtil.createVolumeBlockLocationCallables(conf, datanodeBlocks, timeoutMs, connectToDnViaHostname, tracer, parentSpanId);
        List<Object> futures = new ArrayList();
        ScheduledThreadPoolExecutor executor = new ScheduledThreadPoolExecutor(poolsize);
        try {
            futures = executor.invokeAll(callables, timeoutMs, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        executor.shutdown();
        HashMap<DatanodeInfo, HdfsBlocksMetadata> metadatas = Maps.newHashMapWithExpectedSize(datanodeBlocks.size());
        for (int i = 0; i < futures.size(); ++i) {
            VolumeBlockLocationCallable callable = callables.get(i);
            DatanodeInfo datanode = callable.getDatanodeInfo();
            Future future = (Future)futures.get(i);
            try {
                HdfsBlocksMetadata metadata = (HdfsBlocksMetadata)future.get();
                metadatas.put(callable.getDatanodeInfo(), metadata);
                continue;
            }
            catch (CancellationException e) {
                LOG.info("Cancelled while waiting for datanode " + datanode.getIpcAddr(false) + ": " + e.toString());
                continue;
            }
            catch (ExecutionException e) {
                Throwable t = e.getCause();
                if (t instanceof InvalidBlockTokenException) {
                    LOG.warn("Invalid access token when trying to retrieve information from datanode " + datanode.getIpcAddr(false));
                    throw (InvalidBlockTokenException)t;
                }
                if (t instanceof UnsupportedOperationException) {
                    LOG.info("Datanode " + datanode.getIpcAddr(false) + " does not support" + " required #getHdfsBlocksMetadata() API");
                    throw (UnsupportedOperationException)t;
                }
                LOG.info("Failed to query block locations on datanode " + datanode.getIpcAddr(false) + ": " + t);
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Could not fetch information from datanode", t);
                continue;
            }
            catch (InterruptedException e) {
                LOG.info("Interrupted while fetching HdfsBlocksMetadata");
            }
        }
        return metadatas;
    }

    static Map<LocatedBlock, List<VolumeId>> associateVolumeIdsWithBlocks(List<LocatedBlock> blocks, Map<DatanodeInfo, HdfsBlocksMetadata> metadatas) {
        HashMap<Long, LocatedBlock> blockIdToLocBlock = new HashMap<Long, LocatedBlock>();
        for (LocatedBlock b : blocks) {
            blockIdToLocBlock.put(b.getBlock().getBlockId(), b);
        }
        HashMap<LocatedBlock, List<VolumeId>> blockVolumeIds = new HashMap<LocatedBlock, List<VolumeId>>();
        for (LocatedBlock locatedBlock : blocks) {
            ArrayList<Object> l = new ArrayList<Object>(locatedBlock.getLocations().length);
            for (int i = 0; i < locatedBlock.getLocations().length; ++i) {
                l.add(null);
            }
            blockVolumeIds.put(locatedBlock, l);
        }
        for (Map.Entry entry : metadatas.entrySet()) {
            DatanodeInfo datanode = (DatanodeInfo)entry.getKey();
            HdfsBlocksMetadata metadata = (HdfsBlocksMetadata)entry.getValue();
            if (metadata == null) continue;
            long[] metaBlockIds = metadata.getBlockIds();
            List<byte[]> metaVolumeIds = metadata.getVolumeIds();
            List<Integer> metaVolumeIndexes = metadata.getVolumeIndexes();
            for (int j = 0; j < metaBlockIds.length; ++j) {
                int volumeIndex = metaVolumeIndexes.get(j);
                long blockId = metaBlockIds[j];
                if (volumeIndex == Integer.MAX_VALUE || volumeIndex >= metaVolumeIds.size() || !blockIdToLocBlock.containsKey(blockId)) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug("No data for block " + blockId);
                    continue;
                }
                byte[] volumeId = metaVolumeIds.get(volumeIndex);
                HdfsVolumeId id = new HdfsVolumeId(volumeId);
                LocatedBlock locBlock = (LocatedBlock)blockIdToLocBlock.get(blockId);
                DatanodeInfo[] dnInfos = locBlock.getLocations();
                int index = -1;
                for (int k = 0; k < dnInfos.length; ++k) {
                    if (!dnInfos[k].equals(datanode)) continue;
                    index = k;
                    break;
                }
                if (index < 0) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug("Datanode responded with a block volume id we did not request, omitting.");
                    continue;
                }
                List volumeIds = (List)blockVolumeIds.get(locBlock);
                volumeIds.set(index, id);
            }
        }
        return blockVolumeIds;
    }

    static BlockStorageLocation[] convertToVolumeBlockLocations(List<LocatedBlock> blocks, Map<LocatedBlock, List<VolumeId>> blockVolumeIds) throws IOException {
        BlockLocation[] locations = DFSUtilClient.locatedBlocks2Locations(blocks);
        ArrayList<BlockStorageLocation> volumeBlockLocs = new ArrayList<BlockStorageLocation>(locations.length);
        for (int i = 0; i < locations.length; ++i) {
            LocatedBlock locBlock = blocks.get(i);
            List<VolumeId> volumeIds = blockVolumeIds.get(locBlock);
            BlockStorageLocation bsLoc = new BlockStorageLocation(locations[i], volumeIds.toArray(new VolumeId[0]));
            volumeBlockLocs.add(bsLoc);
        }
        return volumeBlockLocs.toArray(new BlockStorageLocation[0]);
    }

    private static class VolumeBlockLocationCallable
    implements Callable<HdfsBlocksMetadata> {
        private final Configuration configuration;
        private final int timeout;
        private final DatanodeInfo datanode;
        private final String poolId;
        private final long[] blockIds;
        private final List<Token<BlockTokenIdentifier>> dnTokens;
        private final boolean connectToDnViaHostname;
        private final Tracer tracer;
        private final SpanId parentSpanId;

        VolumeBlockLocationCallable(Configuration configuration, DatanodeInfo datanode, String poolId, long[] blockIds, List<Token<BlockTokenIdentifier>> dnTokens, int timeout, boolean connectToDnViaHostname, Tracer tracer, SpanId parentSpanId) {
            this.configuration = configuration;
            this.timeout = timeout;
            this.datanode = datanode;
            this.poolId = poolId;
            this.blockIds = blockIds;
            this.dnTokens = dnTokens;
            this.connectToDnViaHostname = connectToDnViaHostname;
            this.tracer = tracer;
            this.parentSpanId = parentSpanId;
        }

        public DatanodeInfo getDatanodeInfo() {
            return this.datanode;
        }

        /*
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        @Override
        public HdfsBlocksMetadata call() throws Exception {
            HdfsBlocksMetadata metadata = null;
            ClientDatanodeProtocol cdp = null;
            TraceScope scope = this.tracer.newScope("getHdfsBlocksMetadata", this.parentSpanId);
            try {
                cdp = DFSUtilClient.createClientDatanodeProtocolProxy(this.datanode, this.configuration, this.timeout, this.connectToDnViaHostname);
                metadata = cdp.getHdfsBlocksMetadata(this.poolId, this.blockIds, this.dnTokens);
                scope.close();
                if (cdp == null) return metadata;
            }
            catch (IOException e) {
                try {
                    throw e;
                }
                catch (Throwable throwable) {
                    scope.close();
                    if (cdp == null) throw throwable;
                    RPC.stopProxy(cdp);
                    throw throwable;
                }
            }
            RPC.stopProxy(cdp);
            return metadata;
        }
    }
}

