/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.client.impl;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSOutputStream;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class LeaseRenewer {
    static final Logger LOG = LoggerFactory.getLogger(LeaseRenewer.class);
    static final long LEASE_RENEWER_GRACE_DEFAULT = 60000L;
    static final long LEASE_RENEWER_SLEEP_DEFAULT = 1000L;
    private long emptyTime = Long.MAX_VALUE;
    private long renewal = 30000L;
    private Daemon daemon = null;
    private int currentId = 0;
    private long gracePeriod;
    private long sleepPeriod;
    private final Factory.Key factorykey;
    private final List<DFSClient> dfsclients = new ArrayList<DFSClient>();
    private final String instantiationTrace;

    public static LeaseRenewer getInstance(String authority, UserGroupInformation ugi, DFSClient dfsc) {
        LeaseRenewer r = Factory.INSTANCE.get(authority, ugi);
        r.addClient(dfsc);
        return r;
    }

    private LeaseRenewer(Factory.Key factorykey) {
        this.factorykey = factorykey;
        this.unsyncSetGraceSleepPeriod(60000L);
        this.instantiationTrace = LOG.isTraceEnabled() ? StringUtils.stringifyException(new Throwable("TRACE")) : null;
    }

    private synchronized long getRenewalTime() {
        return this.renewal;
    }

    @VisibleForTesting
    public synchronized void setRenewalTime(long renewal) {
        this.renewal = renewal;
    }

    private synchronized void addClient(DFSClient dfsc) {
        long half;
        for (DFSClient c : this.dfsclients) {
            if (c != dfsc) continue;
            return;
        }
        this.dfsclients.add(dfsc);
        int hdfsTimeout = dfsc.getConf().getHdfsTimeout();
        if (hdfsTimeout > 0 && (half = (long)(hdfsTimeout / 2)) < this.renewal) {
            this.renewal = half;
        }
    }

    private synchronized boolean clientsRunning() {
        Iterator<DFSClient> i = this.dfsclients.iterator();
        while (i.hasNext()) {
            if (i.next().isClientRunning()) continue;
            i.remove();
        }
        return !this.dfsclients.isEmpty();
    }

    private synchronized long getSleepPeriod() {
        return this.sleepPeriod;
    }

    synchronized void setGraceSleepPeriod(long gracePeriod) {
        this.unsyncSetGraceSleepPeriod(gracePeriod);
    }

    private void unsyncSetGraceSleepPeriod(long gracePeriod) {
        if (gracePeriod < 100L) {
            throw new HadoopIllegalArgumentException(gracePeriod + " = gracePeriod < 100ms is too small.");
        }
        this.gracePeriod = gracePeriod;
        long half = gracePeriod / 2L;
        this.sleepPeriod = half < 1000L ? half : 1000L;
    }

    synchronized boolean isRunning() {
        return this.daemon != null && this.daemon.isAlive();
    }

    public boolean isEmpty() {
        return this.dfsclients.isEmpty();
    }

    synchronized String getDaemonName() {
        return this.daemon.getName();
    }

    private synchronized boolean isRenewerExpired() {
        return this.emptyTime != Long.MAX_VALUE && Time.monotonicNow() - this.emptyTime > this.gracePeriod;
    }

    public synchronized void put(long inodeId, DFSOutputStream out, DFSClient dfsc) {
        if (dfsc.isClientRunning()) {
            if (!this.isRunning() || this.isRenewerExpired()) {
                final int id = ++this.currentId;
                this.daemon = new Daemon(new Runnable(){

                    /*
                     * WARNING - Removed try catching itself - possible behaviour change.
                     */
                    @Override
                    public void run() {
                        try {
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("Lease renewer daemon for " + LeaseRenewer.this.clientsString() + " with renew id " + id + " started");
                            }
                            LeaseRenewer.this.run(id);
                        }
                        catch (InterruptedException e) {
                            LOG.debug("LeaseRenewer is interrupted.", (Throwable)e);
                        }
                        finally {
                            LeaseRenewer leaseRenewer = LeaseRenewer.this;
                            synchronized (leaseRenewer) {
                                Factory.INSTANCE.remove(LeaseRenewer.this);
                            }
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("Lease renewer daemon for " + LeaseRenewer.this.clientsString() + " with renew id " + id + " exited");
                            }
                        }
                    }

                    public String toString() {
                        return String.valueOf(LeaseRenewer.this);
                    }
                });
                this.daemon.start();
            }
            dfsc.putFileBeingWritten(inodeId, out);
            this.emptyTime = Long.MAX_VALUE;
        }
    }

    @VisibleForTesting
    synchronized void setEmptyTime(long time) {
        this.emptyTime = time;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void closeFile(long inodeId, DFSClient dfsc) {
        dfsc.removeFileBeingWritten(inodeId);
        LeaseRenewer leaseRenewer = this;
        synchronized (leaseRenewer) {
            if (dfsc.isFilesBeingWrittenEmpty()) {
                this.dfsclients.remove(dfsc);
            }
            if (this.emptyTime == Long.MAX_VALUE) {
                for (DFSClient c : this.dfsclients) {
                    if (c.isFilesBeingWrittenEmpty()) continue;
                    return;
                }
                this.emptyTime = Time.monotonicNow();
            }
        }
    }

    public synchronized void closeClient(DFSClient dfsc) {
        this.dfsclients.remove(dfsc);
        if (this.dfsclients.isEmpty()) {
            if (!this.isRunning() || this.isRenewerExpired()) {
                Factory.INSTANCE.remove(this);
                return;
            }
            if (this.emptyTime == Long.MAX_VALUE) {
                this.emptyTime = Time.monotonicNow();
            }
        }
        if (this.renewal == (long)(dfsc.getConf().getHdfsTimeout() / 2)) {
            long min = 60000L;
            for (DFSClient c : this.dfsclients) {
                int timeout = c.getConf().getHdfsTimeout();
                if (timeout <= 0 || (long)timeout >= min) continue;
                min = timeout;
            }
            this.renewal = min / 2L;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void interruptAndJoin() throws InterruptedException {
        Daemon daemonCopy = null;
        LeaseRenewer leaseRenewer = this;
        synchronized (leaseRenewer) {
            if (this.isRunning()) {
                this.daemon.interrupt();
                daemonCopy = this.daemon;
            }
        }
        if (daemonCopy != null) {
            LOG.debug("Wait for lease checker to terminate");
            daemonCopy.join();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void renew() throws IOException {
        ArrayList<DFSClient> copies;
        LeaseRenewer leaseRenewer = this;
        synchronized (leaseRenewer) {
            copies = new ArrayList<DFSClient>(this.dfsclients);
        }
        Collections.sort(copies, new Comparator<DFSClient>(){

            @Override
            public int compare(DFSClient left, DFSClient right) {
                return left.getClientName().compareTo(right.getClientName());
            }
        });
        String previousName = "";
        for (DFSClient c : copies) {
            if (c.getClientName().equals(previousName)) continue;
            if (!c.renewLease()) {
                LOG.debug("Did not renew lease for client {}", (Object)c);
                continue;
            }
            previousName = c.getClientName();
            LOG.debug("Lease renewed for client {}", (Object)previousName);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void run(int id) throws InterruptedException {
        long lastRenewed = Time.monotonicNow();
        while (!Thread.interrupted()) {
            long elapsed = Time.monotonicNow() - lastRenewed;
            if (elapsed >= this.getRenewalTime()) {
                try {
                    this.renew();
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Lease renewer daemon for " + this.clientsString() + " with renew id " + id + " executed");
                    }
                    lastRenewed = Time.monotonicNow();
                }
                catch (SocketTimeoutException ie) {
                    LOG.warn("Failed to renew lease for " + this.clientsString() + " for " + elapsed / 1000L + " seconds.  Aborting ...", (Throwable)ie);
                    LeaseRenewer leaseRenewer = this;
                    synchronized (leaseRenewer) {
                        while (!this.dfsclients.isEmpty()) {
                            DFSClient dfsClient = this.dfsclients.get(0);
                            dfsClient.closeAllFilesBeingWritten(true);
                            this.closeClient(dfsClient);
                        }
                        this.emptyTime = 0L;
                        break;
                    }
                }
                catch (IOException ie) {
                    LOG.warn("Failed to renew lease for " + this.clientsString() + " for " + elapsed / 1000L + " seconds.  Will retry shortly ...", (Throwable)ie);
                }
            }
            LeaseRenewer leaseRenewer = this;
            synchronized (leaseRenewer) {
                if (id != this.currentId || this.isRenewerExpired()) {
                    if (LOG.isDebugEnabled()) {
                        if (id != this.currentId) {
                            LOG.debug("Lease renewer daemon for " + this.clientsString() + " with renew id " + id + " is not current");
                        } else {
                            LOG.debug("Lease renewer daemon for " + this.clientsString() + " with renew id " + id + " expired");
                        }
                    }
                    return;
                }
                if (!this.clientsRunning() && this.emptyTime == Long.MAX_VALUE) {
                    this.emptyTime = Time.monotonicNow();
                }
            }
            Thread.sleep(this.getSleepPeriod());
        }
    }

    public String toString() {
        String s = this.getClass().getSimpleName() + ":" + this.factorykey;
        if (LOG.isTraceEnabled()) {
            return s + ", clients=" + this.clientsString() + ", created at " + this.instantiationTrace;
        }
        return s;
    }

    private synchronized String clientsString() {
        if (this.dfsclients.isEmpty()) {
            return "[]";
        }
        StringBuilder b = new StringBuilder("[").append(this.dfsclients.get(0).getClientName());
        for (int i = 1; i < this.dfsclients.size(); ++i) {
            b.append(", ").append(this.dfsclients.get(i).getClientName());
        }
        return b.append("]").toString();
    }

    private static class Factory {
        private static final Factory INSTANCE = new Factory();
        private final Map<Key, LeaseRenewer> renewers = new HashMap<Key, LeaseRenewer>();

        private Factory() {
        }

        private synchronized LeaseRenewer get(String authority, UserGroupInformation ugi) {
            Key k = new Key(authority, ugi);
            LeaseRenewer r = this.renewers.get(k);
            if (r == null) {
                r = new LeaseRenewer(k);
                this.renewers.put(k, r);
            }
            return r;
        }

        private synchronized void remove(LeaseRenewer r) {
            LeaseRenewer stored = this.renewers.get(r.factorykey);
            if (r == stored && !r.clientsRunning()) {
                this.renewers.remove(r.factorykey);
            }
        }

        private static class Key {
            final String authority;
            final UserGroupInformation ugi;

            private Key(String authority, UserGroupInformation ugi) {
                if (authority == null) {
                    throw new HadoopIllegalArgumentException("authority == null");
                }
                if (ugi == null) {
                    throw new HadoopIllegalArgumentException("ugi == null");
                }
                this.authority = authority;
                this.ugi = ugi;
            }

            public int hashCode() {
                return this.authority.hashCode() ^ this.ugi.hashCode();
            }

            public boolean equals(Object obj) {
                if (obj == this) {
                    return true;
                }
                if (obj != null && obj instanceof Key) {
                    Key that = (Key)obj;
                    return this.authority.equals(that.authority) && this.ugi.equals(that.ugi);
                }
                return false;
            }

            public String toString() {
                return this.ugi.getShortUserName() + "@" + this.authority;
            }
        }
    }
}

