/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.shortcircuit;

import com.google.common.base.Preconditions;
import com.google.common.collect.ComparisonChain;
import com.google.common.primitives.Ints;
import java.io.FileInputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.BitSet;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Random;
import javax.annotation.Nonnull;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.hadoop.fs.InvalidRequestException;
import org.apache.hadoop.hdfs.ExtendedBlockId;
import org.apache.hadoop.io.nativeio.NativeIO;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.misc.Unsafe;

public class ShortCircuitShm {
    private static final Logger LOG = LoggerFactory.getLogger(ShortCircuitShm.class);
    protected static final int BYTES_PER_SLOT = 64;
    private static final Unsafe unsafe = ShortCircuitShm.safetyDance();
    private final ShmId shmId;
    private final long baseAddress;
    private final int mmappedLength;
    private final Slot[] slots;
    private final BitSet allocatedSlots;

    private static Unsafe safetyDance() {
        try {
            Field f = Unsafe.class.getDeclaredField("theUnsafe");
            f.setAccessible(true);
            return (Unsafe)f.get(null);
        }
        catch (Throwable e) {
            LOG.error("failed to load misc.Unsafe", e);
            return null;
        }
    }

    private static int getUsableLength(FileInputStream stream) throws IOException {
        int intSize = Ints.checkedCast(stream.getChannel().size());
        int slots = intSize / 64;
        if (slots == 0) {
            throw new IOException("size of shared memory segment was " + intSize + ", but that is not enough to hold even one slot.");
        }
        return slots * 64;
    }

    public ShortCircuitShm(ShmId shmId, FileInputStream stream) throws IOException {
        if (!NativeIO.isAvailable()) {
            throw new UnsupportedOperationException("NativeIO is not available.");
        }
        if (Shell.WINDOWS) {
            throw new UnsupportedOperationException("DfsClientShm is not yet implemented for Windows.");
        }
        if (unsafe == null) {
            throw new UnsupportedOperationException("can't use DfsClientShm because we failed to load misc.Unsafe.");
        }
        this.shmId = shmId;
        this.mmappedLength = ShortCircuitShm.getUsableLength(stream);
        this.baseAddress = NativeIO.POSIX.mmap(stream.getFD(), 3, true, this.mmappedLength);
        this.slots = new Slot[this.mmappedLength / 64];
        this.allocatedSlots = new BitSet(this.slots.length);
        LOG.trace("creating {}(shmId={}, mmappedLength={}, baseAddress={}, slots.length={})", new Object[]{this.getClass().getSimpleName(), shmId, this.mmappedLength, String.format("%x", this.baseAddress), this.slots.length});
    }

    public final ShmId getShmId() {
        return this.shmId;
    }

    public final synchronized boolean isEmpty() {
        return this.allocatedSlots.nextSetBit(0) == -1;
    }

    public final synchronized boolean isFull() {
        return this.allocatedSlots.nextClearBit(0) >= this.slots.length;
    }

    private long calculateSlotAddress(int slotIdx) {
        long offset = slotIdx;
        return this.baseAddress + (offset *= 64L);
    }

    public final synchronized Slot allocAndRegisterSlot(ExtendedBlockId blockId) {
        int idx = this.allocatedSlots.nextClearBit(0);
        if (idx >= this.slots.length) {
            throw new RuntimeException(this + ": no more slots are available.");
        }
        this.allocatedSlots.set(idx, true);
        Slot slot = new Slot(this.calculateSlotAddress(idx), blockId);
        slot.clear();
        slot.makeValid();
        this.slots[idx] = slot;
        if (LOG.isTraceEnabled()) {
            LOG.trace(this + ": allocAndRegisterSlot " + idx + ": allocatedSlots=" + this.allocatedSlots + StringUtils.getStackTrace(Thread.currentThread()));
        }
        return slot;
    }

    public final synchronized Slot getSlot(int slotIdx) throws InvalidRequestException {
        if (!this.allocatedSlots.get(slotIdx)) {
            throw new InvalidRequestException(this + ": slot " + slotIdx + " does not exist.");
        }
        return this.slots[slotIdx];
    }

    public final synchronized Slot registerSlot(int slotIdx, ExtendedBlockId blockId) throws InvalidRequestException {
        if (slotIdx < 0) {
            throw new InvalidRequestException(this + ": invalid negative slot " + "index " + slotIdx);
        }
        if (slotIdx >= this.slots.length) {
            throw new InvalidRequestException(this + ": invalid slot " + "index " + slotIdx);
        }
        if (this.allocatedSlots.get(slotIdx)) {
            throw new InvalidRequestException(this + ": slot " + slotIdx + " is already in use.");
        }
        Slot slot = new Slot(this.calculateSlotAddress(slotIdx), blockId);
        if (!slot.isValid()) {
            throw new InvalidRequestException(this + ": slot " + slotIdx + " is not marked as valid.");
        }
        this.slots[slotIdx] = slot;
        this.allocatedSlots.set(slotIdx, true);
        if (LOG.isTraceEnabled()) {
            LOG.trace(this + ": registerSlot " + slotIdx + ": allocatedSlots=" + this.allocatedSlots + StringUtils.getStackTrace(Thread.currentThread()));
        }
        return slot;
    }

    public final synchronized void unregisterSlot(int slotIdx) {
        Preconditions.checkState(this.allocatedSlots.get(slotIdx), "tried to unregister slot " + slotIdx + ", which was not registered.");
        this.allocatedSlots.set(slotIdx, false);
        this.slots[slotIdx] = null;
        LOG.trace("{}: unregisterSlot {}", (Object)this, (Object)slotIdx);
    }

    public SlotIterator slotIterator() {
        return new SlotIterator();
    }

    public void free() {
        try {
            NativeIO.POSIX.munmap(this.baseAddress, this.mmappedLength);
        }
        catch (IOException e) {
            LOG.warn(this + ": failed to munmap", (Throwable)e);
        }
        LOG.trace(this + ": freed");
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(" + this.shmId + ")";
    }

    public class Slot {
        private static final long VALID_FLAG = Long.MIN_VALUE;
        private static final long ANCHORABLE_FLAG = 0x4000000000000000L;
        private final long slotAddress;
        private final ExtendedBlockId blockId;

        Slot(long slotAddress, ExtendedBlockId blockId) {
            this.slotAddress = slotAddress;
            this.blockId = blockId;
        }

        public ShortCircuitShm getShm() {
            return ShortCircuitShm.this;
        }

        public ExtendedBlockId getBlockId() {
            return this.blockId;
        }

        public SlotId getSlotId() {
            return new SlotId(ShortCircuitShm.this.getShmId(), this.getSlotIdx());
        }

        public int getSlotIdx() {
            return Ints.checkedCast((this.slotAddress - ShortCircuitShm.this.baseAddress) / 64L);
        }

        void clear() {
            unsafe.putLongVolatile(null, this.slotAddress, 0L);
        }

        private boolean isSet(long flag) {
            long prev = unsafe.getLongVolatile(null, this.slotAddress);
            return (prev & flag) != 0L;
        }

        private void setFlag(long flag) {
            long prev;
            do {
                if (((prev = unsafe.getLongVolatile(null, this.slotAddress)) & flag) == 0L) continue;
                return;
            } while (!unsafe.compareAndSwapLong(null, this.slotAddress, prev, prev | flag));
        }

        private void clearFlag(long flag) {
            long prev;
            do {
                if (((prev = unsafe.getLongVolatile(null, this.slotAddress)) & flag) != 0L) continue;
                return;
            } while (!unsafe.compareAndSwapLong(null, this.slotAddress, prev, prev & (flag ^ 0xFFFFFFFFFFFFFFFFL)));
        }

        public boolean isValid() {
            return this.isSet(Long.MIN_VALUE);
        }

        public void makeValid() {
            this.setFlag(Long.MIN_VALUE);
        }

        public void makeInvalid() {
            this.clearFlag(Long.MIN_VALUE);
        }

        public boolean isAnchorable() {
            return this.isSet(0x4000000000000000L);
        }

        public void makeAnchorable() {
            this.setFlag(0x4000000000000000L);
        }

        public void makeUnanchorable() {
            this.clearFlag(0x4000000000000000L);
        }

        public boolean isAnchored() {
            long prev = unsafe.getLongVolatile(null, this.slotAddress);
            return (prev & Long.MIN_VALUE) != 0L && (prev & Integer.MAX_VALUE) != 0L;
        }

        public boolean addAnchor() {
            long prev;
            do {
                if (((prev = unsafe.getLongVolatile(null, this.slotAddress)) & Long.MIN_VALUE) == 0L) {
                    return false;
                }
                if ((prev & 0x4000000000000000L) == 0L) {
                    return false;
                }
                if ((prev & Integer.MAX_VALUE) != Integer.MAX_VALUE) continue;
                return false;
            } while (!unsafe.compareAndSwapLong(null, this.slotAddress, prev, prev + 1L));
            return true;
        }

        public void removeAnchor() {
            long prev;
            do {
                Preconditions.checkState(((prev = unsafe.getLongVolatile(null, this.slotAddress)) & Integer.MAX_VALUE) != 0L, "Tried to remove anchor for slot " + this.slotAddress + ", which was " + "not anchored.");
            } while (!unsafe.compareAndSwapLong(null, this.slotAddress, prev, prev - 1L));
        }

        public String toString() {
            return "Slot(slotIdx=" + this.getSlotIdx() + ", shm=" + this.getShm() + ")";
        }
    }

    public class SlotIterator
    implements Iterator<Slot> {
        int slotIdx = -1;

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public boolean hasNext() {
            ShortCircuitShm shortCircuitShm = ShortCircuitShm.this;
            synchronized (shortCircuitShm) {
                return ShortCircuitShm.this.allocatedSlots.nextSetBit(this.slotIdx + 1) != -1;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Slot next() {
            ShortCircuitShm shortCircuitShm = ShortCircuitShm.this;
            synchronized (shortCircuitShm) {
                int nextSlotIdx = ShortCircuitShm.this.allocatedSlots.nextSetBit(this.slotIdx + 1);
                if (nextSlotIdx == -1) {
                    throw new NoSuchElementException();
                }
                this.slotIdx = nextSlotIdx;
                return ShortCircuitShm.this.slots[nextSlotIdx];
            }
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("SlotIterator doesn't support removal");
        }
    }

    public static class SlotId {
        private final ShmId shmId;
        private final int slotIdx;

        public SlotId(ShmId shmId, int slotIdx) {
            this.shmId = shmId;
            this.slotIdx = slotIdx;
        }

        public ShmId getShmId() {
            return this.shmId;
        }

        public int getSlotIdx() {
            return this.slotIdx;
        }

        public boolean equals(Object o) {
            if (o == null || o.getClass() != this.getClass()) {
                return false;
            }
            SlotId other = (SlotId)o;
            return new EqualsBuilder().append((Object)this.shmId, (Object)other.shmId).append(this.slotIdx, other.slotIdx).isEquals();
        }

        public int hashCode() {
            return new HashCodeBuilder().append((Object)this.shmId).append(this.slotIdx).toHashCode();
        }

        public String toString() {
            return String.format("SlotId(%s:%d)", this.shmId.toString(), this.slotIdx);
        }
    }

    public static class ShmId
    implements Comparable<ShmId> {
        private static final Random random = new Random();
        private final long hi;
        private final long lo;

        public static ShmId createRandom() {
            return new ShmId(random.nextLong(), random.nextLong());
        }

        public ShmId(long hi, long lo) {
            this.hi = hi;
            this.lo = lo;
        }

        public long getHi() {
            return this.hi;
        }

        public long getLo() {
            return this.lo;
        }

        public boolean equals(Object o) {
            if (o == null || o.getClass() != this.getClass()) {
                return false;
            }
            ShmId other = (ShmId)o;
            return new EqualsBuilder().append(this.hi, other.hi).append(this.lo, other.lo).isEquals();
        }

        public int hashCode() {
            return new HashCodeBuilder().append(this.hi).append(this.lo).toHashCode();
        }

        public String toString() {
            return String.format("%016x%016x", this.hi, this.lo);
        }

        @Override
        public int compareTo(@Nonnull ShmId other) {
            return ComparisonChain.start().compare(this.hi, other.hi).compare(this.lo, other.lo).result();
        }
    }
}

