/*
 * Decompiled with CFR 0.152.
 */
package com.github.akurilov.netty.connection.pool;

import com.github.akurilov.netty.connection.pool.NonBlockingConnPool;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.pool.ChannelPoolHandler;
import java.io.IOException;
import java.net.ConnectException;
import java.net.InetSocketAddress;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Logger;

public class MultiNodeConnPoolImpl
implements NonBlockingConnPool {
    private static final Logger LOG = Logger.getLogger(MultiNodeConnPoolImpl.class.getName());
    private final String[] nodes;
    private final int n;
    private final int connAttemptsLimit;
    private final long connectTimeOut;
    private final TimeUnit connectTimeUnit;
    private final Map<String, Bootstrap> bootstraps;
    private final Map<String, Queue<Channel>> allConns;
    private final Map<String, Queue<Channel>> availableConns;
    private final Map<String, AtomicInteger> connCounts;
    private final Map<String, AtomicInteger> failedConnAttemptCounts;
    private final Lock closeLock = new ReentrantLock();

    public MultiNodeConnPoolImpl(String[] nodes, Bootstrap bootstrap, final ChannelPoolHandler connPoolHandler, int defaultPort, int connAttemptsLimit, long connectTimeOut, TimeUnit connectTimeUnit) {
        if (nodes.length == 0) {
            throw new IllegalArgumentException("Empty nodes array argument");
        }
        this.nodes = nodes;
        this.connAttemptsLimit = connAttemptsLimit;
        this.connectTimeOut = connectTimeOut;
        this.connectTimeUnit = connectTimeUnit;
        this.n = nodes.length;
        this.bootstraps = new HashMap<String, Bootstrap>(this.n);
        this.allConns = new ConcurrentHashMap<String, Queue<Channel>>(this.n);
        this.availableConns = new ConcurrentHashMap<String, Queue<Channel>>(this.n);
        this.connCounts = new ConcurrentHashMap<String, AtomicInteger>(this.n);
        this.failedConnAttemptCounts = new ConcurrentHashMap<String, AtomicInteger>(this.n);
        for (String node : nodes) {
            InetSocketAddress nodeAddr;
            if (node.contains(":")) {
                String[] addrParts = node.split(":");
                nodeAddr = new InetSocketAddress(addrParts[0], Integer.parseInt(addrParts[1]));
            } else {
                nodeAddr = new InetSocketAddress(node, defaultPort);
            }
            this.bootstraps.put(node, (Bootstrap)bootstrap.clone().remoteAddress(nodeAddr).handler(new ChannelInitializer<Channel>(){

                @Override
                protected final void initChannel(Channel conn) throws Exception {
                    if (!conn.eventLoop().inEventLoop()) {
                        throw new AssertionError();
                    }
                    connPoolHandler.channelCreated(conn);
                }
            }));
            this.availableConns.put(node, new ConcurrentLinkedQueue());
            this.connCounts.put(node, new AtomicInteger(0));
            this.failedConnAttemptCounts.put(node, new AtomicInteger(0));
        }
    }

    @Override
    public void preConnect(int count) throws ConnectException, IllegalArgumentException, InterruptedException {
        if (count > 0) {
            CountDownLatch latch = new CountDownLatch(count);
            for (int i = 0; i < count; ++i) {
                String node = this.nodes[i % this.nodes.length];
                this.bootstraps.get(node).connect().addListener(future -> {
                    try {
                        Channel conn = future.channel();
                        conn.closeFuture().addListener(new CloseChannelListener(node, conn));
                        conn.attr(ATTR_KEY_NODE).set(node);
                        this.allConns.computeIfAbsent(node, na -> new ConcurrentLinkedQueue()).add(conn);
                        Map<String, AtomicInteger> map = this.connCounts;
                        synchronized (map) {
                            this.connCounts.get(node).incrementAndGet();
                        }
                        if (this.connAttemptsLimit > 0) {
                            this.failedConnAttemptCounts.get(node).set(0);
                        }
                        LOG.fine("New connection to " + node + " created");
                        if (conn.isActive()) {
                            Queue<Channel> connQueue = this.availableConns.get(node);
                            if (connQueue != null) {
                                connQueue.add(conn);
                            }
                        } else {
                            conn.close();
                        }
                    }
                    finally {
                        latch.countDown();
                    }
                });
            }
            if (latch.await(this.connectTimeOut, this.connectTimeUnit)) {
                LOG.info("Pre-created " + count + " connections");
            } else {
                LOG.warning("Pre-created " + ((long)count - latch.getCount()) + " connections");
            }
        } else {
            throw new IllegalArgumentException("Connection count should be > 0, but got " + count);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Channel connectToAnyNode() throws ConnectException {
        int i;
        Channel conn = null;
        String nodeAddr = null;
        int minConnsCount = Integer.MAX_VALUE;
        int nextConnsCount = 0;
        for (int j = i = ThreadLocalRandom.current().nextInt(this.n); j < this.n; ++j) {
            String nextNodeAddr = this.nodes[j % this.n];
            nextConnsCount = this.connCounts.get(nextNodeAddr).intValue();
            if (nextConnsCount == 0) {
                nodeAddr = nextNodeAddr;
                break;
            }
            if (nextConnsCount >= minConnsCount) continue;
            minConnsCount = nextConnsCount;
            nodeAddr = nextNodeAddr;
        }
        if (nodeAddr != null) {
            LOG.fine("New connection to \"" + nodeAddr + "\"");
            try {
                conn = this.connect(nodeAddr);
            }
            catch (Exception e) {
                int connAttempts;
                LOG.warning("Failed to create a new connection to " + nodeAddr + ": " + e.toString());
                if (this.connAttemptsLimit > 0 && (connAttempts = this.failedConnAttemptCounts.get(nodeAddr).incrementAndGet()) > this.connAttemptsLimit) {
                    LOG.warning("Failed to connect to the node \"" + nodeAddr + "\" " + connAttempts + " times successively, excluding the node from the connection pool forever");
                    this.connCounts.get(nodeAddr).set(Integer.MAX_VALUE);
                    boolean allNodesExcluded = true;
                    for (String node : this.nodes) {
                        if (this.connCounts.get(node).get() >= Integer.MAX_VALUE) continue;
                        allNodesExcluded = false;
                        break;
                    }
                    if (allNodesExcluded) {
                        LOG.severe("No endpoint nodes left in the connection pool!");
                    }
                }
                if (e instanceof ConnectException) {
                    throw (ConnectException)e;
                }
                throw new ConnectException(e.getMessage());
            }
        }
        if (conn != null) {
            conn.closeFuture().addListener(new CloseChannelListener(nodeAddr, conn));
            conn.attr(ATTR_KEY_NODE).set(nodeAddr);
            this.allConns.computeIfAbsent(nodeAddr, na -> new ConcurrentLinkedQueue()).add(conn);
            Map<String, AtomicInteger> map = this.connCounts;
            synchronized (map) {
                this.connCounts.get(nodeAddr).incrementAndGet();
            }
            if (this.connAttemptsLimit > 0) {
                this.failedConnAttemptCounts.get(nodeAddr).set(0);
            }
            LOG.fine("New connection to " + nodeAddr + " created");
        }
        return conn;
    }

    protected Channel connect(String addr) throws Exception {
        Channel conn = null;
        Bootstrap bootstrap = this.bootstraps.get(addr);
        if (bootstrap != null) {
            ChannelFuture connFuture = bootstrap.connect();
            if (this.connectTimeOut > 0L) {
                if (connFuture.await(this.connectTimeOut, this.connectTimeUnit)) {
                    conn = connFuture.channel();
                }
            } else {
                conn = connFuture.sync().channel();
            }
        }
        return conn;
    }

    protected Channel poll() {
        Queue<Channel> connQueue;
        int i = ThreadLocalRandom.current().nextInt(this.n);
        Channel conn = null;
        for (int j = i; !(j >= i + this.n || (connQueue = this.availableConns.get(this.nodes[j % this.n])) != null && (conn = connQueue.poll()) != null && conn.isOpen()); ++j) {
        }
        return conn;
    }

    @Override
    public final Channel lease() throws ConnectException {
        Channel conn = this.poll();
        if (null == conn) {
            conn = this.connectToAnyNode();
        }
        if (conn == null) {
            throw new ConnectException();
        }
        return conn;
    }

    @Override
    public final int lease(List<Channel> conns, int count) throws ConnectException {
        for (int i = 0; i < count; ++i) {
            Channel conn = this.poll();
            if (null == conn) {
                conn = this.connectToAnyNode();
            }
            if (conn == null) {
                throw new ConnectException();
            }
            conns.add(conn);
        }
        return count;
    }

    @Override
    public final void release(Channel conn) {
        String nodeAddr = (String)conn.attr(ATTR_KEY_NODE).get();
        if (conn.isActive()) {
            Queue<Channel> connQueue = this.availableConns.get(nodeAddr);
            if (connQueue != null) {
                connQueue.add(conn);
            }
        } else {
            conn.close();
        }
    }

    @Override
    public final void release(List<Channel> conns) {
        for (Channel conn : conns) {
            String nodeAddr = (String)conn.attr(ATTR_KEY_NODE).get();
            if (conn.isActive()) {
                Queue<Channel> connQueue = this.availableConns.get(nodeAddr);
                connQueue.add(conn);
                continue;
            }
            conn.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        this.closeLock.lock();
        int closedConnCount = 0;
        try {
            for (String nodeAddr : this.availableConns.keySet()) {
                for (Channel conn : this.availableConns.get(nodeAddr)) {
                    if (!conn.isOpen()) continue;
                    conn.close();
                    ++closedConnCount;
                }
            }
            this.availableConns.clear();
            for (String nodeAddr : this.allConns.keySet()) {
                for (Channel conn : this.allConns.get(nodeAddr)) {
                    if (!conn.isOpen()) continue;
                    conn.close();
                    ++closedConnCount;
                }
            }
            this.allConns.clear();
            this.bootstraps.clear();
            this.connCounts.clear();
        }
        finally {
            this.closeLock.unlock();
        }
        LOG.fine("Closed " + closedConnCount + " connections");
    }

    private final class CloseChannelListener
    implements ChannelFutureListener {
        private final String nodeAddr;
        private final Channel conn;

        private CloseChannelListener(String nodeAddr, Channel conn) {
            this.nodeAddr = nodeAddr;
            this.conn = conn;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public final void operationComplete(ChannelFuture future) throws Exception {
            LOG.fine("Connection to " + this.nodeAddr + " closed");
            MultiNodeConnPoolImpl.this.closeLock.lock();
            try {
                Map<String, Object> map = MultiNodeConnPoolImpl.this.connCounts;
                synchronized (map) {
                    if (MultiNodeConnPoolImpl.this.connCounts.containsKey(this.nodeAddr)) {
                        MultiNodeConnPoolImpl.this.connCounts.get(this.nodeAddr).decrementAndGet();
                    }
                }
                map = MultiNodeConnPoolImpl.this.allConns;
                synchronized (map) {
                    Queue<Channel> nodeConns = MultiNodeConnPoolImpl.this.allConns.get(this.nodeAddr);
                    if (nodeConns != null) {
                        nodeConns.remove(this.conn);
                    }
                }
            }
            finally {
                MultiNodeConnPoolImpl.this.closeLock.unlock();
            }
        }
    }
}

