/*
 * Copyright (c) 2014, Francis Galiegue (fgaliegue@gmail.com)
 *
 * This software is dual-licensed under:
 *
 * - the Lesser General Public License (LGPL) version 3.0 or, at your option, any
 *   later version;
 * - the Apache Software License (ASL) version 2.0.
 *
 * The text of both licenses is available under the src/resources/ directory of
 * this project (under the names LGPL-3.0.txt and ASL-2.0.txt respectively).
 *
 * Direct link to the sources:
 *
 * - LGPL 3.0: https://www.gnu.org/licenses/lgpl-3.0.txt
 * - ASL 2.0: http://www.apache.org/licenses/LICENSE-2.0.txt
 */

package com.github.fge.jsonschema.format.common;

import com.github.fge.jackson.NodeType;
import com.github.fge.jsonschema.core.exceptions.ProcessingException;
import com.github.fge.jsonschema.core.report.ProcessingReport;
import com.github.fge.jsonschema.core.util.RhinoHelper;
import com.github.fge.jsonschema.format.AbstractFormatAttribute;
import com.github.fge.jsonschema.format.FormatAttribute;
import com.github.fge.jsonschema.processors.data.FullData;
import com.github.fge.msgsimple.bundle.MessageBundle;

/**
 * Validator for the {@code regex} format attribute.
 *
 * <p>Again, here, we do <b>not</b> use {@link java.util.regex} because it does
 * not fit the bill.</p>
 *
 * @see RhinoHelper
 */
public final class RegexAttribute
    extends AbstractFormatAttribute
{
    private static final FormatAttribute INSTANCE = new RegexAttribute();

    public static FormatAttribute getInstance()
    {
        return INSTANCE;
    }

    private RegexAttribute()
    {
        super("regex", NodeType.STRING);
    }

    @Override
    public void validate(final ProcessingReport report,
        final MessageBundle bundle, final FullData data)
        throws ProcessingException
    {
        final String value = data.getInstance().getNode().textValue();

        if (!RhinoHelper.regexIsValid(value))
            report.error(newMsg(data, bundle, "err.format.invalidRegex")
                .putArgument("value", value));
    }
}
