package fun.gen;

import fun.tuple.Pair;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Supplier;
import java.util.random.RandomGenerator;

import static java.util.Objects.requireNonNull;

/**
 * Represents a generator of byte arrays.
 * This class implements the {@link Gen} interface to generate byte arrays with specified lengths.
 * It provides methods for generating byte arrays with arbitrary lengths and biasing towards specific lengths.
 *
 * @see Gen
 * @see Combinators
 */
public final class BytesGen implements Gen<byte[]> {

    private final int length;

    private BytesGen(int length) {
        this.length = length;
    }

    /**
     * Creates an arbitrary byte array generator with lengths between {@code minLength} and {@code maxLength} (inclusive).
     *
     * @param minLength The minimum length of generated byte arrays.
     * @param maxLength The maximum length of generated byte arrays.
     * @return An arbitrary generator for byte arrays with variable lengths.
     * @throws IllegalArgumentException If {@code maxLength} is less than {@code minLength}.
     */

    public static Gen<byte[]> arbitrary(int minLength,
                                        int maxLength) {
        if (maxLength < minLength) throw new IllegalArgumentException("max < min");
        return seed -> genBytes(seed,
                                IntGen.arbitrary(minLength,
                                                 maxLength)
                                      .apply(seed));
    }


    /**
     * Creates a biased byte array generator with lengths between {@code minLength} and {@code maxLength} (inclusive).
     * The generator is biased towards generating byte arrays with lengths equal to {@code minLength} and {@code maxLength}.
     * <p>
     * Example usage:
     * <pre>
     * // Create a biased byte array generator biased towards lengths 5 and 10.
     * Gen&lt;byte[]&gt; biasedGenerator = BytesGen.biased(5, 10);
     * byte[] biasedBytes = biasedGenerator.sample(new Random()).get();
     * </pre>
     *
     * @param minLength The minimum length of generated byte arrays.
     * @param maxLength The maximum length of generated byte arrays.
     * @return A biased generator for byte arrays with specific lengths.
     * @throws IllegalArgumentException If {@code minLength} is less than zero or if {@code maxLength} is less than {@code minLength}.
     */
    public static Gen<byte[]> biased(int minLength,
                                     int maxLength) {
        if (minLength < 0) throw new IllegalArgumentException("min < 0");
        if (maxLength < minLength) throw new IllegalArgumentException("max < min");

        List<Pair<Integer, Gen<? extends byte[]>>> gens = new ArrayList<>();
        gens.add(Pair.of(1,
                         new BytesGen(minLength)));
        if (maxLength != minLength)
            gens.add(Pair.of(1,
                             new BytesGen(maxLength)));

        gens.add(Pair.of(gens.size(),
                         arbitrary(minLength,
                                   maxLength)));

        return Combinators.freqList(gens);
    }

    private static Supplier<byte[]> genBytes(RandomGenerator gen,
                                             Supplier<Integer> size) {
        return () -> {
            byte[] bytes = new byte[size.get()];
            gen.nextBytes(bytes);
            return bytes;
        };
    }

    @Override
    public Supplier<byte[]> apply(final RandomGenerator gen) {
        requireNonNull(gen);
        return genBytes(gen,
                        () -> length);
    }

}
