package com.github.krr.schema.generator.protobuf.impl;

import com.github.jknack.handlebars.Context;
import com.github.jknack.handlebars.Handlebars;
import com.github.jknack.handlebars.Template;
import com.github.jknack.handlebars.context.JavaBeanValueResolver;
import com.github.jknack.handlebars.context.MapValueResolver;
import com.github.jknack.handlebars.context.MethodValueResolver;
import com.github.jknack.handlebars.helper.ConditionalHelpers;
import com.github.jknack.handlebars.helper.StringHelpers;
import com.github.jknack.handlebars.io.ClassPathTemplateLoader;
import com.github.krr.schema.generator.api.ProtobufSchemaGeneratorContext;
import com.github.krr.schema.generator.api.SchemaGenerator;
import com.github.krr.schema.generator.protobuf.mappercodegen.AbstractMapperCodegenModel;
import com.github.krr.schema.generator.protobuf.mappercodegen.MapperCodegenModelFactory;
import com.github.krr.schema.generator.protobuf.mappercodegen.MapperConfig;
import com.github.krr.schema.generator.protobuf.model.MessageNodeBuilder;
import com.github.krr.schema.generator.protobuf.model.nodes.TypeNode;
import com.github.krr.schema.generator.protobuf.utils.MustacheHelpers;
import com.google.common.annotations.VisibleForTesting;
import lombok.NonNull;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import java.io.*;
import java.nio.file.Paths;
import java.util.*;

import static com.github.krr.schema.generator.protobuf.impl.ProtobufSchemaGenerator.ProtoSyntax.PROTO2;

@SuppressWarnings({"unused"})
@Slf4j
public class ProtobufSchemaGenerator implements SchemaGenerator {

  public static final Set<String> PACKAGES_TO_SKIP = new HashSet<>();

  private static final String ABSTRACT_CLASS_ATTRIBUTE_WRAPPER_SUFFIX = "Attributes";

  public static final String PRIMITIVE_TYPE = "_PrimitiveType";

  private final ProtoSyntax syntax;

  public static final String REPEATED = "repeated ";

  static {
    PACKAGES_TO_SKIP.add("org.slf4j");
  }

  private MapperConfig mapperConfig;

  @VisibleForTesting
  MessageNodeBuilder builder;

  public ProtobufSchemaGenerator(ProtoSyntax syntax) {
    this.syntax = syntax;
  }

  public ProtobufSchemaGenerator() {
    this(ProtoSyntax.PROTO3);
  }

  @SneakyThrows
  public void generateSchema(@NonNull ProtobufSchemaGeneratorContext protoContext) throws IOException {
    log.trace("Generating protobuf definitions");
    builder = new MessageNodeBuilder();
    List<Class<?>> classes = protoContext.getClasses();
    for (Class<?> clazz : classes) {
      builder.build(new TypeNode(clazz.getName(), clazz), syntax);
    }
    Map<String, Object> renderContext = getRenderContext(protoContext, builder, syntax);
    writeModelFile(protoContext, renderContext);
    if (protoContext.generateMappers()) {
      generateMappers(protoContext, builder, renderContext);
    }
  }

  public Map<String, Object> getRenderContext(ProtobufSchemaGeneratorContext protoContext, MessageNodeBuilder builder, ProtoSyntax syntax) {
    Map<String, Object> renderContext = new HashMap<>();
    renderContext.put("models", builder.getMessageNodes().values());
    renderContext.put("packageName", protoContext.packageName());
    renderContext.put("syntax", syntax.name().toLowerCase());
    renderContext.put("isProto2", syntax == PROTO2);
    return renderContext;
  }

  @SneakyThrows
  private void generateMappers(ProtobufSchemaGeneratorContext protoContext,
                               MessageNodeBuilder builder,
                               Map<String, Object> renderContext) {
    if (protoContext.generateMappers()) {
      mapperConfig = getMapperConfig(protoContext);
      MapperCodegenModelFactory factory = new MapperCodegenModelFactory(mapperConfig, builder);
      List<AbstractMapperCodegenModel> models = new ArrayList<>();
      renderContext.put("mapperModels", factory.getMapperCodegenModels());
      renderContext.put("mapperConfig", mapperConfig);
      final String mapperFilePath = protoContext.getMappersOutputDir().getAbsolutePath();
      log.debug("Mapper files will be located at {}", mapperFilePath);
      File javaOutputDir = Paths.get(mapperFilePath).toFile();
      writeMapperFiles(protoContext, javaOutputDir, renderContext);
    }
  }

  private MapperConfig getMapperConfig(ProtobufSchemaGeneratorContext protoContext) {
    mapperConfig = new MapperConfig();
    mapperConfig.setClassname(protoContext.mapperClassname());
    mapperConfig.setPackageName(protoContext.mapperPackageName());
    mapperConfig.setProtoPackageName(protoContext.protoPackageName());
    mapperConfig.setProtoOuterClassname(protoContext.protoOuterClass());
    return mapperConfig;
  }

  private void writeMapperFiles(ProtobufSchemaGeneratorContext protoContext, File outputDir, Map<String, Object> renderContext) throws IOException {
    String packageName = protoContext.mapperPackageName();
    File javaMapperFileLocation = getJavaFileLocation(outputDir, packageName);
    log.debug(">>> Writing java files to {}", javaMapperFileLocation.getAbsolutePath());
    createDirs(javaMapperFileLocation);

    if (mapperConfig.getClassname() == null) {
      throw new IllegalArgumentException("To generate mappers, mapperClassname() must be specified");
    }
    File javaFile = new File(javaMapperFileLocation, protoContext.mapperClassname().concat(".java"));
    writeFile(renderContext, javaFile, "/mappers2/mapperCodegen");
  }

  private File getJavaFileLocation(File outputDir, String packageName) {
    return new File(outputDir, packageName.replaceAll("\\.", File.separator));
  }

  private void createDirs(File javaOutputDir) {
    String outputDir = javaOutputDir.getAbsolutePath();
    log.error("Creating output dirs at " + outputDir);
    if(!new File(outputDir).exists()) {
      boolean dirs = javaOutputDir.mkdirs();
      if (!dirs) {
        throw new IllegalArgumentException("Could not create directory for class " + javaOutputDir);
      }
    }
  }

  private void writeModelFile(ProtobufSchemaGeneratorContext context,
                              Map<String, Object> renderContext) throws IOException {
    File protoFile = new File(context.getProtosOutputDir(), context.outputFilename());
    writeFile(renderContext, protoFile, "/models2/model");
  }

  private void writeFile(Map<String, Object> renderContext, File outputFile, String templateFile) throws IOException {
    try (Writer os = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(outputFile)))) {
      ClassPathTemplateLoader templateLoader = new ClassPathTemplateLoader("/templates", ".mustache");
      Handlebars handlebars = new Handlebars(templateLoader);
      handlebars.registerHelpers(MustacheHelpers.class);
      handlebars.registerHelpers(ConditionalHelpers.class);
      handlebars.registerHelpers(StringHelpers.class);
      Template template = handlebars.compile(templateFile);
      log.debug("Writing proto file {}", outputFile.getAbsolutePath());
      Context context = Context.newBuilder(renderContext)
                               .resolver(MapValueResolver.INSTANCE,
                                         JavaBeanValueResolver.INSTANCE,
                                         MethodValueResolver.INSTANCE
                                        ).build();
      os.write(template.apply(context));
    }
  }


  public enum ProtoSyntax {
    PROTO2,
    PROTO3
  }

}
