package com.github.krr.schema.generator.protobuf.mappercodegen;

import com.github.krr.schema.generator.protobuf.model.MessageNodeBuilder;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.AbstractAttribute;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.EnumAttribute;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.JavaAttribute;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.ProtoPrimitiveAttribute;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.*;
import com.github.krr.schema.generator.protobuf.models.TypeInfo;
import org.springframework.util.Assert;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

@SuppressWarnings({"FieldCanBeLocal", "unused"})
public class MapperCodegenModelFactory {

  private final MapperConfig mapperConfig;

  private final MessageNodeBuilder builder;

  private final Map<String, AbstractMapperCodegenModel> mapperCodegenModelMap;

  public MapperCodegenModelFactory(MapperConfig mapperConfig, MessageNodeBuilder builder) {
    this.mapperConfig = mapperConfig;
    this.builder = builder;
    mapperCodegenModelMap = builder.getMessageNodes().values().stream()
                                   .map(m -> createMapperCodegenModel(mapperConfig, m, this))
                                   .filter(Objects::nonNull)
                                   .collect(Collectors.toMap(m -> m.getMessageNode().getName(), m -> m));
  }

  private static AbstractMapperCodegenModel createMapperCodegenModel(MapperConfig mapperConfig,
                                                                     AbstractMessageNode messageNode,
                                                                     MapperCodegenModelFactory factory) {
    if (messageNode instanceof ProtoPrimitiveMessageNode) {
      return null;
    }
    if (messageNode.isRenderModel()) {
      if (messageNode instanceof EnumMessageNode) {
        return new EnumMessageCodegenModel(mapperConfig, factory, (EnumMessageNode) messageNode);
      }
      if (messageNode instanceof JavaBeanMessageNode) {
        return new PojoMapperCodegenModel(mapperConfig, factory, (JavaBeanMessageNode) messageNode);
      }
      else if (messageNode instanceof ObjectMessageNode) {
        return new ObjectModelMapperCodegenModel(mapperConfig, factory, (ObjectMessageNode) messageNode);
      }
      else if (messageNode instanceof OneOfSyntheticMessageNode) {
        return new OneOfMapperCodegenModel(mapperConfig, factory, (OneOfSyntheticMessageNode) messageNode);
      }
      else if (messageNode instanceof GenericMessageNode) {
        return new GenericMessageMapperCodegenModel(mapperConfig, factory, messageNode);
      }
      else if (messageNode instanceof CollectionMessageNode || messageNode instanceof MapMessageNode
               || messageNode instanceof ProtoPrimitiveCollectionMessageNode || messageNode instanceof ProtoPrimitiveMapMessageNode) {
        return new NestedMessageCodegenModel(mapperConfig, factory, (AbstractSyntheticMessageNode) messageNode);
      }
      throw new UnsupportedOperationException("Unsupported message node " + messageNode.getClass());
    }
    return null;
  }

  public AbstractAttributeCodegenModel createNestedMessageAttributeCodegenModel(MapperConfig mapperConfig,
                                                                                AbstractAttribute attribute) {
    AbstractMessageNode typeMessageNode = attribute.getTypeMessageNode();
    String typeName;
    if (typeMessageNode.isRenderModel()) {
      typeName = typeMessageNode.getName();
    }
    else {
      typeName = typeMessageNode.getWrappedMessage().getName();
    }
    AbstractMapperCodegenModel attrTypeCodegenModel = mapperCodegenModelMap.get(typeName);
    Assert.notNull(attrTypeCodegenModel, "Could not find mapper codegen model for " + typeName);
    return new NestedAttributeCodegenModel(this, mapperConfig, attrTypeCodegenModel, attribute);
  }

  public AbstractCodegenModel createSyntheticMessageAttributeCodegenModel(MapperConfig mapperConfig,
                                                                          AbstractMapperCodegenModel owningModel,
                                                                          ProtoPrimitiveAttribute attribute) {
    return new SyntheticProtoPrimitiveAttributeCodegenModel(this, mapperConfig, owningModel, attribute);
  }

  public AbstractCodegenModel createSyntheticMessageAttributeCodegenModel(MapperConfig mapperConfig,
                                                                          AbstractMapperCodegenModel owningModel,
                                                                          AbstractMessageNode wrappedMessage) {
    if (owningModel instanceof OneOfMapperCodegenModel) {
      OneOfMapperCodegenModel oneOfMapperCodegenModel = (OneOfMapperCodegenModel) owningModel;
      AbstractMapperCodegenModel subclassCodegenModel = mapperCodegenModelMap.get(wrappedMessage.getName());
      return new OneOfAttributeCodegenModel(this, mapperConfig, owningModel, subclassCodegenModel);
    }
    throw new UnsupportedOperationException("Unsupported model type " + owningModel.getClass().getName());
  }

  public AbstractAttributeCodegenModel createMapperAttributeCodegenModel(MapperConfig mapperConfig,
                                                                         JavaAttribute attribute) {
    AbstractMessageNode attrTypeMessageNode = attribute.getTypeMessageNode();
    TypeInfo typeInfo = attribute.getTypeInfo();
    AbstractMessageNode wrappedMessage = attrTypeMessageNode.getWrappedMessage();
    if (attribute.isIterableAttr() && typeInfo.isSimpleCollection()) {
      if (attrTypeMessageNode instanceof ProtoPrimitiveCollectionMessageNode) {
        return new SimpleCollectionAttributeCodegenModel(this,
                                                         mapperConfig,
                                                         attrTypeMessageNode.isModelVisible() ? mapperCodegenModelMap.get(attrTypeMessageNode.getName()) : null,
                                                         attribute);
      }
      else {
        return new SimpleCollectionAttributeCodegenModel(this,
                                                         mapperConfig,
                                                         attrTypeMessageNode.isModelVisible() ? mapperCodegenModelMap.get(attrTypeMessageNode.getName())
                                                                                              : mapperCodegenModelMap.get(wrappedMessage.getName()),
                                                         attribute);
      }
    }
    else if (attribute.isIterableAttr() && typeInfo.isSimpleMap()) {
      if (attrTypeMessageNode instanceof ProtoPrimitiveMapMessageNode) {
        return new SimpleMapAttributeCodegenModel(this,
                                                  mapperConfig,
                                                  attrTypeMessageNode.isModelVisible() ? mapperCodegenModelMap.get(attrTypeMessageNode.getName()) : null,
                                                  attribute);
      }
      else {
        return new SimpleMapAttributeCodegenModel(this,
                                                  mapperConfig,
                                                  attrTypeMessageNode.isModelVisible() ? mapperCodegenModelMap.get(attrTypeMessageNode.getName())
                                                                                       : mapperCodegenModelMap.get(wrappedMessage.getName()),
                                                  attribute);
      }
    }
    else if (attribute.isIterableAttr() && (typeInfo.isCollection() || typeInfo.isMap())) {
      return createNestedMessageAttributeCodegenModel(mapperConfig, attribute);
    }
    else if (attribute instanceof ProtoPrimitiveAttribute) {
      return new ProtoPrimitiveAttributeCodegenModel(this, mapperConfig, attribute);
    }
    else if (attribute instanceof EnumAttribute) {
      AbstractMapperCodegenModel messageCodegenModel = get(attrTypeMessageNode.getKey());
      Assert.notNull(messageCodegenModel, "Mapper code gen model not found for " + messageCodegenModel);
      return new EnumAttributeCodegenModel( mapperConfig, this, messageCodegenModel, attribute);
    }
    else {
      AbstractMapperCodegenModel attributeTypeModel;
      if (attrTypeMessageNode.isRenderModel()) {
        attributeTypeModel = mapperCodegenModelMap.get(attrTypeMessageNode.getName());
      }
      else {
        attributeTypeModel = mapperCodegenModelMap.get(wrappedMessage.getName());
      }
      Assert.notNull(attributeTypeModel, "attribute type model is null for attribute:" + attribute.getName());
      return new JavaAttributeCodegenModel(this,
                                           mapperConfig,
                                           attributeTypeModel,
                                           attribute);
    }
  }

  public List<AbstractMapperCodegenModel> getMapperCodegenModels() {
    return List.copyOf(mapperCodegenModelMap.values());
  }

  public AbstractMapperCodegenModel get(String key) {
    return mapperCodegenModelMap.get(key);
  }
}
