package com.github.krr.schema.generator.protobuf.mappercodegen;

import com.github.krr.schema.generator.protobuf.model.nodes.attributes.AbstractAttribute;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.SyntheticAttribute;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.*;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

@SuppressWarnings("unused")
public class NestedAttributeCodegenModel extends AbstractAttributeCodegenModel {

  private final SyntheticAttribute syntheticAttribute;

  public NestedAttributeCodegenModel(MapperCodegenModelFactory factory,
                                     MapperConfig mapperConfig,
                                     AbstractMapperCodegenModel attrTypeCodegenModel,
                                     AbstractAttribute attribute) {
    super(mapperConfig, factory, attrTypeCodegenModel, attribute);
    if(attribute instanceof SyntheticAttribute) {
      this.syntheticAttribute = (SyntheticAttribute)attribute;
    }
    else {
      AbstractMessageNode typeMessageNode = attribute.getTypeMessageNode();
      if(typeMessageNode instanceof AbstractSyntheticMessageNode) {
        syntheticAttribute = typeMessageNode.getNestedAttribute();
      }
      else {
        throw new UnsupportedOperationException("Unsupported type for NestedAttribute:" + attribute);
      }
    }
  }

  @Override
  public String getTemplateName() {
    return "mappers2/nestedAttr";
  }

  @Override
  public String getToProtoMethodName() {
    // if this attribute is a proto primitive - protomethod name is null
    // else
    if(syntheticAttribute.getTypeMessageNode() instanceof ProtoPrimitiveCollectionMessageNode ||
       syntheticAttribute.getTypeMessageNode() instanceof ProtoPrimitiveMapMessageNode ) {
      AbstractMessageNode typeMessageNode = syntheticAttribute.getTypeMessageNode();
      if(typeMessageNode.isModelVisible()) {
        return factory.get(typeMessageNode.getKey()).getToProtoMethodName();
      }
      return null;
    }
    return attributeTypeCodegenModel.getToProtoMethodName();
  }

  @Override
  public String getFromProtoMethodName() {
    if(syntheticAttribute.getTypeMessageNode() instanceof ProtoPrimitiveCollectionMessageNode ||
       syntheticAttribute.getTypeMessageNode() instanceof ProtoPrimitiveMapMessageNode ) {
      AbstractMessageNode typeMessageNode = syntheticAttribute.getTypeMessageNode();
      if(typeMessageNode.isModelVisible()) {
        return factory.get(typeMessageNode.getKey()).getFromProtoMethodName();
      }
      return null;
    }
    return attributeTypeCodegenModel.getFromProtoMethodName();
  }

  @Override
  public String getProtoSetterMethod() {
    String name = getSanitizedName(attribute.getName());
    AbstractMessageNode wrappedMessage = syntheticAttribute.getTypeMessageNode();
    // get the type of the java attribute
    AbstractMessageNode attrNativeMessageType = attribute.getTypeMessageNode();
    String capitalizedName = StringUtils.capitalize(name);
    if(syntheticAttribute.isCollectionAttr()) {
      return String.format("add%s", capitalizedName);
    }
    if(syntheticAttribute.isMapAttr()) {
      return String.format("put%s", capitalizedName);
    }
    throw new UnsupportedOperationException("Unsupported attribute type:" + attribute);
  }

  @Override
  public String getProtoGetterMethod() {
    String name = getSanitizedName(attribute.getName());
    Assert.notNull(name, "Attribute name was null for " + attribute);
    if(syntheticAttribute.isCollectionAttr()) {
      return String.format("get%sList()", StringUtils.capitalize(name));
    }
    if(syntheticAttribute.isMapAttr()) {
      return String.format("get%sMap()", StringUtils.capitalize(name));
    }
    throw new UnsupportedOperationException("Unsupported attribute :" + attribute);
  }

  private String getSanitizedName(String name) {
    if(name.startsWith("_")) {
      name = name.substring(1);
    }

    return name;
  }

  @Override
  public String getBeanGetterMethod() {
    if(syntheticAttribute.isCollectionAttr()) {
      return String.format("get%s()", StringUtils.capitalize(attribute.getName()));
    }
    if(syntheticAttribute.isMapAttr()) {
      return String.format("get%s()", StringUtils.capitalize(attribute.getName()));
    }
    throw new UnsupportedOperationException("Unsupported attribute type:" + attribute);
  }

  @Override
  public String getBeanSetterMethod() {
    if(syntheticAttribute.isCollectionAttr()) {
      return String.format("set%s", StringUtils.capitalize(attribute.getName()));
    }
    if(syntheticAttribute.isMapAttr()) {
      return String.format("set%s", StringUtils.capitalize(attribute.getName()));
    }
    throw new UnsupportedOperationException("Unsupported attribute type:" + syntheticAttribute);
  }

  @Override
  public String getBeanItemJavaType() {
    AbstractMessageNode attrMessageType = syntheticAttribute.getTypeMessageNode();
    Assert.notNull(attrMessageType, "Expecting non null message node for attribute:" + syntheticAttribute);
    AbstractMessageNode containingMessage = syntheticAttribute.getContainingMessageNode();
    if(attrMessageType.isModelVisibleTo(containingMessage)) {
      // this attribute is a wrapper which is visible -> return the underlying
      // javaType as the value type.
      return attrMessageType.getBeanJavaType();
    }
    return attrMessageType.getBeanItemJavaType();
  }

  @Override
  public String getProtoItemJavaType() {
    AbstractMessageNode typeMessageNode = syntheticAttribute.getTypeMessageNode();
    Assert.notNull(typeMessageNode, "Expecting non null message node for attribute:" + syntheticAttribute);
    return String.format("%s.%s", mapperConfig.getProtoOuterClassname(), typeMessageNode.getProtoItemJavaType());
  }

  public String getWrappedItemType() {
    if(syntheticAttribute.isCollectionAttr()) {
      return "list";
    }
    if(syntheticAttribute.isMapAttr()) {
      return "map";
    }
    throw new UnsupportedOperationException("only map/list type is supported");
  }

  public String getBeanJavaType() {
    if(syntheticAttribute.isCollectionAttr() || syntheticAttribute.isMapAttr() ) {
      return syntheticAttribute.getBeanJavaType();
    }
    throw new UnsupportedOperationException("only map/list type is supported");
  }

  public String getProtoJavaType() {
    return String.format("%s.%s", mapperConfig.getProtoOuterClassname(), syntheticAttribute.getProtoJavaType(mapperConfig));
  }

  @Override
  public boolean isSuperclassAttribute() {
    return false;
  }

  @Override
  public boolean isAbstract() {
    return false;
  }

  @Override
  public boolean isProtoPrimitive() {
    return false;
  }
}
