package com.github.krr.schema.generator.protobuf.mappercodegen;

import com.github.krr.schema.generator.protobuf.model.nodes.messages.AbstractMessageNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.AbstractSyntheticMessageNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.ProtoPrimitiveCollectionMessageNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.ProtoPrimitiveMapMessageNode;
import com.github.krr.schema.generator.protobuf.models.TypeInfo;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.stream.Collectors;

@SuppressWarnings("unused")
public class NestedMessageCodegenModel extends AbstractMapperCodegenModel {

  private final AbstractSyntheticMessageNode syntheticMessageNode;
  private final TypeInfo javaTypeInfo;

  protected NestedMessageCodegenModel(MapperConfig mapperConfig,
                                      MapperCodegenModelFactory factory,
                                      AbstractSyntheticMessageNode messageNode) {
    super(mapperConfig, factory, messageNode);
    this.syntheticMessageNode = messageNode;
    this.javaTypeInfo = new TypeInfo();
    TypeInfo.updateTypeInfoForType(javaTypeInfo, messageNode.getJavaType());
  }

  @Override
  public String getTemplateName() {
    return "mappers2/nested";
  }

  @Override
  public String getBeanJavaType() {
    return syntheticMessageNode.getJavaType().getTypeName();
  }

  @Override
  public String getProtoJavaType() {
    return String.format("%s.%s", mapperConfig.getProtoOuterClassname(), messageNode.getProtoMessageName());
  }

  @Override
  public String getBeanItemJavaType() {
    // the item type here is the element that's held by this synthetic message
    // we can derive it from the typeInfo
    return syntheticMessageNode.getBeanItemJavaType();
  }

  @Override
  public String getProtoItemJavaType() {
    return String.format("%s.%s", mapperConfig.getProtoOuterClassname(), syntheticMessageNode.getProtoItemJavaType());
  }

  @Override
  public String getNewBeanInstance() {
    if (javaTypeInfo.isCollection()) {
      return "new java.util.ArrayList<>();";
    }
    else if (javaTypeInfo.isMap()) {
      return "new java.util.HashMap<>();";
    }
    else if (javaTypeInfo.isGenericClass()) {
      return String.format("new %s<>();", javaTypeInfo.getRawType().getTypeName());
    }
    throw new UnsupportedOperationException("Only support list/map/generic types");
  }

  @Override
  public String getNewProtoInstance() {
    return String.format("%s.%s.newBuilder();", mapperConfig.getProtoOuterClassname(), messageNode.getName());
  }

  @Override
  public String getToProtoMethodName() {
    if (syntheticMessageNode instanceof ProtoPrimitiveCollectionMessageNode ||
        syntheticMessageNode instanceof ProtoPrimitiveMapMessageNode) {
      if (!syntheticMessageNode.isModelVisible()) {
        return null;
      }
    }
    String key = AbstractSyntheticMessageNode.getKey(syntheticMessageNode.getJavaType());
    AbstractMessageNode wrappedMessage = syntheticMessageNode.getWrappedMessage();
    if (wrappedMessage instanceof AbstractSyntheticMessageNode && ((AbstractSyntheticMessageNode) wrappedMessage).getJavaType() == Object.class) {
      key = key.replaceAll("_PrimitiveTypes", "Object");
    }
    return String.format("%sToProto%s", StringUtils.uncapitalize(key),
                         StringUtils.capitalize(messageNode.getProtoMessageName()));
  }

  @Override
  public String getFromProtoMethodName() {
    if (syntheticMessageNode instanceof ProtoPrimitiveCollectionMessageNode ||
        syntheticMessageNode instanceof ProtoPrimitiveMapMessageNode) {
      if (!syntheticMessageNode.isModelVisible()) {
        return null;
      }
    }
    String key = AbstractSyntheticMessageNode.getKey(syntheticMessageNode.getJavaType());
    AbstractMessageNode wrappedMessage = syntheticMessageNode.getWrappedMessage();
    if (wrappedMessage instanceof AbstractSyntheticMessageNode && ((AbstractSyntheticMessageNode) wrappedMessage).getJavaType() == Object.class) {
      key = key.replaceAll("_PrimitiveTypes", "Object");
    }
    return String.format("proto%sTo%s", StringUtils.capitalize(messageNode.getProtoMessageName()),
                         StringUtils.capitalize(key));
  }

  public String getToProtoItemMethodName() {
    AbstractMessageNode wrappedMessage = syntheticMessageNode.getWrappedMessage();
    if (wrappedMessage == null) {
      return null;
    }
    AbstractMapperCodegenModel wrappedModel = factory.get(wrappedMessage.getKey());
    if(wrappedModel == null) {
      // primitive types with no conversion
      return null;
    }
    return wrappedModel.getToProtoMethodName();
  }


  public String getFromProtoItemMethodName() {
    AbstractMessageNode wrappedMessage = syntheticMessageNode.getWrappedMessage();
    if (wrappedMessage == null) {
      return null;
    }
    AbstractMapperCodegenModel wrappedModel = factory.get(wrappedMessage.getKey());
    if(wrappedModel == null) {
      // primitive types with no conversion
      return null;
    }
    return wrappedModel.getFromProtoMethodName();
  }

  @Override
  public List<AbstractCodegenModel> getAttributes() {
    return messageNode.getAttributes().stream()
                      .map(a -> factory.createNestedMessageAttributeCodegenModel(mapperConfig, a))
                      .collect(Collectors.toList());
  }
}
