package com.github.krr.schema.generator.protobuf.model;

import com.github.krr.schema.generator.annotations.SchemaItem;
import com.github.krr.schema.generator.protobuf.api.MessageModelNodeBuilder;
import com.github.krr.schema.generator.protobuf.impl.ProtobufSchemaGenerator.ProtoSyntax;
import com.github.krr.schema.generator.protobuf.model.builders.*;
import com.github.krr.schema.generator.protobuf.model.nodes.TypeNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.AbstractMessageNode;
import com.github.krr.schema.generator.protobuf.utils.MapperCodegenModelUtils;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@SuppressWarnings("rawtypes")
@Slf4j
public class MessageNodeBuilder {

  @Getter
  private final Map<String, AbstractMessageNode> messageNodes = new HashMap<>();

  @Getter
  private final Map<String, AbstractMessageNode> messagesInCreation = new HashMap<>();

  private final List<MessageModelNodeBuilder> MESSAGE_NODE_BUILDERS = List.of(new EnumMessageModelNodeBuilder(),
                                                                              new WildcardTypeImplMessageNodeBuilder(),
                                                                              new ProtoPrimitiveCollectionParameterizedMessageNodeBuilder(),
                                                                              new ProtoPrimitiveMapParameterizedMessageNodeBuilder(),
                                                                              new TypeVariableMessageNodeBuilder(),
                                                                              new GenericParameterizedTypeMessageNodeBuilder(),
                                                                              new MapSubclassMessageNodeBuilder(),
                                                                              new CollectionSubclassMessageNodeBuilder(),
                                                                              new CollectionParameterizedMessageNodeBuilder(),
                                                                              new MapParameterizedMessageNodeBuilder(),
                                                                              new ObjectMessageNodeBuilder(),
                                                                              new PrimitiveMessageModelNodeBuilder(),
                                                                              new GenericMessageModelNodeBuilder(),
                                                                              new PojoMessageModelNodeBuilder(),
                                                                              new PojoWithSubclassesMessageModelNodeBuilder()
                                                                             );

  @SuppressWarnings("unchecked")
  public AbstractMessageNode build(TypeNode typeNode, ProtoSyntax syntax) {
    String name = typeNode.getName();
    Type type = typeNode.getType();
    if (type instanceof Class) {
      Class clazz = (Class) type;
      Annotation schemaItemAnnotation = clazz.getAnnotation(SchemaItem.class);
      // Object type and proto primitive types are the only one we process without annotations.
      if (!isObjectOrProtoPrimitiveType(clazz) && (schemaItemAnnotation == null || clazz.getAnnotation(SchemaItem.Transient.class) != null)) {
        log.warn("No SchemaItem on class {} or class is marked as Transient.  Ignoring", clazz);
        // skip classes marked as transient
        return null;
      }
    }
    TypeNode nodeForClass = new TypeNode(type.getTypeName(), type);
    for (MessageModelNodeBuilder builder : MESSAGE_NODE_BUILDERS) {
      if (builder.supports(nodeForClass)) {
        String key = builder.getKey(type);
        AbstractMessageNode node = findNode(key);
        if (node != null) {
          return node;
        }
        log.debug("Processing {} using {}", name, builder.getClass().getName());
        node = builder.buildNode(this, nodeForClass, syntax);
        return node;
      }
    }
    throw new UnsupportedOperationException("No builder found for typeNode:" + typeNode);
  }

  private boolean isObjectOrProtoPrimitiveType(Class clazz) {
    return clazz == Object.class || MapperCodegenModelUtils.isProtoNativeType(clazz);
  }

  public String getProtoMessageName(Type type) {
    String name = type.getTypeName();
    return name.substring(name.lastIndexOf(".") + 1).replaceAll("\\$", "Dollar");
  }

  public AbstractMessageNode findNode(String name) {
    if (messageNodes.containsKey(name)) {
      return messageNodes.get(name);
    }
    else if (messagesInCreation.containsKey(name)) {
      return messagesInCreation.get(name);
    }
    return null;
  }

  public void registerMessage(String key, AbstractMessageNode messageNode) {
    if (!messageNodes.containsKey(key)) {
      messageNodes.put(key, messageNode);
    }
    log.debug("Message {} is already registered.  Ignoring", key);
  }

  public void start(String key, AbstractMessageNode node) {
    messagesInCreation.put(key, node);
  }

  public void finish(String key) {
    if (messagesInCreation.containsKey(key)) {
      AbstractMessageNode node = messagesInCreation.remove(key);
      messageNodes.put(key, node);
    }
  }


}
