package com.github.krr.schema.generator.protobuf.model.builders;

import com.github.krr.schema.generator.annotations.SchemaItem;
import com.github.krr.schema.generator.protobuf.api.MessageModelNodeBuilder;
import com.github.krr.schema.generator.protobuf.impl.FieldsFilterImpl;
import com.github.krr.schema.generator.protobuf.impl.ProtobufSchemaGenerator;
import com.github.krr.schema.generator.protobuf.model.MessageNodeBuilder;
import com.github.krr.schema.generator.protobuf.model.nodes.TypeNode;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.*;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.*;
import com.github.krr.schema.generator.protobuf.models.TypeInfo;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.StringUtils;

import java.lang.reflect.*;

@SuppressWarnings("rawtypes")
@Slf4j
public abstract class AbstractMessageModelNodeBuilder implements MessageModelNodeBuilder {

  @SneakyThrows
  protected long getIndexForEnum(Class<? extends java.lang.Enum> enumClass, java.lang.Enum enumConstant) {
    String enumName = enumConstant.name();
    return getIndex(enumClass.getField(enumName));
  }

  protected static long getIndex(AnnotatedElement annotatedElement) {
    SchemaItem schemaItem = AnnotationUtils.getAnnotation(annotatedElement, SchemaItem.class);
    if (schemaItem == null) {
      String name;
      if (annotatedElement instanceof Field) {
        name = ((Field) annotatedElement).getName();
      }
      else {
        name = ((Class) annotatedElement).getName();
      }
      String elName = annotatedElement.getClass() == Field.class ? ((Field) annotatedElement).getDeclaringClass().getName() : ((Class) annotatedElement).getName();
      throw new IllegalArgumentException(
          "No schema item found on object - schema item must be specified with model index" +
          " on element " + name + " of type " + elName + " or declared as Transient " +
          "with the SchemaItem.Transient annotation");
    }
    return schemaItem.modelIndex();
  }

  @SuppressWarnings("DuplicatedCode")
  protected Class getSuperclassToCompose(Class clazz) {
    SchemaItem schemaItem = AnnotationUtils.findAnnotation(clazz, SchemaItem.class);
    if (schemaItem != null) {
      final Class<?> superclassToCompose = schemaItem.superclassToCompose();
      if (superclassToCompose != SchemaItem.SkipSuperclass.class) {
        return superclassToCompose;
      }
    }
    return null;
  }

  protected boolean isProtoPrimitive(Type type) {
    // not primitive or primitive wrapper (e.g. int/Integer)
    return type instanceof Class &&
           ClassUtils.isPrimitiveOrWrapper((Class) type) ||
           // not string
           (type == String.class) ||
           // not byte []
           (type == byte[].class);
  }

  protected boolean isEnum(Type type) {
    // not primitive or primitive wrapper (e.g. int/Integer)
    return type instanceof Class && ((Class)type).isEnum();
  }

  public static String getProtoCompatibleName(String name) {
    return name.substring(name.lastIndexOf(".") + 1).replaceAll("\\$", "Dollar");
  }

  protected boolean isCollectionOrMapTypeField(Field field) {
    TypeInfo typeInfo = TypeInfo.getTypeInfoForField(field);
    return typeInfo.isCollection() || typeInfo.isMap();
  }

  @Override
  public String getKey(Type type) {
    if (type instanceof Class && type != Object.class) {
      return ((Class) type).getName();
    }
    else if (type == Object.class) {
      return ObjectMessageNodeBuilder.PROTO_MESSAGE_TYPE;
    }
    else if (type instanceof ParameterizedType || type instanceof TypeVariable || type instanceof WildcardType) {
      return AbstractSyntheticMessageNode.getKey(type);
    }
    throw new UnsupportedOperationException("Unsupported type:" + type);
  }

  protected void addSuperclassAttribute(MessageNodeBuilder messageNodeBuilder,
                                        ProtobufSchemaGenerator.ProtoSyntax syntax,
                                        Class clazz,
                                        AbstractMessageNode node) {
    Class superclass = getSuperclassToCompose(clazz);
    if (superclass != null) {
      String superclassKey = superclass.getName();
      AbstractMessageNode superclassNode = messageNodeBuilder.findNode(superclassKey);
      if (superclassNode == null) {
        superclassNode = messageNodeBuilder.build(new TypeNode(superclassKey, superclass), syntax);
        Assert.notNull(superclassNode, "Could not create super class message for " + superclassKey);
      }
      ((JavaBeanMessageNode) node).setSuperclassMessage(superclassNode);
      // add this superclass as an attribute in the subclass for composition
      String superClassProtoMessageName = getProtoCompatibleName(superclassKey);
      SuperclassAttribute superclassAttribute = new SuperclassAttribute(StringUtils.uncapitalize(superClassProtoMessageName), superClassProtoMessageName, superclass, getIndex(superclass));
      superclassAttribute.setContainingMessageNode(node);
      superclassAttribute.setTypeMessageNode(superclassNode);
      superclassAttribute.setSyntax(syntax);
      node.addAttribute(superclassAttribute);
      messageNodeBuilder.registerMessage(superclassKey, superclassNode);
    }
  }

  public void addPojoAttributes(MessageNodeBuilder messageNodeBuilder,
                                ProtobufSchemaGenerator.ProtoSyntax syntax,
                                Class clazz,
                                AbstractMessageNode finalNode) {
    ReflectionUtils.doWithFields(clazz, field -> addPojoAttribute(messageNodeBuilder, syntax, clazz, finalNode, field), new FieldsFilterImpl(clazz));
  }

  public void addPojoAttribute(MessageNodeBuilder builder,
                               ProtobufSchemaGenerator.ProtoSyntax syntax,
                               Class clazz,
                               AbstractMessageNode node,
                               Field field) {
    log.debug("Adding model for field {} in class {}", field.getName(), clazz.getName());
    Type genericType = field.getGenericType();
    AbstractAttribute attribute;
    AbstractMessageNode fieldNode = builder.build(new TypeNode(field.getName(), genericType), syntax);
    String fieldName = getProtoCompatibleName(field.getName());
    long fieldIndex = getIndex(field);
    if (isProtoPrimitive(genericType)) {
      attribute = new ProtoPrimitiveAttribute(fieldName, (Class) genericType, fieldIndex);
    }
    else if (genericType == Object.class) {
      attribute = new JavaAttribute(fieldName, fieldNode.getProtoMessageName(), genericType, fieldIndex);
    }
    else if(genericType instanceof ParameterizedType && fieldNode instanceof ProtoPrimitiveCollectionMessageNode) {
      TypeInfo typeInfo = TypeInfo.getTypeInfoForField(field);
      if(!typeInfo.isEnum()) {
        attribute = new JavaAttribute(fieldName, fieldNode.getProtoMessageName(), genericType, fieldIndex);
      }
      else {
        attribute = new EnumAttribute(fieldName, fieldNode.getProtoMessageName(), genericType, fieldIndex);
      }
    }
    else if(genericType instanceof TypeVariable) {
      Type bound = ((TypeVariable<?>) genericType).getBounds()[0];
      attribute = new JavaAttribute(fieldName, fieldNode.getProtoMessageName(), bound, fieldIndex);
    }
    else {
      attribute = new JavaAttribute(fieldName, fieldNode.getProtoMessageName(), genericType, fieldIndex);
    }
    node.addAttribute(attribute);
    setAttributeProperties(builder, syntax, node, field, attribute, fieldNode);
  }

  @SuppressWarnings("unused")
  public void setAttributeProperties(MessageNodeBuilder builder,
                                     ProtobufSchemaGenerator.ProtoSyntax syntax,
                                     AbstractMessageNode owningMessageNode,
                                     Field field,
                                     AbstractAttribute attribute,
                                     AbstractMessageNode fieldNode) {
    attribute.setContainingMessageNode(owningMessageNode);
    attribute.setSyntax(syntax);
    attribute.setTypeMessageNode(fieldNode);
    attribute.setIterableAttr(isCollectionOrMapTypeField(field));
    if(fieldNode instanceof ObjectMessageNode) {
      // this synthetic model is referenced by this attribute.  Update its references
      ((AbstractSyntheticMessageNode) fieldNode).getReferencedNodes().add(owningMessageNode);
    }
  }

}
