package com.github.krr.schema.generator.protobuf.model.builders;

import com.github.krr.schema.generator.protobuf.model.nodes.TypeNode;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.SyntheticAttribute;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.AbstractMessageNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.AbstractSyntheticMessageNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.CollectionMessageNode;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Collection;

/**
 * Collections need to be wrapped as messages only if they're nested
 * e.g. List&lt;List&lt;Foo&gt;&gt;, List&lt;Map&lt;Foo, Bar&gt;&gt; etc.  Primitive types of
 * Collections and collections of POJOs don't need any nesting.
 * &lt;p&gt;
 * List&lt;List&lt;Foo&gt;&gt; -&gt; ListOfFooTypes proto message and a repeated ListOfFooTypes
 * is the converted proto attribute type.  In other words a POJO attribute
 * List&lt;List&lt;Foo&gt;&gt; listOfListOfFoos -&gt; repeated ListOfFooTypes
 */
@SuppressWarnings("rawtypes")
@Slf4j
public class CollectionParameterizedMessageNodeBuilder extends ParameterizedTypeMessageNodeBuilder {

  public static final String ITEMS = "_items";

  protected AbstractSyntheticMessageNode getMessageNodeInstance(TypeNode typeNode,
                                                                AbstractMessageNode valueMessageNode,
                                                                ParameterizedType type) {
    // create a one of from the parameterized type.
    CollectionMessageNode collectionMessageNode = new CollectionMessageNode(type, typeNode.getName());
    SyntheticAttribute attribute = new SyntheticAttribute(ITEMS, getItemType(valueMessageNode), 10);
    attribute.setContainingMessageNode(collectionMessageNode);
    updateOneOfNode(typeNode, valueMessageNode, collectionMessageNode, attribute);
    attribute.setCollectionAttr(true);
    attribute.setTypeMessageNode(valueMessageNode);
    collectionMessageNode.addAttribute(attribute);
    return collectionMessageNode;
  }

  protected String getItemType(AbstractMessageNode messageNode) {
    return String.format("repeated %s", messageNode.getProtoMessageName());
  }

  protected Type getActualTypeArgument(ParameterizedType type) {
    return type.getActualTypeArguments()[0];
  }

  @Override
  public boolean supports(TypeNode typeNode) {
    Type type = typeNode.getType();
    return super.supports(typeNode) &&
           Collection.class.isAssignableFrom((Class) ((ParameterizedType) type).getRawType());
  }
}
