package com.github.krr.schema.generator.protobuf.model.builders;

import com.github.krr.schema.generator.protobuf.impl.ProtobufSchemaGenerator;
import com.github.krr.schema.generator.protobuf.model.MessageNodeBuilder;
import com.github.krr.schema.generator.protobuf.model.nodes.TypeNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.AbstractMessageNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.JavaBeanMessageNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.OneOfSyntheticMessageNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.Assert;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;

@SuppressWarnings("rawtypes")
@Slf4j
public class OneOfMessageNodeBuilder extends AbstractMessageModelNodeBuilder {

  @Override
  public AbstractMessageNode buildNode(MessageNodeBuilder messageNodeBuilder, TypeNode typeNode, ProtobufSchemaGenerator.ProtoSyntax syntax) {

    Type baseClassType = typeNode.getType();
    Type typeToUse = baseClassType;
    if(baseClassType instanceof ParameterizedType) {
      typeToUse = ((ParameterizedType)baseClassType).getRawType();
    }
    else if(baseClassType instanceof TypeVariable) {
      Type[] bounds = ((TypeVariable) baseClassType).getBounds();
      Assert.isTrue(bounds.length == 1, "Only support single type var in.  " + baseClassType + " has " + bounds.length + " vars");
      typeToUse = bounds[0];
    }
    else if(baseClassType instanceof WildcardType) {
      Type[] bounds = ((WildcardType) baseClassType).getUpperBounds();
      Assert.isTrue(bounds.length == 1, "Only support single type var in.  " + baseClassType + " has " + bounds.length + " vars");
      typeToUse = bounds[0];
    }
    String oneOfMessageName = getKey(messageNodeBuilder, typeToUse);
    AbstractMessageNode oneOfMessageNode = messageNodeBuilder.findNode(oneOfMessageName);
    if (oneOfMessageNode != null) {
      return oneOfMessageNode;
    }
    oneOfMessageNode = addClassToOneOf(messageNodeBuilder, typeNode, syntax, oneOfMessageName, typeToUse);
    return oneOfMessageNode;
  }

  private AbstractMessageNode addClassToOneOf(MessageNodeBuilder messageNodeBuilder,
                                              TypeNode typeNode,
                                              ProtobufSchemaGenerator.ProtoSyntax syntax,
                                              String oneOfMessageName,
                                              Type baseClassType) {
    Assert.isTrue(baseClassType instanceof Class, "OneOfMessageNodeBuilder can only be used with Class type.  type is " + typeNode);
    Class beanClass = (Class) baseClassType;
    String key = beanClass.getName();
    AbstractMessageNode baseClassMessage = messageNodeBuilder.findNode(key);
    if (baseClassMessage == null) {
      baseClassMessage = messageNodeBuilder.build(new TypeNode(key, beanClass), syntax);
      Assert.notNull(baseClassMessage, "Could not create model for " + beanClass);
    }
    OneOfSyntheticMessageNode oneOfMessageNode = new OneOfSyntheticMessageNode(oneOfMessageName, beanClass);
    messageNodeBuilder.start(oneOfMessageName, oneOfMessageNode);
    oneOfMessageNode.setRealMessage(baseClassMessage);
    oneOfMessageNode.setProtoMessageName(oneOfMessageName);
    // now process each subclass.  create the message nodes and add it to the oneOf.
    // if originalBeanType is not abstract add it to oneOf.
    Assert.isAssignable(JavaBeanMessageNode.class, baseClassMessage.getClass(), "Expecting JavaBeanMessageNode as base class");
    PojoWithSubclassesMessageModelNodeBuilder.processSubclasses(messageNodeBuilder, (JavaBeanMessageNode) baseClassMessage, beanClass, oneOfMessageNode, syntax);
    baseClassMessage.setWrappedMessage(oneOfMessageNode);
    messageNodeBuilder.registerMessage(oneOfMessageName, oneOfMessageNode);
    messageNodeBuilder.finish(oneOfMessageName);
    return oneOfMessageNode;
  }

  public String getKey(MessageNodeBuilder messageNodeBuilder, Type type) {
    return messageNodeBuilder.getProtoMessageName(type).concat("Types");
  }

  // this always returns true.  It's intended to be used directly and not with
  // the other factories.
  @Override
  public boolean supports(TypeNode typeNode) {
    return true;
  }
}
