package com.github.krr.schema.generator.protobuf.model.builders;

import com.github.krr.schema.generator.protobuf.impl.ProtobufSchemaGenerator;
import com.github.krr.schema.generator.protobuf.model.MessageNodeBuilder;
import com.github.krr.schema.generator.protobuf.model.nodes.TypeNode;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.ProtoPrimitiveAttribute;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.SyntheticAttribute;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.AbstractMessageNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.AbstractSyntheticMessageNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.GenericMessageNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.Assert;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;

@SuppressWarnings("rawtypes")
@Slf4j
public abstract class ParameterizedTypeMessageNodeBuilder extends AbstractMessageModelNodeBuilder {

  @Override
  public boolean supports(TypeNode typeNode) {
    return typeNode.getType() instanceof ParameterizedType;
  }

  protected final OneOfMessageNodeBuilder oneOfMessageBuilder = new OneOfMessageNodeBuilder();

  @Override
  public AbstractMessageNode buildNode(MessageNodeBuilder messageNodeBuilder, TypeNode typeNode, ProtobufSchemaGenerator.ProtoSyntax syntax) {

    ParameterizedType type = (ParameterizedType) typeNode.getType();
    String thisMessageTypeKey = AbstractSyntheticMessageNode.getKey(type);
    AbstractMessageNode node = messageNodeBuilder.findNode(thisMessageTypeKey);
    if (node != null) {
      return node;
    }
    // this is a parameterized type (map/list/generic).  get the inner type
    // that this message node will wrap
    Type valueType = getActualTypeArgument(type);
    String valueKey = AbstractSyntheticMessageNode.getKey(valueType);
    AbstractMessageNode valueMessageNode;
    if (!isProtoPrimitive(valueType)) {
      TypeNode valueTypeNode = new TypeNode(valueKey, valueType);
      valueMessageNode = messageNodeBuilder.findNode(valueKey);
      if (valueMessageNode == null) {
        valueMessageNode = messageNodeBuilder.build(valueTypeNode, syntax);
        Assert.notNull(valueMessageNode, "Expecting non-null wrapped node for " + valueTypeNode);
      }
      // get the one type that represents all the subclasses of the valueType
      AbstractSyntheticMessageNode syntheticMessageNode;
      boolean enumType = false;
      if(valueType instanceof Class) {
        enumType = ((Class) valueType).isEnum();
      }
      boolean syntheticNode = valueMessageNode instanceof AbstractSyntheticMessageNode;
      TypeNode thisMessageTypeNode = new TypeNode(thisMessageTypeKey, type);
      if (!syntheticNode && !enumType) {
        syntheticMessageNode = (AbstractSyntheticMessageNode) oneOfMessageBuilder.buildNode(messageNodeBuilder, valueTypeNode, syntax);
        node = getMessageNodeInstance(thisMessageTypeNode, syntheticMessageNode, type);
      }
      else if (syntheticNode && !enumType) {
        syntheticMessageNode = (AbstractSyntheticMessageNode) valueMessageNode;
        node = getMessageNodeInstance(thisMessageTypeNode, syntheticMessageNode, type);
      }
      else {
        // enum type.
        node = getMessageNodeInstance(thisMessageTypeNode, valueMessageNode, type);
      }
      // we have a nested node -> add it as a child of the wrappedNode
      // this will trigger creation of a message.
      log.debug("Adding nested message {} to {}", node.getKey(), valueMessageNode.getKey());
    }
    else if(valueType == Object.class) {
      TypeNode valueTypeNode = new TypeNode(valueKey, valueType);
      valueMessageNode = messageNodeBuilder.findNode(valueKey);
      if (valueMessageNode == null) {
        valueMessageNode = messageNodeBuilder.build(valueTypeNode, syntax);
        Assert.notNull(valueMessageNode, "Expecting non-null wrapped node for " + valueTypeNode);
      }
      node = getMessageNodeInstance(new TypeNode(thisMessageTypeKey, type), valueMessageNode, type);
    }
    else if(isProtoPrimitive(valueType)) {
      // proto primitive node.
      String attrType = ProtoPrimitiveAttribute.PROTO_PRIMITIVE_TYPE_MAP.get((Class)valueType);
      valueMessageNode = new GenericMessageNode(attrType, (Class) valueType, true);
      valueMessageNode.setProtoMessageName(attrType);
      node = getMessageNodeInstance(new TypeNode(thisMessageTypeKey, type), valueMessageNode, type);
    }
    else {
      throw new UnsupportedOperationException("Unsupported type fr ParameterizedTypeMessageNodeBuilder" + valueType);
    }
    if (node != null) {
      // proto primitive types are not created as messages
      log.debug("Node {} for Parameterized node builder created {}", valueKey, node);
      messageNodeBuilder.registerMessage(thisMessageTypeKey, node);
    }
    return node;
  }

  protected abstract AbstractSyntheticMessageNode getMessageNodeInstance(TypeNode typeNode,
                                                                         AbstractMessageNode oneOfNode,
                                                                         ParameterizedType type);

  protected Type getActualTypeArgument(ParameterizedType type) {
    throw new UnsupportedOperationException("Must be implemented by subclasses");
  }

  protected void updateOneOfNode(TypeNode typeNode, AbstractMessageNode oneOfNode, AbstractMessageNode messageNode, SyntheticAttribute attribute) {
    attribute.setIterableAttr(true);
    if(oneOfNode instanceof AbstractSyntheticMessageNode) {
      AbstractSyntheticMessageNode syntheticMessageNode = (AbstractSyntheticMessageNode) oneOfNode;
      messageNode.setWrappedMessage(syntheticMessageNode);
      syntheticMessageNode.getReferencedNodes().add(messageNode);
      // if the collection message node turns out to not be visible, this synthetic node will
      // present the oneOfNode as its type.
      attribute.setTypeMessageNode(oneOfNode);
    }
    messageNode.setProtoMessageName(typeNode.getName());
  }


}
