package com.github.krr.schema.generator.protobuf.model.builders;

import com.github.krr.schema.generator.annotations.SchemaItem;
import com.github.krr.schema.generator.protobuf.impl.ProtobufSchemaGenerator;
import com.github.krr.schema.generator.protobuf.model.MessageNodeBuilder;
import com.github.krr.schema.generator.protobuf.model.nodes.TypeNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.AbstractMessageNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.JavaBeanMessageNode;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.OneOfSyntheticMessageDecorator;
import com.github.krr.schema.generator.protobuf.model.nodes.messages.OneOfSyntheticMessageNode;
import com.google.common.graph.Graph;
import com.google.common.graph.GraphBuilder;
import com.google.common.graph.ImmutableGraph;
import com.google.common.graph.Traverser;
import lombok.extern.slf4j.Slf4j;
import org.reflections.Reflections;
import org.reflections.scanners.Scanners;
import org.reflections.util.ConfigurationBuilder;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;

import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Processes a Pojo with subclasses.
 * A class Foo with subclasses
 * - creates a message for each subclass
 * - creates a synthetic message containing all subclasses and the
 * base class if the base class is not abstract.  Ex. if Foo is not
 * abstract, and has subclasses (Bar and Baz) the synthetic class will
 * be composed of (Foo, Bar, Boz).  If Foo is abstract the one of will have
 * (Bar, Baz).
 */
@SuppressWarnings({"rawtypes", "DuplicatedCode", "unchecked", "UnstableApiUsage"})
@Slf4j
public class PojoWithSubclassesMessageModelNodeBuilder extends PojoMessageModelNodeBuilder {

  private final OneOfMessageNodeBuilder oneOfMessageNodeBuilder = new OneOfMessageNodeBuilder();

  private static final Map<Class, ImmutableGraph<Class>> SUBCLASS_MAP = new HashMap<>();

  private static final Map<String, Reflections> REFLECTIONS_CACHE = new HashMap<>();

  @Override
  public AbstractMessageNode buildNode(MessageNodeBuilder builder, TypeNode typeNode, ProtobufSchemaGenerator.ProtoSyntax syntax) {
    // get the message node for this type

    Class clazz = (Class) typeNode.getType();
    AbstractMessageNode baseClassMessage = super.buildNode(builder, typeNode, syntax);
    Assert.notNull(baseClassMessage, "Expecting non-null message for " + typeNode);
    OneOfSyntheticMessageNode oneOfMessage = (OneOfSyntheticMessageNode) oneOfMessageNodeBuilder.buildNode(builder, typeNode, syntax);
    Assert.isAssignable(JavaBeanMessageNode.class, baseClassMessage.getClass(), "Expecting JavaBeanMessageNode as base class");
    if (clazz.getAnnotation(SchemaItem.class) != null) {
      processSubclasses(builder, (JavaBeanMessageNode) baseClassMessage, clazz, oneOfMessage, syntax);
      baseClassMessage.setWrappedMessage(oneOfMessage);
    }
    return baseClassMessage;
  }

  public static void processSubclasses(MessageNodeBuilder messageNodeBuilder,
                                       JavaBeanMessageNode baseClassMessage,
                                       Class beanClass,
                                       OneOfSyntheticMessageNode oneOfSyntheticMessageNode,
                                       ProtobufSchemaGenerator.ProtoSyntax syntax) {
    Graph<Class> subTypesGraphForClass = getSubTypesForClass(beanClass);
    Traverser<Class> traverser = Traverser.forGraph(subTypesGraphForClass);
    int subclassCount = 0;
    for (Class<?> subtype : traverser.depthFirstPostOrder(beanClass)) {
      SchemaItem annotation = subtype.getAnnotation(SchemaItem.class);
      // only add the subtype to oneOf it it is not abstract
      if (annotation != null && !Modifier.isAbstract(subtype.getModifiers())) {
        log.debug("Processing message for {}", subtype);
        String key = subtype.getName();
        TypeNode typeNode = new TypeNode(key, subtype);
        AbstractMessageNode subclassTypeMessage = messageNodeBuilder.findNode(key);
        if (subclassTypeMessage == null) {
          subclassTypeMessage = messageNodeBuilder.build(typeNode, syntax);
          Assert.notNull(subclassTypeMessage, "Expecting non-null message for " + typeNode);
          subclassTypeMessage.setProtoMessageName(getProtoCompatibleName(key));
        }
        long oneOfModelIndex = getIndex(subtype);
        OneOfSyntheticMessageDecorator decorator = new OneOfSyntheticMessageDecorator(subclassTypeMessage, oneOfModelIndex);
        decorator.setProtoMessageName(getProtoCompatibleName(key));
        oneOfSyntheticMessageNode.addNestedMessage(decorator);
        subclassCount++;
      }
    }
    if (subclassCount > 0) {
      log.debug("Class {} has {} subclasses", beanClass, subclassCount);
      baseClassMessage.setHasSubclasses(true);
    }
    log.debug("Added {} subtypes to oneof message {}", subTypesGraphForClass.nodes().size(), oneOfSyntheticMessageNode.getName());
  }

  public static Graph<Class> getSubTypesForClass(Class clazz) {
    return SUBCLASS_MAP.computeIfAbsent(clazz, c -> {
      if (shouldConsiderSuperclass(clazz)) {
        return GraphBuilder.directed().<Class>immutable().build();
      }
      String packageName = clazz.getPackage().getName();
      Reflections reflections = REFLECTIONS_CACHE.computeIfAbsent(packageName, n -> new Reflections(new ConfigurationBuilder().addScanners(Scanners.values())
                                                                                                                              .forPackages(n)));
      Set<Class<?>> subTypes = reflections.getSubTypesOf((Class<Object>) clazz);
      log.debug("Found {} subtypes for {}", subTypes.size(), clazz.getName());
      ImmutableGraph.Builder<Class> graphBuilder = GraphBuilder.directed().immutable();
      if (!Modifier.isAbstract(clazz.getModifiers())) {
        log.debug("Class {} is not abstract adding to oneOf", clazz);
        graphBuilder.addNode(clazz);
      }

      for (Class<?> subType : subTypes) {
        Class type = subType;
        do {
          graphBuilder.putEdge(type.getSuperclass(), type);
          type = type.getSuperclass();
        } while (type != clazz);
      }
      return graphBuilder.build();
    });
  }

  private static boolean shouldConsiderSuperclass(Class clazz) {
    return clazz == Object.class || clazz.getPackageName().startsWith("java") || ClassUtils.isPrimitiveOrWrapper(clazz);
  }

  @Override
  public boolean supports(TypeNode typeNode) {
    Type type = typeNode.getType();
    // non-generic class
    if (!(type instanceof Class)) {
      return false;
    }// not generic
    // not a proto primitive type
    // has subclasses
    Class clazz = (Class) type;
    return clazz.getTypeParameters().length == 0 &&
           // not a proto primitive type
           !isProtoPrimitive(type) &&
           // has subclasses
           getSubTypesForClass(clazz).nodes().size() > 0;
  }
}
