package com.github.krr.schema.generator.protobuf.model.nodes.messages;

import com.github.krr.schema.generator.protobuf.mappercodegen.MapperConfig;
import com.github.krr.schema.generator.protobuf.model.builders.ObjectMessageNodeBuilder;
import com.github.krr.schema.generator.protobuf.model.builders.PojoWithSubclassesMessageModelNodeBuilder;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.AbstractAttribute;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.ProtoPrimitiveAttribute;
import com.github.krr.schema.generator.protobuf.model.nodes.attributes.SyntheticAttribute;
import lombok.Getter;
import lombok.SneakyThrows;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

@SuppressWarnings({"rawtypes", "UnstableApiUsage"})
public abstract class AbstractSyntheticMessageNode extends AbstractMessageNode {

  @Getter
  protected Set<AbstractMessageNode> nestedNodes = new LinkedHashSet<>();

  @Getter
  protected Set<AbstractMessageNode> referencedNodes = new LinkedHashSet<>();

  @Getter
  protected final Type javaType;

  protected AbstractSyntheticMessageNode(String name, Type javaType) {
    super(name);
    this.javaType = javaType;
  }

  @Override
  public void addAttribute(AbstractAttribute attribute) {
    super.addAttribute(attribute);

  }

  @Override
  public String getBeanJavaType() {
    // the type variables will not be resolvable in the generated
    // mapper code.  So we just return a Map
//    if(javaType instanceof ParameterizedType) {
//      return ((ParameterizedType) javaType).getRawType().getTypeName();
//    }
    return javaType.getTypeName();
  }

  @Override
  public String getProtoJavaType(MapperConfig mapperConfig) {
    return protoMessageName;
  }

  /**
   * @return - boolean if we should create nested messages
   */
  @SuppressWarnings("unused")
  public boolean isCreateMessage() {
    return nestedNodes.size() > 0;
  }

  public static String getKey(Type type) {
    Type currentType = type;
    if (currentType instanceof ParameterizedType) {
      Type rawType = ((ParameterizedType) currentType).getRawType();
      String typeName = rawType.getTypeName();
      String prefix = typeName.substring(typeName.lastIndexOf(".") + 1);
      String suffix;
      if (Collection.class.isAssignableFrom((Class<?>) rawType)) {
        currentType = ((ParameterizedType) currentType).getActualTypeArguments()[0];
        suffix = "Of".concat(StringUtils.capitalize(getKey(currentType)));
      }
      else if (Map.class.isAssignableFrom((Class<?>) rawType)) {
        currentType = ((ParameterizedType) currentType).getActualTypeArguments()[1];
        suffix = "Of".concat(StringUtils.capitalize(getKey(currentType)));
      }
      else {
        Type[] actualTypeArguments = ((ParameterizedType) currentType).getActualTypeArguments();
        suffix = getKeyForTypes(actualTypeArguments);
      }
      return String.format("%s%s", prefix, StringUtils.capitalize(suffix));
    }
    else {
      if (currentType instanceof Class && ClassUtils.isPrimitiveOrWrapper((Class<?>) currentType) || currentType == String.class) {
        return ProtoPrimitiveAttribute.PROTO_PRIMITIVE_TYPE_MAP.get(currentType);
      }
      else if (currentType instanceof Class) {
        TypeVariable<? extends Class<?>>[] typeParameters = ((Class<?>) currentType).getTypeParameters();
        if (typeParameters.length > 0) {
          String suffix = getKeyForTypes(typeParameters);
          return String.format("%s%s", getProtoMessageName((Class) currentType), StringUtils.capitalize(suffix));
        }
        else {
          return getProtoMessageName((Class) currentType);
        }
      }
      else if (currentType instanceof TypeVariable) {
        TypeVariable typeVar = (TypeVariable) currentType;
        Type[] bounds = typeVar.getBounds();
        if (bounds.length > 1) {
          throw new UnsupportedOperationException("More than one parameter is not supported in a generic " + currentType);
        }
        Type actualType = bounds[0];
        return getKey(actualType);
      }
      else if (currentType instanceof WildcardType) {
        WildcardType typeVar = (WildcardType) currentType;
        Type[] bounds = typeVar.getUpperBounds();
        if (bounds.length > 1) {
          throw new UnsupportedOperationException("More than one parameter is not supported in a wildcard generic type " + currentType);
        }
        Type actualType = bounds[0];
        return getKey(actualType);
      }
      else {
        throw new UnsupportedOperationException("Unsupported Type: " + currentType);
      }
    }
  }

  private static String getKeyForTypes(Type[] actualTypeArguments) {
    String suffix = "WithTypesOf";
    boolean first = true;
    for (Type actualTypeArgument : actualTypeArguments) {
      if (!first) {
        suffix = suffix.concat("And");
      }
      if (first) {
        first = false;
      }
      suffix = suffix.concat(getKey(actualTypeArgument));
    }
    return suffix;
  }

  @SneakyThrows
  public static String getProtoMessageName(Class clazz) {
    String suffix = "";
    String typeName = clazz.getName();
    if (clazz == Object.class) {
      typeName = ObjectMessageNodeBuilder.PROTO_MESSAGE_TYPE;
    }
    if (PojoWithSubclassesMessageModelNodeBuilder.getSubTypesForClass(clazz).nodes().size() > 0) {
      suffix = "Types";
    }
    return typeName.substring(typeName.lastIndexOf(".") + 1).concat(suffix).replaceAll("\\$", "Dollar");
  }

  public void addNestedMessage(AbstractMessageNode messageNode) {
    nestedNodes.add(messageNode);
    if (messageNode instanceof AbstractSyntheticMessageNode) {
      ((AbstractSyntheticMessageNode) messageNode).getReferencedNodes().add(this);
    }
  }

  /**
   * A model is visible (i.e. proto message is generated) if it has nested nodes which means that
   * this
   *
   * @return - true if it is referenced by other models.
   */
  @Override
  public boolean isModelVisible() {
    return referencedNodes.size() > 0 && referencedNodes.contains(this);
  }

  @Override
  public String getWrappedAttributeType() {
    return attributes.iterator().next().getType();
  }

  @Override
  public String getBeanItemJavaType() {
    if (!isModelVisible() && wrappedMessage != null) {
      return wrappedMessage.getBeanItemJavaType();
    }
    return javaType.getTypeName().replaceAll("\\$", ".");
  }

  @Override
  public String getProtoItemJavaType() {
    if (!isModelVisible() && wrappedMessage != null) {
      return wrappedMessage.getProtoItemJavaType();
    }
    return protoMessageName;
  }

  @Override
  public SyntheticAttribute getNestedAttribute() {
    return (SyntheticAttribute) attributes.iterator().next();
  }

  public String getToProtoItemMethodName() {
    throw new UnsupportedOperationException("Must be implemented by subclasses");
  }

  public String getFromProtoItemMethodName() {
    throw new UnsupportedOperationException("Must be implemented by subclasses");
  }

  @Override
  public boolean isRenderModel() {
    return referencedNodes.size() > 0;
  }

  @Override
  public boolean isModelVisibleTo(AbstractMessageNode message) {
    return referencedNodes.contains(message);
  }
}
