package com.github.krr.schema.generator.protobuf.models;

import lombok.AccessLevel;
import lombok.Data;
import lombok.Setter;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.ClassUtils;

import java.lang.reflect.*;
import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @todo This entire class needs to be revisited.
 */
@SuppressWarnings({"rawtypes", "unused"})
@Data
@Slf4j
public class TypeInfo {

  public static final String DOLLAR = "Dollar";
  public static final String DOT = ".";

  private static final List<GenericInfoUpdater> UPDATERS = List.of(new ClassTypeUpdater(),
                                                                   new TypeVariableTypeUpdater(),
                                                                   new ParameterizedTypeUpdater(),
                                                                   new WildcardTypeUpdater());

  // the class whose type info is held here.
  private Class containedClass;

  private TypeInfo containedType;

  // the raw type if this class is a generic type.
  private Type rawType;

  @Setter(AccessLevel.NONE)
  private int collectionCount;

  @Setter(AccessLevel.NONE)
  private int mapCount;

  /**
   * Set to true only if the outermost class is a Map
   * i.e. List&lt;Map&lt;String, List&lt;Foo&gt;&gt; would be false
   * but Map&lt;String, Foo&gt; would be true.
   */
  private boolean isMap;

  /**
   * Set to true only if the outermost class is List.
   * i.e. Map&lt;String, List&lt;Foo&gt;&gt; would be false (although
   * this is currently not supported)
   */
  private boolean isCollection;

  private boolean isMapSubclass;
  private boolean isCollectionSubclass;
  private Set<Class<?>>subclasses =null;

  public void setMap(boolean map) {
    this.isMap = map && !isCollection;
    if (map) {
      this.mapCount++;
    }
  }

  public boolean isAbstract() {
    return Modifier.isAbstract(containedClass.getModifiers());
  }

  /**
   * @return - the innermost class (assuming only one parameterized type)
   * @todo keeps things backward compatible.  Needs fixing.
   */
  public Class getContainedClass() {
    if (containedType == null) {
      return containedClass;
    }
    return containedType.getContainedClass();
  }

  public Class getInnerContainedClass() {
    if (containedType != null) {
      return containedType.containedClass;
    }
    return null;
  }

  public void setCollection(boolean collection) {
    this.isCollection = collection && !isMap;
    if (collection) {
      this.collectionCount++;
    }
  }

  public boolean isSimpleCollection() {
    return collectionCount == 1 && mapCount == 0;
  }

  public boolean isSimpleMap() {
    return mapCount == 1 && collectionCount == 0;
  }

  public boolean isListOfList() {
    return this.isCollection && this.collectionCount == 2 && !isCollectionSubclass();
  }

  public boolean isMapOfMap() {
    return this.isMap && this.mapCount == 2 && !isMapSubclass();
  }

  public boolean isListOfMap() {
    return this.isCollection && this.mapCount == 1 && this.collectionCount == 1;
  }

  public boolean isMapOfList() {
    return this.isMap && this.mapCount == 1 && this.collectionCount == 1;
  }

  public String getCollectionType() {
    return rawType.getTypeName();
  }

  public String getCollectionNewInstance() {
    final String typeName = rawType.getTypeName();
    if ("java.util.List".equals(typeName)) {
      return "new java.util.ArrayList<>()";
    }
    else if ("java.util.Set".equals(typeName)) {
      return "new java.util.HashSet<>()";
    }
    else if ("java.util.Map".equals(typeName)) {
      return "new java.util.HashMap<>()";
    }
    else {
      return String.format("new %s<>()", typeName);
    }
  }

  public boolean isPojo() {
    return (mapCount == 0 && collectionCount == 0 && !isPrimitiveOrWrapper());
  }

  public boolean isPrimitiveOrWrapper() {
    return ClassUtils.isPrimitiveOrWrapper(containedClass) ||
           ClassUtils.isPrimitiveWrapperArray(containedClass) ||
           ClassUtils.isPrimitiveArray(containedClass) ||
           containedClass == String.class
        ;

  }

  public String getCollectionTypeLabel() {
    if (isSimple()) {
      return "";
    }
    final String typeName = rawType.getTypeName();
    if ("java.util.List".equals(typeName)) {
      return "list";
    }
    if ("java.util.Set".equals(rawType.getTypeName())) {
      return "set";
    }
    if ("java.util.Map".equals(rawType.getTypeName())) {
      return "map";
    }
    throw new UnsupportedOperationException("Unrecognized collection type:" + rawType);
  }

  public String getFqJavaType() {
    return getFqClassname().replaceAll("\\$", "\\.");
  }

  public String getFqClassname() {
    String classname = containedClass.getName();
    if (isListOfList()) {
      return String.format("java.util.Collection<%s>", classname);
    }
    else if (isListOfMap()) {
      return String.format("java.util.Map<String, %s>", classname);
    }
    else if (isCollection()) {
      return String.format("%s<%s>", getCollectionType(), classname);
    }
    else if (isMap()) {
      return String.format("java.util.Map<String, %s>", classname);
    }
    else {
      return classname;
    }
  }

  @SneakyThrows
  public String getProtoCollectionJavaTypePrefix() {
    Class rawTypeClass = Class.forName(rawType.getTypeName());
    if (Map.class.isAssignableFrom(rawTypeClass)) {
      return "java.util.Map<String,";
    }
    else if (List.class.isAssignableFrom(rawTypeClass)) {
      return "java.util.List<";
    }
    else if (Set.class.isAssignableFrom(rawTypeClass)) {
      return "java.util.Set<";
    }
    throw new UnsupportedOperationException("Unsupported raw type " + rawType);
  }


  /**
   * Returns a string that can be used as a valid class name.
   * Protos don't use $ so in proto schemas, java names with
   * a $ get converted to a literal 'Dollar'.  This method gets
   * the corresponding class name used for the Differ.
   *
   * @return - A string containing the proto compatible classname
   */
  public String getProtoMessageName() {
    String classname = containedClass.getName();
    classname = classname.substring(classname.lastIndexOf(DOT) + 1);
    return classname.replaceAll("\\$", DOLLAR);
  }

  public String getSimpleType() {
    String name = containedClass.getName();
    return name.substring(name.lastIndexOf(DOT) + 1);
  }

  public String getJavaType() {
    return containedClass.getName().replaceAll("\\$", ".");
  }

  public String getMessageKey() {
    return containedClass.getName();
  }

  public boolean isSimple() {
    return !isCollection && !isMap;
  }

  public boolean isPrimitive() {
    return ClassUtils.isPrimitiveOrWrapper(containedClass);
  }

  public boolean isPrimitiveOrStringOrObject() {
    return isPrimitive() || containedClass == String.class || containedClass == Object.class ||
           containedClass == Long.class || containedClass == Double.class || containedClass == Float.class ||
           containedClass == BigDecimal.class || containedClass == Integer.class;
  }

  public boolean isEnum() {
    return containedClass.isEnum();
  }

  public boolean isBoolean() {
    return Boolean.class.isAssignableFrom(containedClass) || boolean.class.isAssignableFrom(containedClass);
  }

  public boolean isUnboxedType() {
    return boolean.class.isAssignableFrom(containedClass) ||
           int.class.isAssignableFrom(containedClass) ||
           long.class.isAssignableFrom(containedClass) ||
           double.class.isAssignableFrom(containedClass) ||
           float.class.isAssignableFrom(containedClass);
  }

//  public static TypeNode buildTypeNodeGraphForClass(Class clazz) {
//  }

  public static TypeInfo getTypeInfoForField(Field field) {
    TypeInfo typeInfo = new TypeInfo();
    Type genericType = field.getGenericType();
    Class<?> rawType = field.getType();

    Class genericClass;
    updateTypeInfoForType(typeInfo, genericType);

    return typeInfo;
  }

  public static TypeInfo getTypeInfoForType(Class clazz) {
    TypeInfo typeInfo = new TypeInfo();
    updateTypeInfoForType(typeInfo, clazz);
    return typeInfo;
  }

  public static boolean isCollectionType(Class<?> clazz) {
    return Collection.class.isAssignableFrom(clazz);
  }

  public static boolean isMapType(Class<?> clazz) {
    return Map.class.isAssignableFrom(clazz);
  }

  public static void updateTypeInfoForType(TypeInfo typeInfo, Type genericType) {
    for (GenericInfoUpdater updater : UPDATERS) {
      if (updater.supports(genericType)) {
        log.debug("TypeInfo:updateTypeInfoForType: Updating type for {}", genericType);
        updater.updateTypeInfo(genericType,
                               typeInfo.isMap ? VariableType.MAP_VALUE : VariableType.COLL_ITEM,
                               typeInfo);
      }
    }
  }

  public boolean isGenericClass() {
    return rawType != null && !isListClass() && !isMapClass();
  }

  private boolean isListClass() {
    return compareRawType("java.util.List");
  }

  private boolean isMapClass() {
    return compareRawType("java.util.Map");
  }

  private boolean compareRawType(String type) {
    return type.equals(rawType.getTypeName());
  }

  public boolean isNested() {
    return isListOfList() ||
           isListOfMap() ||
           isMapOfList() ||
           isMapOfMap() ||
           isGenericClass()
        ;
  }

  private interface GenericInfoUpdater {
    void updateTypeInfo(Type type, VariableType variableType, TypeInfo typeInfo);

    default void doUpdate(TypeInfo info, Type type) {
      log.debug("Updating type for {}", type);
      try {
        // this is a type that this class understands.  Try casting
        // this to a Class type.  if it works we're done.  Else we
        // have to recurse till we get to the last level.
        Class clazz = (Class) type;
        info.setContainedClass(clazz);
      }
      catch (ClassCastException e) {
        // this is not a type that this class understands.
        updateTypeInfoForType(info, type);
      }
    }

    default Type getType(Type[] types, VariableType variableType) {
      return types[variableType.getIndexInTypeArray()];
    }

    boolean supports(Type genericType);

  }

  private static class ClassTypeUpdater implements GenericInfoUpdater {

    @Override
    public void updateTypeInfo(Type type, VariableType variableType, TypeInfo typeInfo) {
      Class clazz = (Class) type;
      if (isMapType(clazz)) {
        typeInfo.setMapSubclass(true);
        typeInfo.setMap(true);
        // the class itself is a subclass of Map.  Get the type params
        // from the superclass
        //@todo: Add Support for generic value types in map.
        updateTypeInfoForType(typeInfo, clazz.getGenericSuperclass());
      }
      else if (isCollectionType(clazz)) {
        typeInfo.setCollection(true);
        typeInfo.setCollectionSubclass(true);
        // subclass of collection.  Get type parameters from the super class.
        updateTypeInfoForType(typeInfo, clazz.getGenericSuperclass());
      }
      else {
        doUpdate(typeInfo, type);
      }
    }

    @Override
    public boolean supports(Type genericType) {
      return genericType instanceof Class;
    }
  }

  private static class TypeVariableTypeUpdater implements GenericInfoUpdater {

    @Override
    public void updateTypeInfo(Type type, VariableType variableType, TypeInfo typeInfo) {
      TypeVariable typeVariable = (TypeVariable) type;
      Type[] bounds = typeVariable.getBounds();
      Type boundType;
      if (bounds.length == 1) {
        boundType = bounds[0];
      }
      else {
        boundType = getType(bounds, variableType);
      }
      doUpdate(typeInfo, boundType);
    }

    @Override
    public boolean supports(Type genericType) {
      return genericType instanceof TypeVariable;
    }

  }

  private static class ParameterizedTypeUpdater implements GenericInfoUpdater {

    @Override
    public boolean supports(Type type) {
      return type instanceof ParameterizedType;
    }

    @Override
    public void updateTypeInfo(Type type, VariableType variableType, TypeInfo typeInfo) {
      ParameterizedType parameterizedType = (ParameterizedType) type;
      Type[] actualTypeArguments = parameterizedType.getActualTypeArguments();
      Type rawType = parameterizedType.getRawType();
      if (typeInfo.rawType == null) {
        typeInfo.rawType = rawType;
        Type collectionItemType;
        if ( isCollectionType((Class<?>)rawType)){
          typeInfo.setCollection(true);
          collectionItemType = actualTypeArguments[0];
          // if the collectionItemType is a List/Map/Generic, it'll be a parameterized type.
        }
        else if ( isMapType((Class<?>)rawType)){
          typeInfo.setMap(true);
          collectionItemType = actualTypeArguments[1];
        }
        else{
          collectionItemType = actualTypeArguments[0];
        }
        doUpdate(typeInfo, collectionItemType);
      }
      else {
        // the outer raw type has already been set.  Get the inner typeinfo
        TypeInfo innerTypeInfo = new TypeInfo();
        updateTypeInfoForType(innerTypeInfo, type);
        typeInfo.containedType = innerTypeInfo;
        typeInfo.containedClass = (Class) innerTypeInfo.getRawType();
        typeInfo.collectionCount += innerTypeInfo.collectionCount;
        typeInfo.mapCount += innerTypeInfo.mapCount;
      }
    }
  }

  private static class WildcardTypeUpdater implements GenericInfoUpdater {

    @Override
    public void updateTypeInfo(Type type, VariableType variableType, TypeInfo typeInfo) {
      WildcardType wildcardType = (WildcardType) type;
      // the item type itself is a parameterized type e.g. List<Foo<X>>
      // here only support if Foo = Map or List.
      Type[] upperBounds = wildcardType.getUpperBounds();
      log.debug("Updating type info for variable type {}", wildcardType);
      doUpdate(typeInfo, getType(upperBounds, VariableType.COLL_ITEM));
    }

    @Override
    public boolean supports(Type genericType) {
      return genericType instanceof WildcardType;
    }
  }

}