package com.luues.redis.cluster.config;

import com.luues.util.TypeConvert;
import com.luues.util.logs.LogUtil;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;

import java.text.ParseException;
import java.util.HashSet;
import java.util.Set;

public class JedisClusterFactory implements FactoryBean<JedisCluster>, InitializingBean {

    @Value("${redis.maxIdle:300}")
    private Integer maxIdle;
    @Value("${redis.maxTotal:1000}")
    private Integer maxTotal;
    @Value("${redis.maxWaitMillis:1000}")
    private Integer maxWaitMillis;
    @Value("${redis.minEvictableIdleTimeMillis:1800000}")
    private Integer minEvictableIdleTimeMillis;
    @Value("${redis.numTestsPerEvictionRun:3}")
    private Integer numTestsPerEvictionRun;
    @Value("${redis.timeBetweenEvictionRunsMillis:-1}")
    private long timeBetweenEvictionRunsMillis;
    @Value("${redis.testOnBorrow:true}")
    private boolean testOnBorrow;
    @Value("${redis.testOnReturn:true}")
    private boolean testOnReturn;
    @Value("${redis.testWhileIdle:true}")
    private boolean testWhileIdle;
    @Value("${redis.timeout:2000}")
    private Integer timeout;
    @Value("${redis.max-redirections:5}")
    private Integer maxRedirections;
    @Value("${redis.password:null}")
    private String password;

    private JedisCluster jedisCluster;
    private Set<String> jedisClusterNodes;

    @Override
    public JedisCluster getObject() {
        return jedisCluster;
    }

    @Override
    public Class<?> getObjectType() {
        return (this.jedisCluster != null ? this.jedisCluster.getClass() : JedisCluster.class);
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    @Override
    public void afterPropertiesSet() throws ParseException {
        if (jedisClusterNodes == null || jedisClusterNodes.size() == 0) {
            throw new NullPointerException("jedisClusterNodes is null.");
        }
        Set<HostAndPort> haps = new HashSet<HostAndPort>();
        for (String node : jedisClusterNodes) {
            String[] arr = node.split(":");
            if (arr.length != 2) {
                throw new ParseException("node address error !",node.length()-1);
            }
            haps.add(new HostAndPort(arr[0], Integer.valueOf(arr[1])));
        }
        GenericObjectPoolConfig genericObjectPoolConfig = new GenericObjectPoolConfig();
        genericObjectPoolConfig.setMaxIdle(maxIdle);
        genericObjectPoolConfig.setMaxTotal(maxTotal);
        genericObjectPoolConfig.setMaxWaitMillis(maxWaitMillis);
        genericObjectPoolConfig.setMinEvictableIdleTimeMillis(minEvictableIdleTimeMillis);
        genericObjectPoolConfig.setNumTestsPerEvictionRun(numTestsPerEvictionRun);
        genericObjectPoolConfig.setTimeBetweenEvictionRunsMillis(timeBetweenEvictionRunsMillis);
        genericObjectPoolConfig.setTestOnBorrow(testOnBorrow);
        genericObjectPoolConfig.setTestOnReturn(testOnReturn);
        genericObjectPoolConfig.setTestWhileIdle(testWhileIdle);
        if (!TypeConvert.isNull(password))
            jedisCluster = new JedisCluster(haps, timeout, timeout, maxRedirections, password, genericObjectPoolConfig);
        jedisCluster = new JedisCluster(haps, timeout, timeout, maxRedirections, genericObjectPoolConfig);
        LogUtil.info("\n{\n　　　　　redis-cluster start...\n　　　　　nodes:{}\n　　　　　password:{}" +
                        "\n　　　　　MaxTotal:{}\n　　　　　MaxIdle:{}\n　　　　　MaxWaitMillis:{}" +
                        "\n　　　　　TestOnBorrow:{}\n　　　　　testOnReturn:{}\n　　　　　" +
                        "minEvictableIdleTimeMillis:{}\n" +
                        "　　　　　numTestsPerEvictionRun:{}\n　　　　　timeBetweenEvictionRunsMillis:{}\n" +
                        "　　　　　testWhileIdle:{}\n}",
                String.join(",", String.join(",", jedisCluster.getClusterNodes().keySet())),
                password, maxTotal, maxIdle, maxWaitMillis, testOnBorrow, testOnReturn,
                minEvictableIdleTimeMillis, numTestsPerEvictionRun, timeBetweenEvictionRunsMillis, testWhileIdle);
    }

    public void setJedisClusterNodes(Set<String> jedisClusterNodes) {
        this.jedisClusterNodes = jedisClusterNodes;
    }

}
