package com.luues.redis.cluster.service;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import redis.clients.jedis.*;

import java.util.*;
import java.util.function.Consumer;

@Component
public class JedisClusterTemplate {

    @Autowired(required = false)
    private JedisCluster jedisCluster;
    /*@Autowired(required = false)
    private Jedis jedis;

    public Jedis getJedisClient(){
        return jedis;
    }*/

    public JedisCluster getJedisCluster(){
        return jedisCluster;
    }


    public Set<String> keys(String pattern) {
        Set<String> stringSet = new HashSet<String>();
        String cursor = ScanParams.SCAN_POINTER_START;
        ScanParams scanParams = new ScanParams();
        scanParams.match(pattern);
        scanParams.count(1000);
        while (true) {
            ScanResult<String> sscanResult = getJedisCluster().scan(cursor, scanParams);
            cursor = sscanResult.getStringCursor();// 返回0 说明遍历完成
            stringSet.addAll(sscanResult.getResult());
            if ("0".equals(cursor)) {
                break;
            }
        }
        return stringSet;
    }

    public void delKeys(String pattern) {
        Set<String> strings = keys(pattern);
        if (strings.size() == 0) {
            return;
        }
        Iterator<String> iterator = strings.iterator();
        while (iterator.hasNext()){
            del(iterator.next());
        }
    }

    /**
     * 功能:设置key过期时间
     *
     * @param key     键
     * @param seconds 过期时间(单位:秒)
     * @return Boolean 是否设置过期时间异常
     * @throws Exception
     */
    public Long expire(String key, Integer seconds) {
        return jedisCluster.expire(key, seconds);
    }


    /**
     * 功能:设置key过期时间
     *
     * @param key     键
     * @param seconds 过期时间(单位:秒)
     * @return Boolean 是否设置过期时间异常
     * @throws Exception
     */
    public Long expire(byte[] key, Integer seconds) {
        return jedisCluster.expire(key, seconds);
    }


    /**
     * 功能:设置String类型值
     *
     * @param key   键
     * @param value 值
     * @return
     */
    public void set(String key, String value) {
        jedisCluster.set(key, value);
    }


    /**
     * 功能:设置String类型值
     *
     * @param key   键
     * @param value 值
     * @return
     */
    public void set(byte[] key, byte[] value) {
        jedisCluster.set(key, value);
    }


    /**
     * 功能:设置String类型值和过期时间
     *
     * @param key     键
     * @param seconds 时间(秒)
     * @param value   值
     * @return
     */
    public void setex(String key, int seconds, String value) {
        jedisCluster.setex(key, seconds, value);
    }


    /**
     * 功能:设置String类型值和过期时间
     *
     * @param key     键
     * @param seconds 时间(秒)
     * @param value   值
     * @return
     */
    public void setex(byte[] key, int seconds, byte[] value) {
        jedisCluster.setex(key, seconds, value);
    }


    /**
     * 功能:获取String类型值
     *
     * @param key 键
     * @return
     */
    public String get(String key) {
        return jedisCluster.get(key);
    }


    /**
     * 功能:获取String类型值
     *
     * @param key 键
     * @return
     */
    public byte[] get(byte[] key) {
        return jedisCluster.get(key);
    }


    /**
     * 功能:新增有序集合
     * 有序集合的成员是唯一的,但分数(score)却可以重复。
     * 集合是通过哈希表实现的，所以添加，删除，查找的复杂度都是O(1)。 集合中最大的成员数为 232 - 1 (4294967295, 每个集合可存储40多亿个成员)。
     *
     * @param key 键
     * @return
     */
    public void zadd(String key, double score, String value) {
        jedisCluster.zadd(key, score, value);
    }


    /**
     * 功能:新增有序集合
     * 有序集合的成员是唯一的,但分数(score)却可以重复。
     * 集合是通过哈希表实现的，所以添加，删除，查找的复杂度都是O(1)。 集合中最大的成员数为 232 - 1 (4294967295, 每个集合可存储40多亿个成员)。
     *
     * @param key 键
     * @return
     */
    public void zadd(byte[] key, double score, byte[] value) {
        jedisCluster.zadd(key, score, value);
    }


    /**
     * 功能：返回有序集中指定区间内的成员，通过索引，分数从高到底
     *
     * @param key   键
     * @param start 开始
     * @param end   结束   查所有为-1  差第一条为0
     * @return
     */
    public Set<String> zrevrange(String key, long start, long end) {
        return jedisCluster.zrevrange(key, start, end);
    }


    /**
     * 功能：返回有序集中指定区间内的成员，通过索引，分数从高到底
     *
     * @param key   键
     * @param start 开始
     * @param end   结束   查所有为-1  差第一条为0
     * @return
     */
    public Set<byte[]> zrevrange(byte[] key, long start, long end) {
        return jedisCluster.zrevrange(key, start, end);
    }


    /**
     * 功能：返回有序集中指定区间内的成员与分值，通过索引，分数从高到底
     *
     * @param key   键
     * @param start 开始
     * @param end   结束   查所有为-1  差第一条为0
     * @return
     */
    public Set<Tuple> zrevrangeWithScores(String key, long start, long end) {
        return jedisCluster.zrevrangeWithScores(key, start, end);
    }


    /**
     * 功能：返回有序集中指定区间内的成员与分值，通过索引，分数从高到底
     *
     * @param key   键
     * @param start 开始
     * @param end   结束   查所有为-1  差第一条为0
     * @return
     */
    public Set<Tuple> zrevrangeWithScores(byte[] key, long start, long end) {
        return jedisCluster.zrevrangeWithScores(key, start, end);
    }


    /**
     * 添加无序集合数据
     */
    public void sadd(String key, String... value) {
        jedisCluster.sadd(key, value);
    }


    /**
     * 添加无序集合数据
     */
    public void sadd(byte[] key, byte[]... value) {
        jedisCluster.sadd(key, value);
    }


    /**
     * 移除集合中一个或多个成员
     */
    public Long srem(String key, String... value) {
        return jedisCluster.srem(key, value);
    }


    /**
     * 移除集合中一个或多个成员
     */
    public Long srem(byte[] key, byte[]... value) {
        return jedisCluster.srem(key, value);
    }


    /**
     * 获取集合成员数
     */
    public Long scard(String key) {
        return jedisCluster.scard(key);
    }

    /**
     * 获取集合成员数
     */
    public Long scard(byte[] key) {
        return jedisCluster.scard(key);
    }


    /**
     * 功能:为队列头添加值
     *
     * @param key   队列名
     * @param value 值
     */
    public void rpush(String key, String... value) {
        jedisCluster.rpush(key, value);
    }


    /**
     * 功能:为队列头添加值
     *
     * @param key   队列名
     * @param value 值
     */
    public void rpush(byte[] key, byte[]... value) {
        jedisCluster.rpush(key, value);
    }


    /**
     * 功能:从队列尾取出值
     *
     * @param key 队列名
     * @return
     */
    public String rpop(String key) {
        return jedisCluster.rpop(key);
    }


    /**
     * 功能:从队列尾取出值
     *
     * @param key 队列名
     * @return
     */
    public byte[] rpop(byte[] key) {
        return jedisCluster.rpop(key);
    }


    /**
     * 功能:判断SessionId是否存在缓存服务器中 参数:
     *
     * @param key sessionId
     * @return Boolean true:存在 false:不存在
     * @throws Exception
     */
    public Boolean exists(String key) {
        return jedisCluster.exists(key);
    }


    /**
     * 功能:判断SessionId是否存在缓存服务器中 参数:
     *
     * @param key sessionId
     * @return Boolean true:存在 false:不存在
     * @throws Exception
     */
    public Boolean exists(byte[] key) {
        return jedisCluster.exists(key);
    }


    /**
     * 功能:判断SessionId是否存在缓存服务器中,存在则删除
     *
     * @param key sessionId
     * @return Boolean true:删除 false:不存在
     * @throws Exception
     */
    public Long del(String key) {
        if (exists(key)) {
            return jedisCluster.del(key);
        }
        return (long) -1;
    }


    /**
     * 功能:判断SessionId是否存在缓存服务器中,存在则删除
     *
     * @param key sessionId
     * @throws Exception
     */
    public Long del(byte[] key) {
        if (exists(key)) {
            return jedisCluster.del(key);
        }
        return (long) -1;
    }


    public Long incr(String key) {
        return jedisCluster.incr(key);
    }


    public Long incr(byte[] key) {
        return jedisCluster.incr(key);
    }


    /**
     * 功能:将数据存到队列中:
     *
     * @param key 队列名
     * @return
     */
    public void lpush(byte[] key, byte[] value) {
        jedisCluster.lpush(key, value);
    }

    /**
     * 功能:将数据存到队列中:
     *
     * @param key 队列名
     * @return
     */
    public void lpush(String key, String value) {
        jedisCluster.lpush(key, value);
    }

    /**
     * 功能:获取队列中第一条:
     *
     * @param key 队列名
     * @return
     */
    public byte[] lpop(byte[] key) {
        return jedisCluster.lpop(key);
    }

    /**
     * 功能:获取队列中第一条:
     *
     * @param key 队列名
     * @return
     */
    public String lpop(String key) {
        return jedisCluster.lpop(key);
    }


    public Long llen(byte[] key) {
        return jedisCluster.llen(key);
    }

    public Long llen(String key) {
        return jedisCluster.llen(key);
    }


    /**
     * 获取所有的Lists
     *
     * @param key
     * @return
     */
    public List<String> lrange(String key) {
        return jedisCluster.lrange(key, 0, llen(key));
    }


    /**
     * 获取所有的Lists
     *
     * @param key
     * @return
     */
    public List<byte[]> lrange(byte[] key) {
        return jedisCluster.lrange(key, 0, llen(key));
    }


    public Long hset(String key, String field, String value) {
        return jedisCluster.hset(key, field, value);
    }


    public Long hset(byte[] key, byte[] field, byte[] value) {
        return jedisCluster.hset(key, field, value);
    }


    public boolean hexists(String key, String field) {
        return jedisCluster.hexists(key, field);
    }


    public boolean hexists(byte[] key, byte[] field) {
        return jedisCluster.hexists(key, field);
    }


    public Map<String, String> hgetAll(String key) {
        return jedisCluster.hgetAll(key);
    }


    public Map<byte[], byte[]> hgetAll(byte[] key) {
        return jedisCluster.hgetAll(key);
    }


    public String hget(String key, String field) {
        return jedisCluster.hget(key, field);
    }


    public byte[] hget(byte[] key, byte[] field) {
        return jedisCluster.hget(key, field);
    }


    public Long hdel(String key, String field) {
        return jedisCluster.hdel(key, field);
    }


    public Long hdel(byte[] key, byte[] field) {
        return jedisCluster.hdel(key, field);
    }

}