package com.luues.redis.single.aspect;

import com.luues.exception.ExceptionRediscacheKeyInvalid;
import com.luues.exception.ExceptionRedissynchroizedInvalid;
import com.luues.redis.single.cache.RedisCache;
import com.luues.redis.single.cache.RedisCacheEvict;
import com.luues.redis.single.cache.RedisCachePut;
import com.luues.redis.single.lock.RedisSynchroized;
import com.luues.redis.single.service.JedisTemplate;
import com.luues.redis.util.CommontUtil;
import com.luues.util.encryption.SerializingUtil;
import com.luues.util.TypeConvert;
import com.luues.util.logs.LogUtil;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import java.lang.reflect.Method;
import java.util.Iterator;

/**
 * 通过aspect实现注解Rediscache
 * redis实现注解缓存
 */
@Aspect
@Component
public class RedisCacheAspect {
    @Autowired
    private JedisTemplate jedisTemplate;

    @Pointcut("@annotation(com.luues.redis.single.cache.RedisCache)")
    public void cacheAspect() {
    }
    @Around("cacheAspect()")
    public Object cacheAspect(ProceedingJoinPoint joinPoint) throws Throwable {
        Method objMethod = ((MethodSignature) joinPoint.getSignature()).getMethod();
        RedisCache rediscache = objMethod.getAnnotation(RedisCache.class);
        String key = null;
        Integer expire = null;
        String condition = null;
        if (rediscache != null) {
            key = rediscache.key();
            expire = rediscache.expire();
            condition = rediscache.condition();
        }
        String an = "@RedisCache";
        if (TypeConvert.isNull(key)) {
            throw new ExceptionRediscacheKeyInvalid(an + " key() is not null!");
        }
        if (TypeConvert.isNull(expire)) {
            expire = -1;
        }
        Object[] objects = joinPoint.getArgs();
        String condition_ = "true";
        if (!TypeConvert.isNull(condition)) {
            condition_ = CommontUtil.getConditionOrKey(condition, objMethod, objects, an);
            condition_ = CommontUtil.valiCondition(condition_, an);
        }
        if (condition_.equals("true")) {
            String key_ = CommontUtil.getConditionOrKey(key, objMethod, objects, an);
            return CommontUtil.getRedisCacheInfo(jedisTemplate, key_, joinPoint, expire);
        }
        try {
            return joinPoint.proceed();
        } catch (Throwable throwable) {
            return null;
        }
    }


    @Pointcut("@annotation(com.luues.redis.single.cache.RedisCachePut)")
    public void cachePutAspect() {
    }
    @Around("cachePutAspect()")
    public Object cachePutAspect(ProceedingJoinPoint joinPoint) throws Throwable {
        Method objMethod = ((MethodSignature) joinPoint.getSignature()).getMethod();
        RedisCachePut rediscachePut = objMethod.getAnnotation(RedisCachePut.class);
        String key = null;
        Integer expire = null;
        String condition = null;
        if (rediscachePut != null) {
            key = rediscachePut.key();
            expire = rediscachePut.expire();
            condition = rediscachePut.condition();
        }
        String an = "@RedisCachePut";
        if (TypeConvert.isNull(key)) {
            throw new ExceptionRediscacheKeyInvalid(an + " key() is not null!");
        }
        if (TypeConvert.isNull(expire)) {
            expire = -1;
        }
        Object[] objects = joinPoint.getArgs();
        String condition_ = "true";
        if (!TypeConvert.isNull(condition)) {
            condition_ = CommontUtil.getConditionOrKey(condition, objMethod, objects, an);
            condition_ = CommontUtil.valiCondition(condition_, an);
        }
        if (condition_.equals("true")) {
            String key_ = CommontUtil.getConditionOrKey(key, objMethod, objects, an);
            Object object = joinPoint.proceed();
            if (expire >= 0) {
                jedisTemplate.setex(key_.getBytes(), expire, SerializingUtil.serialize(object));
            } else {
                jedisTemplate.set(key_.getBytes(), SerializingUtil.serialize(object));
            }
            LogUtil.debug("\n{\n{　　　　　The return data of this method has been updated by redis}\n}");
            return object;
        }
        try {
            return joinPoint.proceed();
        } catch (Throwable throwable) {
            return null;
        }
    }


    @Pointcut("@annotation(com.luues.redis.single.cache.RedisCacheEvict)")
    public void cacheEvictAspect() {
    }
    @Around("cacheEvictAspect()")
    public Object cacheEvictAspect(ProceedingJoinPoint joinPoint) throws Throwable {
        Method objMethod = ((MethodSignature) joinPoint.getSignature()).getMethod();
        RedisCacheEvict rediscacheEvict = objMethod.getAnnotation(RedisCacheEvict.class);
        String[] keys = null;
        String condition = null;
        if (rediscacheEvict != null) {
            keys = rediscacheEvict.key();
            condition = rediscacheEvict.condition();
        }
        String an = "@RedisCacheEvict";
        if (TypeConvert.isNull(keys)) {
            throw new ExceptionRediscacheKeyInvalid(an + " key() is not null!");
        }
        Object[] objects = joinPoint.getArgs();
        String condition_ = "true";
        if (!TypeConvert.isNull(condition)) {
            condition_ = CommontUtil.getConditionOrKey(condition, objMethod, objects, an);
            condition_ = CommontUtil.valiCondition(condition_, an);
        }
        if (condition_.equals("true")) {
            for(String key : keys){
                String key_ = "";
                if (key.contains("#{*}")) {
                    try{
                        String k = key.split("#\\{\\*\\}")[0];
                        Iterator<String> iterator = jedisTemplate.keys(k + "*").iterator();
                        while (iterator.hasNext()){
                            jedisTemplate.del(iterator.next());
                        }
                        continue;
                    }catch (Exception e){
                        throw new ExceptionRediscacheKeyInvalid(an + " key() is error!");
                    }
                } else if(key.contains("#{")){
                    key_ = CommontUtil.getConditionOrKey(key, objMethod, objects, an);
                } else {
                    key_ = key;
                }
                if(jedisTemplate.exists(key_.getBytes())){
                    jedisTemplate.del(key_.getBytes());
                }
            }
            Object object = joinPoint.proceed();
            LogUtil.debug("\n{\n{　　　　　The redis data cached by this method has been emptied}\n}");
            return object;
        }
        try {
            return joinPoint.proceed();
        } catch (Throwable throwable) {
            return null;
        }
    }


    @Pointcut("@annotation(com.luues.redis.single.lock.RedisSynchroized)")
    public void synchroizedAspect() {
    }
    @Around("synchroizedAspect()")
    public Object synchroizedAspect(ProceedingJoinPoint joinPoint) throws Throwable {
        Method objMethod = ((MethodSignature) joinPoint.getSignature()).getMethod();
        RedisSynchroized redissynchroized = objMethod.getAnnotation(RedisSynchroized.class);
        String key = null;
        Integer expire = null;
        String condition = null;
        Integer time = null;
        if (redissynchroized != null) {
            key = redissynchroized.key();
            expire = redissynchroized.expire();
            condition = redissynchroized.condition();
            time = redissynchroized.time();
        }
        String an = "@RedisSynchroized";
        if (TypeConvert.isNull(key)) {
            throw new ExceptionRediscacheKeyInvalid(an + " key() is not null!");
        }
        if (TypeConvert.isNull(expire)) {
            expire = -1;
        }
        Object[] objects = joinPoint.getArgs();
        String condition_ = "true";
        if (!TypeConvert.isNull(condition)) {
            condition_ = CommontUtil.getConditionOrKey(condition, objMethod, objects, an);
            condition_ = CommontUtil.valiCondition(condition_, an);
        }
        if (condition_.equals("true")) {
            String key_ = CommontUtil.getConditionOrKey(key, objMethod, objects, an);
            synchronized (key_.intern()) {
                byte[] data = jedisTemplate.lpop(key_.getBytes());
                if (!TypeConvert.isNull(data)) {
                    if(time > 0){
                        jedisTemplate.lpush(key_.getBytes(), String.valueOf(System.currentTimeMillis()).getBytes());
                    }else{
                        jedisTemplate.lpush(key_.getBytes(), "SUCCESS".getBytes());
                    }
                    jedisTemplate.expire(key_.getBytes(), expire);
                    if (new String(data, "utf-8").equals("SUCCESS")) {
                        throw new ExceptionRedissynchroizedInvalid("频繁请求!");
                    } else {
                        Long time_ = (System.currentTimeMillis() - Long.valueOf(new String(data, "utf-8"))) / 1000;
                        if (time_ <= time) {
                            throw new ExceptionRedissynchroizedInvalid("您的手速太快了,请过"+time+"秒后再试!");
                        }
                    }
                }else{
                    if (time > 0) {
                        jedisTemplate.lpush(key_.getBytes(), String.valueOf(System.currentTimeMillis()).getBytes());
                    } else {
                        jedisTemplate.lpush(key_.getBytes(), "SUCCESS".getBytes());
                    }
                    jedisTemplate.expire(key_.getBytes(), expire);
                    if (time <= 0) {
                        jedisTemplate.lpop(key_.getBytes());
                    }
                }
                return joinPoint.proceed();
            }
        }
        return joinPoint.proceed();
    }

}
