package com.luues.redis.single.config;

import com.luues.exception.ExceptionRedisInvalid;
import com.luues.util.TypeConvert;
import com.luues.util.logs.LogUtil;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.cache.annotation.CachingConfigurerSupport;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.serializer.*;
import redis.clients.jedis.*;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

/**
 * redis配置类
 *
 * @author Mr-Wu ON 2019/05/07
 **/
@Configuration
@EnableCaching//开启注解
@ConditionalOnExpression(value = "!'null'.equals('${redis.hostName:null}')")
public class RedisConfig extends CachingConfigurerSupport {
    @Value("${redis.maxIdle:300}")
    private Integer maxIdle;
    @Value("${redis.maxTotal:1000}")
    private Integer maxTotal;
    @Value("${redis.maxWaitMillis:1000}")
    private Integer maxWaitMillis;
    @Value("${redis.minEvictableIdleTimeMillis:1800000}")
    private Integer minEvictableIdleTimeMillis;
    @Value("${redis.numTestsPerEvictionRun:3}")
    private Integer numTestsPerEvictionRun;
    @Value("${redis.timeBetweenEvictionRunsMillis:-1}")
    private long timeBetweenEvictionRunsMillis;
    @Value("${redis.testOnBorrow:true}")
    private boolean testOnBorrow;
    @Value("${redis.testOnReturn:true}")
    private boolean testOnReturn;
    @Value("${redis.testWhileIdle:true}")
    private boolean testWhileIdle;
    @Value("${redis.hostName:null}")
    private String hostName;
    @Value("${redis.port:0}")
    private Integer port;
    @Value("${redis.timeout:2000}")
    private Integer timeout;
    @Value("${redis.password:null}")
    private String password;

    @Bean
    public JedisPoolConfig jedisPoolConfig() throws ExceptionRedisInvalid {
        JedisPoolConfig jedisPoolConfig = new JedisPoolConfig();
        // 最大空闲数
        jedisPoolConfig.setMaxIdle(maxIdle);
        // 连接池的最大数据库连接数
        jedisPoolConfig.setMaxTotal(maxTotal);
        // 最大建立连接等待时间
        jedisPoolConfig.setMaxWaitMillis(maxWaitMillis);
        // 逐出连接的最小空闲时间 默认1800000毫秒(30分钟)
        jedisPoolConfig.setMinEvictableIdleTimeMillis(minEvictableIdleTimeMillis);
        // 每次逐出检查时 逐出的最大数目 如果为负数就是 : 1/abs(n), 默认3
        jedisPoolConfig.setNumTestsPerEvictionRun(numTestsPerEvictionRun);
        // 逐出扫描的时间间隔(毫秒) 如果为负数,则不运行逐出线程, 默认-1
        jedisPoolConfig.setTimeBetweenEvictionRunsMillis(timeBetweenEvictionRunsMillis);
        // 是否在从池中取出连接前进行检验,如果检验失败,则从池中去除连接并尝试取出另一个
        jedisPoolConfig.setTestOnBorrow(testOnBorrow);
        // 在return一个jedis实例时，是否提前进行validate操作
        jedisPoolConfig.setTestOnReturn(testOnReturn);
        // 在空闲时检查有效性, 默认false
        jedisPoolConfig.setTestWhileIdle(testWhileIdle);
        if(TypeConvert.isNull(hostName) || port <= 0){
            LogUtil.error("\n{\n　　　　　{}\n}\n", "redis start failed, because no host, port, etc. information is configured\n" +
                    "　　　　　See application.properties below jar for details");
            throw new ExceptionRedisInvalid("redis start failed, because no host, port, etc. information is configured");
        }
        LogUtil.info("\n{\n　　　　　redis start...\n　　　　　hostName:{}\n　　　　　port:{}\n　　　　　password:{}" +
                        "\n　　　　　MaxTotal:{}\n　　　　　MaxIdle:{}\n　　　　　MaxWaitMillis:{}" +
                        "\n　　　　　TestOnBorrow:{}\n　　　　　testOnReturn:{}\n　　　　　" +
                        "minEvictableIdleTimeMillis:{}\n" +
                        "　　　　　numTestsPerEvictionRun:{}\n　　　　　timeBetweenEvictionRunsMillis:{}\n" +
                        "　　　　　testWhileIdle:{}\n}",
                hostName, port, password, maxTotal, maxIdle, maxWaitMillis, testOnBorrow, testOnReturn,
                minEvictableIdleTimeMillis, numTestsPerEvictionRun, timeBetweenEvictionRunsMillis,
                testWhileIdle);
        return jedisPoolConfig;
    }

    @Bean
    @Qualifier(value = "jedisShardInfos")
    public List<JedisShardInfo> jedisShardInfos(){
        List<JedisShardInfo> shards = new ArrayList<JedisShardInfo>();
        JedisShardInfo jedisShardInfo = new JedisShardInfo(hostName, port);
        if(!TypeConvert.isNull(password))
            jedisShardInfo.setPassword(password);
        shards.add(jedisShardInfo);
        return shards;
    }

    @Bean
    public ShardedJedisPool shardedJedisPool(JedisPoolConfig jedisPoolConfig, @Qualifier(value = "jedisShardInfos") List<JedisShardInfo> jedisShardInfos){
        ShardedJedisPool shardedJedisPool = new ShardedJedisPool(jedisPoolConfig, jedisShardInfos);
        return shardedJedisPool;
    }

    @Bean
    public JedisPool jedisPool(JedisPoolConfig jedisPoolConfig){
        JedisPool jedisPool = null;
        if(!TypeConvert.isNull(password)){
            jedisPool = new JedisPool(jedisPoolConfig, hostName, port, timeout, password);
        }else{
            jedisPool = new JedisPool(jedisPoolConfig, hostName, port);
        }
        return jedisPool;
    }

    @Bean
    public RedisConnectionFactory redisConnectionFactory(JedisPoolConfig jedisPoolConfig){
        JedisConnectionFactory jedisConnectionFactory = new JedisConnectionFactory(jedisPoolConfig);
        jedisConnectionFactory.setHostName(hostName);
        jedisConnectionFactory.setTimeout(timeout);
        jedisConnectionFactory.setPort(port);
        if(!TypeConvert.isNull(password))
            jedisConnectionFactory.setPassword(password);
        return jedisConnectionFactory;
    }

    @Bean
    public RedisCacheManager cacheManager(RedisConnectionFactory redisConnectionFactory) {
        RedisCacheConfiguration config = RedisCacheConfiguration.defaultCacheConfig()
                .entryTtl(Duration.ofMinutes(30))
                .serializeKeysWith(RedisSerializationContext.SerializationPair.fromSerializer(keySerializer()))
                .serializeValuesWith(RedisSerializationContext.SerializationPair.fromSerializer(valueSerializer()))
                .disableCachingNullValues();
        RedisCacheManager redisCacheManager = RedisCacheManager.builder(redisConnectionFactory)
                .cacheDefaults(config)
                .transactionAware()
                .build();
        return redisCacheManager;
    }

    @Bean
    public RedisTemplate<String,Object> redisTemplate(RedisConnectionFactory redisConnectionFactory){
        RedisTemplate<String,Object> redisTemplate = new RedisTemplate<>();
        redisTemplate.setConnectionFactory(redisConnectionFactory);
        redisTemplate.setKeySerializer(keySerializer());
        redisTemplate.setHashKeySerializer(keySerializer());
        redisTemplate.setValueSerializer(valueSerializer());
        redisTemplate.setHashValueSerializer(valueSerializer());
        return redisTemplate;
    }

    @Bean
    public StringRedisTemplate stringRedisTemplate(RedisConnectionFactory redisConnectionFactory) {
        StringRedisTemplate stringRedisTemplate = new StringRedisTemplate();
        stringRedisTemplate.setConnectionFactory(redisConnectionFactory);
        stringRedisTemplate.setKeySerializer(keySerializer());
        return stringRedisTemplate;
    }


    private RedisSerializer<String> keySerializer() {
        return new StringRedisSerializer();
    }

    private RedisSerializer<Object> valueSerializer() {
        return new GenericJackson2JsonRedisSerializer();
    }
}