package com.luues.redis.single.service;

import com.luues.util.logs.LogUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import redis.clients.jedis.*;
import java.util.*;

/**
 * Redis
 *
 * @author Mr-Wu
 * @version 1.0
 */
@Component
public class JedisTemplate {

    @Autowired(required = false)
    private ShardedJedisPool shardedJedisPool;
    @Autowired(required = false)
    private JedisPool jedisPool;

    public JedisPool getJedisPool(){
        return jedisPool;
    }

    public ShardedJedisPool getShardedJedisPool() {
        return shardedJedisPool;
    }

    public Jedis getJedis() {
        if (null == jedisPool) {
            LogUtil.error("{}", "single-redis is null,需要使用请先配置！");
        }
        return jedisPool.getResource();
    }

    public ShardedJedis getShardedJedis() {
        if (null == shardedJedisPool) {
            LogUtil.error("{}", "single-redis is null,需要使用请先配置！");
        }
        return shardedJedisPool.getResource();
    }

    public Set<String> keys(String pattern) {
        Set<String> stringSet = new HashSet<String>();
        String cursor = ScanParams.SCAN_POINTER_START;
        ScanParams scanParams = new ScanParams();
        scanParams.match(pattern);
        scanParams.count(1000);
        while (true) {
            ScanResult<String> sscanResult = scan(cursor, scanParams);
            cursor = sscanResult.getStringCursor();// 返回0 说明遍历完成
            stringSet.addAll(sscanResult.getResult());
            if ("0".equals(cursor)) {
                break;
            }
        }
        return stringSet;
    }

    public void delKeys(String pattern) {
        Set<String> strings = keys(pattern);
        if (strings.size() == 0) {
            return;
        }
        Iterator<String> iterator = strings.iterator();
        while (iterator.hasNext()){
            del(iterator.next());
        }
    }

    public ScanResult<String> scan(String cursor, ScanParams scanParams){
        Jedis jedis = getJedis();
        try {
            return jedis.scan(cursor, scanParams);
        } finally {
            jedis.close();
        }
    }

    /**
     * 功能:设置key过期时间
     *
     * @param key     键
     * @param seconds 过期时间(单位:秒)
     * @return Boolean 是否设置过期时间异常
     * @throws Exception
     */
    public Long expire(String key, Integer seconds) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.expire(key, seconds);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:设置key过期时间
     *
     * @param key     键
     * @param seconds 过期时间(单位:秒)
     * @return Boolean 是否设置过期时间异常
     * @throws Exception
     */
    public Long expire(byte[] key, Integer seconds) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.expire(key, seconds);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:设置String类型值
     *
     * @param key   键
     * @param value 值
     * @return
     */
    public void set(String key, String value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.set(key, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:设置String类型值
     *
     * @param key   键
     * @param value 值
     * @return
     */
    public void set(byte[] key, byte[] value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.set(key, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:设置String类型值和过期时间
     *
     * @param key     键
     * @param seconds 时间(秒)
     * @param value   值
     * @return
     */
    public void setex(String key, int seconds, String value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.setex(key, seconds, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:设置String类型值和过期时间
     *
     * @param key     键
     * @param seconds 时间(秒)
     * @param value   值
     * @return
     */
    public void setex(byte[] key, int seconds, byte[] value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.setex(key, seconds, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:获取String类型值
     *
     * @param key 键
     * @return
     */
    public String get(String key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.get(key);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:获取String类型值
     *
     * @param key 键
     * @return
     */
    public byte[] get(byte[] key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.get(key);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:新增有序集合
     * 有序集合的成员是唯一的,但分数(score)却可以重复。
     * 集合是通过哈希表实现的，所以添加，删除，查找的复杂度都是O(1)。 集合中最大的成员数为 232 - 1 (4294967295, 每个集合可存储40多亿个成员)。
     *
     * @param key 键
     * @return
     */
    public void zadd(String key, double score, String value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.zadd(key, score, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:新增有序集合
     * 有序集合的成员是唯一的,但分数(score)却可以重复。
     * 集合是通过哈希表实现的，所以添加，删除，查找的复杂度都是O(1)。 集合中最大的成员数为 232 - 1 (4294967295, 每个集合可存储40多亿个成员)。
     *
     * @param key 键
     * @return
     */
    public void zadd(byte[] key, double score, byte[] value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.zadd(key, score, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能：返回有序集中指定区间内的成员，通过索引，分数从高到底
     *
     * @param key   键
     * @param start 开始
     * @param end   结束   查所有为-1  差第一条为0
     * @return
     */
    public Set<String> zrevrange(String key, long start, long end) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.zrevrange(key, start, end);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能：返回有序集中指定区间内的成员，通过索引，分数从高到底
     *
     * @param key   键
     * @param start 开始
     * @param end   结束   查所有为-1  差第一条为0
     * @return
     */
    public Set<byte[]> zrevrange(byte[] key, long start, long end) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.zrevrange(key, start, end);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能：返回有序集中指定区间内的成员与分值，通过索引，分数从高到底
     *
     * @param key   键
     * @param start 开始
     * @param end   结束   查所有为-1  差第一条为0
     * @return
     */
    public Set<Tuple> zrevrangeWithScores(String key, long start, long end) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.zrevrangeWithScores(key, start, end);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能：返回有序集中指定区间内的成员与分值，通过索引，分数从高到底
     *
     * @param key   键
     * @param start 开始
     * @param end   结束   查所有为-1  差第一条为0
     * @return
     */
    public Set<Tuple> zrevrangeWithScores(byte[] key, long start, long end) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.zrevrangeWithScores(key, start, end);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 添加无序集合数据
     */
    public void sadd(String key, String... value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.sadd(key, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 添加无序集合数据
     */
    public void sadd(byte[] key, byte[]... value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.sadd(key, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 移除集合中一个或多个成员
     */
    public Long srem(String key, String... value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.srem(key, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 移除集合中一个或多个成员
     */
    public Long srem(byte[] key, byte[]... value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.srem(key, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 获取集合成员数
     */
    public Long scard(String key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.scard(key);
        } finally {
            shardedJedis.close();
        }
    }

    /**
     * 获取集合成员数
     */
    public Long scard(byte[] key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.scard(key);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:为队列头添加值
     *
     * @param key   队列名
     * @param value 值
     */
    public void rpush(String key, String... value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.rpush(key, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:为队列头添加值
     *
     * @param key   队列名
     * @param value 值
     */
    public void rpush(byte[] key, byte[]... value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.rpush(key, value);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:从队列尾取出值
     *
     * @param key 队列名
     * @return
     */
    public String rpop(String key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.rpop(key);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:从队列尾取出值
     *
     * @param key 队列名
     * @return
     */
    public byte[] rpop(byte[] key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.rpop(key);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:判断SessionId是否存在缓存服务器中 参数:
     *
     * @param key sessionId
     * @return Boolean true:存在 false:不存在
     * @throws Exception
     */
    public Boolean exists(String key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.exists(key);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:判断SessionId是否存在缓存服务器中 参数:
     *
     * @param key sessionId
     * @return Boolean true:存在 false:不存在
     * @throws Exception
     */
    public Boolean exists(byte[] key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.exists(key);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:判断SessionId是否存在缓存服务器中,存在则删除
     *
     * @param key sessionId
     * @return Boolean true:删除 false:不存在
     * @throws Exception
     */
    public Long del(String key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            if (exists(key)) {
                return shardedJedis.del(key);
            }
            return (long) -1;
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:判断SessionId是否存在缓存服务器中,存在则删除
     *
     * @param key sessionId
     * @throws Exception
     */
    public Long del(byte[] key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            if (exists(key)) {
                return shardedJedis.del(key);
            }
            return (long) -1;
        } finally {
            shardedJedis.close();
        }
    }


    public Long incr(String key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.incr(key);
        } finally {
            shardedJedis.close();
        }
    }


    public Long incr(byte[] key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.incr(key);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 功能:将数据存到队列中:
     *
     * @param key 队列名
     * @return
     */
    public void lpush(byte[] key, byte[] value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.lpush(key, value);
        } finally {
            shardedJedis.close();
        }
    }

    /**
     * 功能:将数据存到队列中:
     *
     * @param key 队列名
     * @return
     */
    public void lpush(String key, String value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            shardedJedis.lpush(key, value);
        } finally {
            shardedJedis.close();
        }
    }

    /**
     * 功能:获取队列中第一条:
     *
     * @param key 队列名
     * @return
     */
    public byte[] lpop(byte[] key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.lpop(key);
        } finally {
            shardedJedis.close();
        }
    }

    /**
     * 功能:获取队列中第一条:
     *
     * @param key 队列名
     * @return
     */
    public String lpop(String key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.lpop(key);
        } finally {
            shardedJedis.close();
        }
    }


    public Long llen(byte[] key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.llen(key);
        } finally {
            shardedJedis.close();
        }
    }

    public Long llen(String key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.llen(key);
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 获取所有的Lists
     *
     * @param key
     * @return
     */
    public List<String> lrange(String key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.lrange(key, 0, llen(key));
        } finally {
            shardedJedis.close();
        }
    }


    /**
     * 获取所有的Lists
     *
     * @param key
     * @return
     */
    public List<byte[]> lrange(byte[] key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.lrange(key, 0, llen(key));
        } finally {
            shardedJedis.close();
        }
    }


    public Long hset(String key, String field, String value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.hset(key, field, value);
        } finally {
            shardedJedis.close();
        }
    }


    public Long hset(byte[] key, byte[] field, byte[] value) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.hset(key, field, value);
        } finally {
            shardedJedis.close();
        }
    }


    public boolean hexists(String key, String field) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.hexists(key, field);
        } finally {
            shardedJedis.close();
        }
    }


    public boolean hexists(byte[] key, byte[] field) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.hexists(key, field);
        } finally {
            shardedJedis.close();
        }
    }


    public Map<String, String> hgetAll(String key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.hgetAll(key);
        } finally {
            shardedJedis.close();
        }
    }


    public Map<byte[], byte[]> hgetAll(byte[] key) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.hgetAll(key);
        } finally {
            shardedJedis.close();
        }
    }


    public String hget(String key, String field) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.hget(key, field);
        } finally {
            shardedJedis.close();
        }
    }


    public byte[] hget(byte[] key, byte[] field) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.hget(key, field);
        } finally {
            shardedJedis.close();
        }
    }


    public Long hdel(String key, String field) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.hdel(key, field);
        } finally {
            shardedJedis.close();
        }
    }


    public Long hdel(byte[] key, byte[] field) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.hdel(key, field);
        } finally {
            shardedJedis.close();
        }
    }

    public ScanResult<String> sscan(String key, String cursor, ScanParams scanParams) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.sscan(key, cursor, scanParams);
        } finally {
            shardedJedis.close();
        }
    }

    public ScanResult<byte[]> sscan(byte[] key, byte[] cursor, ScanParams scanParams) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.sscan(key, cursor, scanParams);
        } finally {
            shardedJedis.close();
        }
    }

    public ScanResult<String> sscan(String key, String cursor) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.sscan(key, cursor);
        } finally {
            shardedJedis.close();
        }
    }

    public ScanResult<byte[]> sscan(byte[] key, byte[] cursor) {
        ShardedJedis shardedJedis = getShardedJedis();
        try {
            return shardedJedis.sscan(key, cursor);
        } finally {
            shardedJedis.close();
        }
    }

}