package cn.luues.tool.http.ssl;

import cn.luues.tool.core.net.SSLContextBuilder;

import javax.net.ssl.*;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.*;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

/**
 * SSLSocketFactory构建器
 *
 * @author Looly
 * @see SSLContextBuilder
 */
public class SSLSocketFactoryBuilder {

	/**
	 * Supports some version of SSL; may support other versions
	 */
	public static final String SSL = SSLContextBuilder.SSL;
	/**
	 * Supports SSL version 2 or later; may support other versions
	 */
	public static final String SSLv2 = SSLContextBuilder.SSLv2;
	/**
	 * Supports SSL version 3; may support other versions
	 */
	public static final String SSLv3 = SSLContextBuilder.SSLv3;

	/**
	 * Supports some version of TLS; may support other versions
	 */
	public static final String TLS = SSLContextBuilder.TLS;
	/**
	 * Supports RFC 2246: TLS version 1.0 ; may support other versions
	 */
	public static final String TLSv1 = SSLContextBuilder.TLSv1;
	/**
	 * Supports RFC 4346: TLS version 1.1 ; may support other versions
	 */
	public static final String TLSv11 = SSLContextBuilder.TLSv11;
	/**
	 * Supports RFC 5246: TLS version 1.2 ; may support other versions
	 */
	public static final String TLSv12 = SSLContextBuilder.TLSv12;

	SSLContextBuilder sslContextBuilder;

	/**
	 * 构造
	 */
	public SSLSocketFactoryBuilder() {
		this.sslContextBuilder = SSLContextBuilder.create();
	}

	/**
	 * 创建 SSLSocketFactoryBuilder
	 *
	 * @return SSLSocketFactoryBuilder
	 */
	public static SSLSocketFactoryBuilder create() {
		return new SSLSocketFactoryBuilder();
	}

	/**
	 * 创建 SSLSocketFactoryBuilder
	 * 默认的SSLSocketFactoryBuilder.create().build()无法请求https时候，使用该方式
	 * @param fileInputStream 证书文件流
	 * @param password 证书密码
	 * @return SSLSocketFactoryBuilder
	 */
	public static SSLSocketFactoryBuilder createTLS(FileInputStream fileInputStream, char[] password) throws KeyStoreException, CertificateException, NoSuchAlgorithmException, IOException, UnrecoverableKeyException, KeyManagementException {
		KeyStore keyStore = KeyStore.getInstance("PKCS12");
		keyStore.load(fileInputStream, password);//设置证书密码
		fileInputStream.close();
		KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
		keyManagerFactory.init(keyStore, password);
		return new SSLSocketFactoryBuilder()
				.setKeyManagers(keyManagerFactory.getKeyManagers())
				.setTrustManagers(null)
				.setSecureRandom(new SecureRandom());
	}

	/**
	 * 创建 SSLSocketFactoryBuilder
	 * 默认的SSLSocketFactoryBuilder.create().build()无法请求https时候，使用该方式
	 * @return SSLSocketFactoryBuilder
	 */
	public static SSLSocketFactoryBuilder createTLS() {
		TrustManager trustManager = new X509TrustManager() {
			@Override
			public void checkClientTrusted(X509Certificate[] x509Certificates, String s) {

			}
			@Override
			public void checkServerTrusted(X509Certificate[] x509Certificates, String s) {

			}
			@Override
			public X509Certificate[] getAcceptedIssuers() {
				return null;
			}
		};
		return new SSLSocketFactoryBuilder()
				.setKeyManagers(null)
				.setTrustManagers(new TrustManager[]{trustManager})
				.setSecureRandom(null);
	}

	/**
	 * 设置协议
	 *
	 * @param protocol 协议
	 * @return 自身
	 */
	public SSLSocketFactoryBuilder setProtocol(String protocol) {
		this.sslContextBuilder.setProtocol(protocol);
		return this;
	}

	/**
	 * 设置信任信息
	 *
	 * @param trustManagers TrustManager列表
	 * @return 自身
	 */
	public SSLSocketFactoryBuilder setTrustManagers(TrustManager... trustManagers) {
		this.sslContextBuilder.setTrustManagers(trustManagers);
		return this;
	}

	/**
	 * 设置 JSSE key managers
	 *
	 * @param keyManagers JSSE key managers
	 * @return 自身
	 */
	public SSLSocketFactoryBuilder setKeyManagers(KeyManager... keyManagers) {
		this.sslContextBuilder.setKeyManagers(keyManagers);
		return this;
	}

	/**
	 * 设置 SecureRandom
	 *
	 * @param secureRandom SecureRandom
	 * @return 自己
	 */
	public SSLSocketFactoryBuilder setSecureRandom(SecureRandom secureRandom) {
		this.sslContextBuilder.setSecureRandom(secureRandom);
		return this;
	}

	/**
	 * 构建SSLSocketFactory
	 *
	 * @return SSLSocketFactory
	 * @throws NoSuchAlgorithmException 无此算法
	 * @throws KeyManagementException   Key管理异常
	 */
	public SSLSocketFactory build() throws NoSuchAlgorithmException, KeyManagementException {
		return this.sslContextBuilder.build().getSocketFactory();
	}
}
