/*
 * Decompiled with CFR 0.152.
 */
package internal.org.springframework.content.elasticsearch;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import internal.org.springframework.content.elasticsearch.IndexManager;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.ingest.GetPipelineRequest;
import org.elasticsearch.action.ingest.GetPipelineResponse;
import org.elasticsearch.action.ingest.PutPipelineRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.rest.RestStatus;
import org.springframework.content.commons.annotations.ContentId;
import org.springframework.content.commons.annotations.MimeType;
import org.springframework.content.commons.renditions.RenditionService;
import org.springframework.content.commons.repository.StoreAccessException;
import org.springframework.content.commons.search.IndexService;
import org.springframework.content.commons.utils.BeanUtils;
import org.springframework.content.elasticsearch.AttributeProvider;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

@Service
public class ElasticsearchIndexServiceImpl<T>
implements IndexService<T> {
    private static final Log LOGGER = LogFactory.getLog(ElasticsearchIndexServiceImpl.class);
    private static final String SPRING_CONTENT_ATTACHMENT = "spring-content-attachment-pipeline";
    private static final int BUFFER_SIZE = 3072;
    private final RestHighLevelClient client;
    private final RenditionService renditionService;
    private final IndexManager manager;
    private final AttributeProvider attributeProvider;
    private final ObjectMapper objectMapper;
    private boolean pipelinedInitialized = false;

    public ElasticsearchIndexServiceImpl(RestHighLevelClient client, RenditionService renditionService, IndexManager manager, AttributeProvider attributeProvider) {
        this.client = client;
        this.renditionService = renditionService;
        this.manager = manager;
        this.attributeProvider = attributeProvider;
        this.objectMapper = new ObjectMapper();
    }

    public void index(T entity, InputStream stream) {
        String strMimeType;
        Object mimeType;
        if (!this.pipelinedInitialized) {
            try {
                this.ensureAttachmentPipeline();
            }
            catch (IOException ioe) {
                throw new StoreAccessException("Unable to initialize attachment pipeline", (Throwable)ioe);
            }
        }
        String id = BeanUtils.getFieldWithAnnotation(entity, ContentId.class).toString();
        if (this.renditionService != null && (mimeType = BeanUtils.getFieldWithAnnotation(entity, MimeType.class)) != null && this.renditionService.canConvert(strMimeType = mimeType.toString(), "text/plain")) {
            stream = this.renditionService.convert(strMimeType, stream, "text/plain");
        }
        StringBuilder result = new StringBuilder();
        try (BufferedInputStream in = new BufferedInputStream(stream, 3072);){
            Base64.Encoder encoder = Base64.getEncoder();
            byte[] chunk = new byte[3072];
            int len = 0;
            while ((len = in.read(chunk)) == 3072) {
                result.append(encoder.encodeToString(chunk));
            }
            if (len > 0) {
                chunk = Arrays.copyOf(chunk, len);
                result.append(encoder.encodeToString(chunk));
            }
        }
        catch (IOException e) {
            throw new StoreAccessException(String.format("Error base64 encoding stream for content %s", id), (Throwable)e);
        }
        IndexRequest req = new IndexRequest(this.manager.indexName(entity.getClass()), entity.getClass().getName(), id);
        req.setPipeline(SPRING_CONTENT_ATTACHMENT);
        Map<String, String> attributesToSync = new HashMap<String, String>();
        if (this.attributeProvider != null) {
            attributesToSync = this.attributeProvider.synchronize(entity);
        }
        attributesToSync.put("data", result.toString());
        String payload = "";
        try {
            payload = this.objectMapper.writeValueAsString(attributesToSync);
        }
        catch (JsonProcessingException e) {
            throw new StoreAccessException(String.format("Unable to serialize payload for content %s", id), (Throwable)e);
        }
        req.source(payload, XContentType.JSON);
        try {
            IndexResponse res = this.client.index(req, RequestOptions.DEFAULT);
            LOGGER.info((Object)String.format("Content '%s' indexed with result %s", id, res.getResult()));
        }
        catch (IOException e) {
            throw new StoreAccessException(String.format("Error indexing content %s", id), (Throwable)e);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void unindex(T entity) {
        Object id;
        if (!this.pipelinedInitialized) {
            try {
                this.ensureAttachmentPipeline();
            }
            catch (IOException ioe) {
                throw new StoreAccessException("Unable to initialize attachment pipeline", (Throwable)ioe);
            }
        }
        if ((id = BeanUtils.getFieldWithAnnotation(entity, ContentId.class)) == null) {
            return;
        }
        DeleteRequest req = new DeleteRequest(this.manager.indexName(entity.getClass()), entity.getClass().getName(), id.toString());
        try {
            DeleteResponse res = this.client.delete(req, RequestOptions.DEFAULT);
            LOGGER.info((Object)String.format("Indexed content '%s' deleted with result %s", id, res.getResult()));
            return;
        }
        catch (ElasticsearchStatusException ese) {
            if (ese.status() == RestStatus.NOT_FOUND) return;
        }
        catch (IOException e) {
            throw new StoreAccessException(String.format("Error deleting indexed content %s", id), (Throwable)e);
        }
    }

    void ensureAttachmentPipeline() throws IOException {
        GetPipelineRequest getRequest = new GetPipelineRequest(new String[]{SPRING_CONTENT_ATTACHMENT});
        GetPipelineResponse res = this.client.ingest().getPipeline(getRequest, RequestOptions.DEFAULT);
        if (!res.isFound()) {
            String source = "{\"description\":\"Extract attachment information encoded in Base64 with UTF-8 charset\",\"processors\":[{\"attachment\":{\"field\":\"data\"}}]}";
            PutPipelineRequest put = new PutPipelineRequest(SPRING_CONTENT_ATTACHMENT, (BytesReference)new BytesArray(source.getBytes(StandardCharsets.UTF_8)), XContentType.JSON);
            AcknowledgedResponse wpr = this.client.ingest().putPipeline(put, RequestOptions.DEFAULT);
            Assert.isTrue((boolean)wpr.isAcknowledged(), (String)"Attachment pipeline not acknowledged by server");
        }
    }
}

