/*
 * Decompiled with CFR 0.152.
 */
package com.github.robtimus.io.stream;

import com.github.robtimus.io.stream.ByteCaptor;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

public final class CapturingInputStream
extends InputStream {
    private final InputStream delegate;
    private final ByteCaptor captor;
    private final int limit;
    private final long doneAfter;
    private long totalBytes = 0L;
    private long mark = 0L;
    private boolean consumed = false;
    private boolean closed = false;
    private Consumer<? super CapturingInputStream> doneCallback;
    private Consumer<? super CapturingInputStream> limitReachedCallback;
    private final BiConsumer<? super CapturingInputStream, ? super IOException> errorCallback;

    public CapturingInputStream(InputStream input, Config config) {
        this.delegate = Objects.requireNonNull(input);
        this.captor = config.expectedCount < 0 ? new ByteCaptor() : new ByteCaptor(Math.min(config.expectedCount, config.limit));
        this.limit = config.limit;
        this.doneAfter = config.doneAfter;
        this.doneCallback = config.doneCallback;
        this.limitReachedCallback = config.limitReachedCallback;
        this.errorCallback = config.errorCallback;
    }

    @Override
    public int read() throws IOException {
        try {
            int b = this.delegate.read();
            if (b == -1) {
                this.markAsConsumed();
            } else {
                ++this.totalBytes;
                if (this.captor.size() < this.limit) {
                    this.captor.write(b);
                    this.checkLimitReached();
                }
                this.checkDone();
            }
            return b;
        }
        catch (IOException e) {
            this.onError(e);
            throw e;
        }
    }

    @Override
    public int read(byte[] b) throws IOException {
        try {
            int n = this.delegate.read(b);
            if (n == -1) {
                this.markAsConsumed();
            } else {
                this.totalBytes += (long)n;
                int allowed = Math.min(this.limit - this.captor.size(), n);
                if (allowed > 0) {
                    this.captor.write(b, 0, allowed);
                    this.checkLimitReached();
                }
                this.checkDone();
            }
            return n;
        }
        catch (IOException e) {
            this.onError(e);
            throw e;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        try {
            int n = this.delegate.read(b, off, len);
            if (n == -1) {
                this.markAsConsumed();
            } else {
                this.totalBytes += (long)n;
                int allowed = Math.min(this.limit - this.captor.size(), n);
                if (allowed > 0) {
                    this.captor.write(b, off, allowed);
                    this.checkLimitReached();
                }
                this.checkDone();
            }
            return n;
        }
        catch (IOException e) {
            this.onError(e);
            throw e;
        }
    }

    @Override
    public void close() throws IOException {
        try {
            this.delegate.close();
            this.markAsClosed();
        }
        catch (IOException e) {
            this.onError(e);
            throw e;
        }
    }

    @Override
    public synchronized void mark(int readlimit) {
        this.delegate.mark(readlimit);
        this.mark = this.totalBytes;
    }

    @Override
    public synchronized void reset() throws IOException {
        try {
            this.delegate.reset();
            this.captor.reset((int)Math.min(this.mark, (long)this.limit));
            this.totalBytes = this.mark;
            this.consumed = false;
        }
        catch (IOException e) {
            this.onError(e);
            throw e;
        }
    }

    @Override
    public boolean markSupported() {
        return this.delegate.markSupported();
    }

    private void markAsConsumed() {
        this.consumed = true;
        if (this.doneCallback != null) {
            this.doneCallback.accept(this);
            this.doneCallback = null;
        }
    }

    private void markAsClosed() {
        this.closed = true;
        if (this.doneCallback != null) {
            this.doneCallback.accept(this);
            this.doneCallback = null;
        }
    }

    private void checkLimitReached() {
        if (this.totalBytes >= (long)this.limit && this.limitReachedCallback != null) {
            this.limitReachedCallback.accept(this);
            this.limitReachedCallback = null;
        }
    }

    private void checkDone() {
        if (this.totalBytes >= this.doneAfter && this.doneCallback != null) {
            this.doneCallback.accept(this);
            this.doneCallback = null;
        }
    }

    private void onError(IOException error) {
        if (this.errorCallback != null) {
            this.errorCallback.accept(this, error);
        }
    }

    public byte[] captured() {
        return this.captor.toByteArray();
    }

    public String captured(Charset charset) {
        return this.captor.toString(charset);
    }

    public long totalBytes() {
        return this.totalBytes;
    }

    public boolean isConsumed() {
        return this.consumed;
    }

    public boolean isClosed() {
        return this.closed;
    }

    public static Config.Builder config() {
        return new Config.Builder();
    }

    public static final class Config {
        private final int limit;
        private final int expectedCount;
        private final long doneAfter;
        private final Consumer<? super CapturingInputStream> doneCallback;
        private final Consumer<? super CapturingInputStream> limitReachedCallback;
        private final BiConsumer<? super CapturingInputStream, ? super IOException> errorCallback;

        private Config(Builder builder) {
            this.limit = builder.limit;
            this.expectedCount = builder.expectedCount;
            this.doneAfter = builder.doneAfter;
            this.doneCallback = builder.doneCallback;
            this.limitReachedCallback = builder.limitReachedCallback;
            this.errorCallback = builder.errorCallback;
        }

        public static final class Builder {
            private int limit = Integer.MAX_VALUE;
            private int expectedCount = -1;
            private long doneAfter = Long.MAX_VALUE;
            private Consumer<? super CapturingInputStream> doneCallback;
            private Consumer<? super CapturingInputStream> limitReachedCallback;
            private BiConsumer<? super CapturingInputStream, ? super IOException> errorCallback;

            private Builder() {
            }

            public Builder withLimit(int limit) {
                if (limit < 0) {
                    throw new IllegalArgumentException(limit + " < 0");
                }
                this.limit = limit;
                return this;
            }

            public Builder withExpectedCount(int expectedCount) {
                this.expectedCount = expectedCount;
                return this;
            }

            public Builder doneAfter(long count) {
                if (count < 0L) {
                    throw new IllegalArgumentException(count + " < 0");
                }
                this.doneAfter = count;
                return this;
            }

            public Builder onDone(Consumer<? super CapturingInputStream> callback) {
                this.doneCallback = Objects.requireNonNull(callback);
                return this;
            }

            public Builder onLimitReached(Consumer<? super CapturingInputStream> callback) {
                this.limitReachedCallback = Objects.requireNonNull(callback);
                return this;
            }

            public Builder onError(BiConsumer<? super CapturingInputStream, ? super IOException> callback) {
                this.errorCallback = Objects.requireNonNull(callback);
                return this;
            }

            public Config build() {
                return new Config(this);
            }
        }
    }
}

