/*
 * Decompiled with CFR 0.152.
 */
package com.github.robtimus.io.stream;

import com.github.robtimus.io.stream.ByteCaptor;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

public final class CapturingOutputStream
extends OutputStream {
    private final OutputStream delegate;
    private final ByteCaptor captor;
    private final int limit;
    private long totalBytes = 0L;
    private boolean closed = false;
    private Consumer<? super CapturingOutputStream> doneCallback;
    private Consumer<? super CapturingOutputStream> limitReachedCallback;
    private final BiConsumer<? super CapturingOutputStream, ? super IOException> errorCallback;

    public CapturingOutputStream(OutputStream output, Config config) {
        this.delegate = Objects.requireNonNull(output);
        this.captor = config.expectedCount < 0 ? new ByteCaptor() : new ByteCaptor(Math.min(config.expectedCount, config.limit));
        this.limit = config.limit;
        this.doneCallback = config.doneCallback;
        this.limitReachedCallback = config.limitReachedCallback;
        this.errorCallback = config.errorCallback;
    }

    @Override
    public void write(int b) throws IOException {
        try {
            this.delegate.write(b);
            ++this.totalBytes;
            if (this.captor.size() < this.limit) {
                this.captor.write(b);
                this.checkLimitReached();
            }
        }
        catch (IOException e) {
            this.onError(e);
            throw e;
        }
    }

    @Override
    public void write(byte[] b) throws IOException {
        try {
            this.delegate.write(b);
            this.totalBytes += (long)b.length;
            int allowed = Math.min(this.limit - this.captor.size(), b.length);
            if (allowed > 0) {
                this.captor.write(b, 0, allowed);
                this.checkLimitReached();
            }
        }
        catch (IOException e) {
            this.onError(e);
            throw e;
        }
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        try {
            this.delegate.write(b, off, len);
            this.totalBytes += (long)len;
            int allowed = Math.min(this.limit - this.captor.size(), len);
            if (allowed > 0) {
                this.captor.write(b, off, allowed);
                this.checkLimitReached();
            }
        }
        catch (IOException e) {
            this.onError(e);
            throw e;
        }
    }

    @Override
    public void flush() throws IOException {
        try {
            this.delegate.flush();
        }
        catch (IOException e) {
            this.onError(e);
            throw e;
        }
    }

    @Override
    public void close() throws IOException {
        try {
            this.delegate.close();
            this.markAsClosed();
        }
        catch (IOException e) {
            this.onError(e);
            throw e;
        }
    }

    public void done() {
        if (this.doneCallback != null) {
            this.doneCallback.accept(this);
            this.doneCallback = null;
        }
    }

    private void markAsClosed() {
        this.closed = true;
        this.done();
    }

    private void checkLimitReached() {
        if (this.totalBytes >= (long)this.limit && this.limitReachedCallback != null) {
            this.limitReachedCallback.accept(this);
            this.limitReachedCallback = null;
        }
    }

    private void onError(IOException error) {
        if (this.errorCallback != null) {
            this.errorCallback.accept(this, error);
        }
    }

    public byte[] captured() {
        return this.captor.toByteArray();
    }

    public String captured(Charset charset) {
        return this.captor.toString(charset);
    }

    public long totalBytes() {
        return this.totalBytes;
    }

    public boolean isClosed() {
        return this.closed;
    }

    public static Config.Builder config() {
        return new Config.Builder();
    }

    public static final class Config {
        private final int limit;
        private final int expectedCount;
        private final Consumer<? super CapturingOutputStream> doneCallback;
        private final Consumer<? super CapturingOutputStream> limitReachedCallback;
        private final BiConsumer<? super CapturingOutputStream, ? super IOException> errorCallback;

        private Config(Builder builder) {
            this.limit = builder.limit;
            this.expectedCount = builder.expectedCount;
            this.doneCallback = builder.doneCallback;
            this.limitReachedCallback = builder.limitReachedCallback;
            this.errorCallback = builder.errorCallback;
        }

        public static final class Builder {
            private int limit = Integer.MAX_VALUE;
            private int expectedCount = -1;
            private Consumer<? super CapturingOutputStream> doneCallback;
            private Consumer<? super CapturingOutputStream> limitReachedCallback;
            private BiConsumer<? super CapturingOutputStream, ? super IOException> errorCallback;

            private Builder() {
            }

            public Builder withLimit(int limit) {
                if (limit < 0) {
                    throw new IllegalArgumentException(limit + " < 0");
                }
                this.limit = limit;
                return this;
            }

            public Builder withExpectedCount(int expectedCount) {
                this.expectedCount = expectedCount;
                return this;
            }

            public Builder onDone(Consumer<? super CapturingOutputStream> callback) {
                this.doneCallback = Objects.requireNonNull(callback);
                return this;
            }

            public Builder onLimitReached(Consumer<? super CapturingOutputStream> callback) {
                this.limitReachedCallback = Objects.requireNonNull(callback);
                return this;
            }

            public Builder onError(BiConsumer<? super CapturingOutputStream, ? super IOException> callback) {
                this.errorCallback = Objects.requireNonNull(callback);
                return this;
            }

            public Config build() {
                return new Config(this);
            }
        }
    }
}

