/*
 * Decompiled with CFR 0.152.
 */
package org.rx.core;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import lombok.NonNull;
import org.rx.core.Constants;
import org.rx.core.Disposable;
import org.rx.core.Extends;
import org.rx.core.IdentityWrapper;
import org.rx.core.Linq;
import org.rx.core.Reflects;
import org.rx.core.Tasks;
import org.rx.exception.InvalidException;
import org.rx.exception.TraceHandler;
import org.rx.util.function.BiAction;
import org.rx.util.function.Func;
import org.rx.util.function.PredicateFunc;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ObjectPool<T>
extends Disposable {
    private static final Logger log = LoggerFactory.getLogger(ObjectPool.class);
    final Func<T> createHandler;
    final PredicateFunc<T> validateHandler;
    final BiAction<T> passivateHandler;
    final ConcurrentLinkedDeque<IdentityWrapper<T>> stack = new ConcurrentLinkedDeque();
    final Map<IdentityWrapper<T>, ObjectConf> conf = new ConcurrentHashMap<IdentityWrapper<T>, ObjectConf>();
    final AtomicInteger size = new AtomicInteger();
    final int minSize;
    final int maxSize;
    long borrowTimeout = 30000L;
    long idleTimeout = 600000L;
    long validationTime = 5000L;
    long leakDetectionThreshold;

    public int size() {
        return this.size.get();
    }

    public void setBorrowTimeout(long borrowTimeout) {
        this.borrowTimeout = Math.max(250L, borrowTimeout);
    }

    public void setIdleTimeout(long idleTimeout) {
        this.idleTimeout = idleTimeout == 0L ? 0L : Math.max(10000L, idleTimeout);
    }

    public void setValidationTime(long validationTime) {
        this.validationTime = Math.max(250L, validationTime);
    }

    public void setLeakDetectionThreshold(long leakDetectionThreshold) {
        this.leakDetectionThreshold = Math.max(2000L, leakDetectionThreshold);
    }

    public ObjectPool(int minSize, Func<T> createHandler, PredicateFunc<T> validateHandler) {
        this(minSize, minSize, createHandler, validateHandler, null);
    }

    public ObjectPool(int minSize, int maxSize, @NonNull Func<T> createHandler, @NonNull PredicateFunc<T> validateHandler, BiAction<T> passivateHandler) {
        if (createHandler == null) {
            throw new NullPointerException("createHandler is marked non-null but is null");
        }
        if (validateHandler == null) {
            throw new NullPointerException("validateHandler is marked non-null but is null");
        }
        if (minSize < 0) {
            throw new InvalidException("MinSize '{}' must greater than or equal to 0", minSize);
        }
        if (maxSize < 1) {
            throw new InvalidException("MaxSize '{}' must greater than or equal to 1", maxSize);
        }
        this.minSize = minSize;
        this.maxSize = Math.max(minSize, maxSize);
        this.createHandler = createHandler;
        this.validateHandler = validateHandler;
        this.passivateHandler = passivateHandler;
        this.insureMinSize();
        Tasks.timer.setTimeout(this::validNow, d -> this.validationTime, (Object)this, Constants.TIMER_PERIOD_FLAG);
    }

    @Override
    protected void freeObjects() {
        for (IdentityWrapper<T> wrapper : this.stack) {
            this.doRetire(wrapper, 0);
        }
    }

    void insureMinSize() {
        for (int i = this.size(); i < this.minSize; ++i) {
            Extends.quietly(this::doCreate);
        }
    }

    void validNow() {
        Extends.eachQuietly(Linq.from(this.conf.entrySet()).orderBy(p -> ((ObjectConf)p.getValue()).stateTime), p -> {
            IdentityWrapper wrapper = (IdentityWrapper)p.getKey();
            ObjectConf c = (ObjectConf)p.getValue();
            if (!this.validateHandler.test(wrapper.instance) || this.size() > this.minSize && c.isIdleTimeout(this.idleTimeout)) {
                this.doRetire(wrapper, 3);
                return;
            }
            if (c.isLeaked(this.leakDetectionThreshold)) {
                TraceHandler.INSTANCE.saveMetric(Constants.MetricName.OBJECT_POOL_LEAK.name(), String.format("Pool %s owned Object '%s' leaked.\n%s", this, wrapper, Reflects.getStackTrace(c.t)));
                this.doRetire(wrapper, 4);
            }
        });
        this.insureMinSize();
    }

    IdentityWrapper<T> doCreate() {
        if (this.size() > this.maxSize) {
            log.warn("ObjPool reject: Reach the maximum");
            return null;
        }
        IdentityWrapper<T> wrapper = new IdentityWrapper<T>(this.createHandler.get());
        if (!this.stack.offer(wrapper)) {
            log.error("ObjPool create object fail: Offer stack fail");
            return null;
        }
        ObjectConf c = new ObjectConf();
        c.setBorrowed(true);
        if (this.conf.putIfAbsent(wrapper, c) != null) {
            throw new InvalidException("create object fail, object '{}' has already in this pool", wrapper);
        }
        this.size.incrementAndGet();
        if (this.passivateHandler != null) {
            this.passivateHandler.accept(wrapper.instance);
        }
        return wrapper;
    }

    boolean doRetire(IdentityWrapper<T> wrapper, int action) {
        boolean ok = this.stack.remove(wrapper);
        ObjectConf c = this.conf.remove(wrapper);
        if (c != null) {
            this.size.decrementAndGet();
            if (!c.isBorrowed()) {
                Extends.tryClose(wrapper);
            }
        }
        log.debug("ObjPool doRetire[{}] {} -> {}", new Object[]{action, wrapper, ok});
        return ok;
    }

    IdentityWrapper<T> doPoll() {
        ObjectConf c;
        IdentityWrapper<T> wrapper = this.stack.poll();
        if (wrapper != null && (c = this.conf.get(wrapper)) != null && !c.isBorrowed()) {
            c.setBorrowed(true);
            return wrapper;
        }
        return null;
    }

    public T borrow() throws TimeoutException {
        IdentityWrapper wrapper;
        long start = System.nanoTime();
        while ((wrapper = Extends.ifNull(this.doPoll(), this::doCreate)) == null || !this.validateHandler.test(wrapper.instance)) {
            long bt = (System.nanoTime() - start) / 1000000L;
            if (this.borrowTimeout > -1L && bt > this.borrowTimeout) {
                log.warn("ObjPool borrow timeout, state: {}", (Object)this);
                throw new TimeoutException("borrow timeout");
            }
            Extends.sleep(bt);
        }
        return wrapper.instance;
    }

    public void recycle(@NonNull T obj) {
        if (obj == null) {
            throw new NullPointerException("obj is marked non-null but is null");
        }
        IdentityWrapper<T> wrapper = new IdentityWrapper<T>(obj);
        if (!this.validateHandler.test(wrapper.instance)) {
            this.doRetire(wrapper, 1);
            return;
        }
        ObjectConf c = this.conf.get(wrapper);
        if (c == null) {
            throw new InvalidException("Object '{}' not belong to this pool", wrapper);
        }
        if (!c.isBorrowed()) {
            throw new InvalidException("Object '{}' has already in this pool", wrapper);
        }
        c.setBorrowed(false);
        if (!this.stack.offer(wrapper)) {
            this.doRetire(wrapper, 2);
            return;
        }
        if (this.passivateHandler != null) {
            this.passivateHandler.accept(wrapper.instance);
        }
    }

    public void retire(@NonNull T obj) {
        if (obj == null) {
            throw new NullPointerException("obj is marked non-null but is null");
        }
        IdentityWrapper<T> wrapper = new IdentityWrapper<T>(obj);
        if (!this.doRetire(wrapper, 10)) {
            throw new InvalidException("Object '{}' not belong to this pool", wrapper);
        }
    }

    public String toString() {
        return "ObjectPool{stack=" + this.stack.size() + ":" + this.conf.size() + ", size=" + this.size + ", poolSize=" + this.minSize + "-" + this.maxSize + ", borrowTimeout=" + this.borrowTimeout + ", idleTimeout=" + this.idleTimeout + ", validationTime=" + this.validationTime + ", leakDetectionThreshold=" + this.leakDetectionThreshold + '}';
    }

    public int getMinSize() {
        return this.minSize;
    }

    public int getMaxSize() {
        return this.maxSize;
    }

    public long getBorrowTimeout() {
        return this.borrowTimeout;
    }

    public long getIdleTimeout() {
        return this.idleTimeout;
    }

    public long getValidationTime() {
        return this.validationTime;
    }

    public long getLeakDetectionThreshold() {
        return this.leakDetectionThreshold;
    }

    static class ObjectConf {
        boolean borrowed;
        long stateTime;
        Thread t;

        ObjectConf() {
        }

        public synchronized boolean isBorrowed() {
            return this.borrowed;
        }

        public synchronized void setBorrowed(boolean borrowed) {
            this.borrowed = borrowed;
            if (this.borrowed) {
                this.t = Thread.currentThread();
            }
            this.stateTime = System.nanoTime();
        }

        public synchronized boolean isIdleTimeout(long idleTimeout) {
            return idleTimeout != 0L && !this.borrowed && (System.nanoTime() - this.stateTime) / 1000000L > idleTimeout;
        }

        public synchronized boolean isLeaked(long threshold) {
            return threshold != 0L && this.borrowed && (System.nanoTime() - this.stateTime) / 1000000L > threshold;
        }
    }
}

