package dev.siris.module;

import com.google.common.base.Charsets;
import org.bukkit.Server;
import org.bukkit.command.Command;
import org.bukkit.command.CommandMap;
import org.bukkit.command.CommandSender;
import org.bukkit.command.PluginCommand;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.generator.ChunkGenerator;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.PluginDescriptionFile;
import org.bukkit.plugin.PluginLoader;
import org.bukkit.plugin.java.JavaPlugin;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Allows for Modularization of components.
 */
public class Module implements Plugin {

    private JavaPlugin owner;
    private String name;
    private boolean enabled;
    private Logger logger;
    protected PluginDescriptionFile description = null;
    private CommandMap commandMap;
    private FileConfiguration newConfig = null;
    private File configFile;
    protected boolean jarLoaded;

    protected void init(JavaPlugin owner, String name, @NotNull ModuleManager manager) {
        // Initialize fields.
        this.owner = owner;
        this.name = name;
        this.logger = new ModuleLogger(this);
        this.setDescription(getDescription());
        this.commandMap = manager.getCommandMap();
        this.configFile = new File(getDataFolder(), "config.yml");
    }

    @NotNull
    @Override
    public File getDataFolder() {
        return new File(getOwner().getDataFolder() + File.separator + getName());
    }

    @NotNull
    @Override
    public PluginDescriptionFile getDescription() {
        return description;
    }

    @NotNull
    @Override
    public FileConfiguration getConfig() {
        if (newConfig == null) {
            reloadConfig();
        }
        return newConfig;
    }

    @Override
    public InputStream getResource(@NotNull String resourcePath) {

        // If this is statically loaded as a bundled class there is no distinct
        // resource folder.
        if (!isJarLoaded()) {
            return owner.getResource(name + File.separator + resourcePath);
        }

        // Get the resource from the jar.
        try {
            URL url = getClass().getClassLoader().getResource(resourcePath);

            if (url == null) {
                return null;
            }

            URLConnection connection = url.openConnection();
            connection.setUseCaches(false);
            return connection.getInputStream();
        } catch (IOException ex) {
            return null;
        }
    }

    @Override
    public void saveConfig() {
        try {
            getConfig().save(configFile);
        } catch (IOException ex) {
            logger.log(Level.SEVERE, "Could not save config to " + configFile, ex);
        }
    }

    @Override
    public void saveDefaultConfig() {
        if (!configFile.exists()) {
            saveResource("config.yml", false);
        }
    }

    @Override
    public void saveResource(@NotNull String resourcePath, boolean replace) {
        this.owner.saveResource(name + File.separator + resourcePath, replace);
    }

    @Override
    public void reloadConfig() {
        newConfig = YamlConfiguration.loadConfiguration(configFile);

        final InputStream defConfigStream = getResource("config.yml");
        if (defConfigStream == null) {
            return;
        }

        newConfig.setDefaults(YamlConfiguration.loadConfiguration(new InputStreamReader(defConfigStream, Charsets.UTF_8)));
    }

    @NotNull
    @Override
    public PluginLoader getPluginLoader() {
        return owner.getPluginLoader();
    }

    @NotNull
    @Override
    public Server getServer() {
        return owner.getServer();
    }

    @Override
    public boolean isEnabled() {
        return enabled;
    }

    /**
     * Sets the Module to be enabled or not.
     *
     * @param enabled The enable state to set.
     */
    public final void setEnabled(boolean enabled) {
        if (this.isEnabled() != enabled) {
            this.enabled = enabled;

            // Check if it was enabled or disabled.
            // Call appropriate methods.
            if (enabled) {
                onEnable();
            } else {
                onDisable();
            }
        }
    }

    // Because these methods should be handled in the Modules themselves,
    // Calling them directly from the Module super class is illegal. These
    // must be overridden and used in subclasses.

    @Override
    public void onDisable() {}

    @Override
    public void onLoad() {}

    @Override
    public void onEnable() {}

    @Override
    public boolean isNaggable() {
        return getOwner().isNaggable();
    }

    @Override
    public void setNaggable(boolean canNag) {
        owner.setNaggable(canNag);
    }

    @Override
    public ChunkGenerator getDefaultWorldGenerator(@NotNull String worldName, String id) {
        return getOwner().getDefaultWorldGenerator(worldName, id);
    }

    @NotNull
    @Override
    public Logger getLogger() {
        return logger;
    }

    @NotNull
    @Override
    public String getName() {
        return name;
    }

    // Once again these methods below need to be handled by subclasses.

    @Override
    public boolean onCommand(@NotNull CommandSender commandSender, @NotNull Command command, @NotNull String s, @NotNull String[] args) {
        return false;
    }

    /**
     * Gets the command with the given name, specific to this plugin. Commands
     * need to be registered in the {@link PluginDescriptionFile#getCommands()
     * PluginDescriptionFile} to exist at runtime.
     *
     * @param name name or alias of the command
     * @return the module command if found, otherwise null
     */
    @Nullable
    @SuppressWarnings("unused")
    public ModuleCommand getCommand(@NotNull String name) {
       return (ModuleCommand) commandMap.getCommand(name);
    }

    @Nullable
    @SuppressWarnings("unused")
    public PluginCommand getOwnerCommand(@NotNull String name) {
        return getOwner().getCommand(name);
    }


    @Override
    public List<String> onTabComplete(@NotNull CommandSender sender, @NotNull Command command, @NotNull String alia, @NotNull String[] args) {
        return null;
    }

    /**
     * Gets the owning plugin of the module.
     *
     * @return The owning plugin.
     */
    public final JavaPlugin getOwner() {
        return owner;
    }

    public InputStream getModuleFile() {
        return getResource("module.yml");
    }

    public void setDescription(PluginDescriptionFile description) {
        this.description = description;
    }

    public boolean isJarLoaded() {
        return jarLoaded;
    }

    protected void setJarLoaded(boolean jarLoaded) {
        this.jarLoaded = jarLoaded;
    }
}
