package dev.siris.module;

import org.bukkit.Bukkit;
import org.bukkit.command.Command;
import org.bukkit.command.PluginCommandYamlParser;
import org.bukkit.plugin.PluginDescriptionFile;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Mostly code based off of {@link PluginCommandYamlParser}, retrofitted for Modules.
 *
 * @see PluginCommandYamlParser
 */
public class ModuleCommandYamlLoader {

    /**
     * Loads all the commands placed in the module's particular <code>module.yml</code>.
     *
     * @param module The module to load commands from.
     * @return A list of {@link ModuleCommand} from the Module.
     */
    public static List<Command> loadCommands(@NotNull Module module) {
        List<Command> pluginCmds = new ArrayList<>();

        Map<String, Map<String, Object>> map = null;

        try {
            map = new PluginDescriptionFile(module.getModuleFile()).getCommands();
        } catch (Exception e) {
            module.getLogger().getParent().warning("No module.yml found for " + module.getName());
        }

        if (map == null) {
            return pluginCmds;
        }

        for (Map.Entry<String, Map<String, Object>> entry : map.entrySet()) {
            if (entry.getKey().contains(":")) {
                Bukkit.getServer().getLogger().severe("Could not load command " + entry.getKey() + " for module " + module.getName() + ": Illegal Characters");
                continue;
            }
            Command newCmd = new ModuleCommand(entry.getKey(), module);
            Object description = entry.getValue().get("description");
            Object usage = entry.getValue().get("usage");
            Object aliases = entry.getValue().get("aliases");
            Object permission = entry.getValue().get("permission");
            Object permissionMessage = entry.getValue().get("permission-message");

            if (description != null) {
                newCmd.setDescription(description.toString());
            }

            if (usage != null) {
                newCmd.setUsage(usage.toString());
            }

            if (aliases != null) {
                List<String> aliasList = new ArrayList<>();

                if (aliases instanceof List) {
                    for (Object o : (List<?>) aliases) {
                        if (o.toString().contains(":")) {
                            Bukkit.getServer().getLogger().severe("Could not load alias " + o.toString() + " for module " + module.getName() + ": Illegal Characters");
                            continue;
                        }
                        aliasList.add(o.toString());
                    }
                } else {
                    if (aliases.toString().contains(":")) {
                        Bukkit.getServer().getLogger().severe("Could not load alias " + aliases.toString() + " for module " + module.getName() + ": Illegal Characters");
                    } else {
                        aliasList.add(aliases.toString());
                    }
                }

                newCmd.setAliases(aliasList);
            }

            if (permission != null) {
                newCmd.setPermission(permission.toString());
            }

            if (permissionMessage != null) {
                newCmd.setPermissionMessage(permissionMessage.toString());
            }

            pluginCmds.add(newCmd);
        }
        return pluginCmds;
    }
}
