package dev.siris.module;

import org.bukkit.Bukkit;
import org.bukkit.command.Command;
import org.bukkit.command.CommandMap;
import org.bukkit.event.HandlerList;
import org.bukkit.plugin.InvalidDescriptionException;
import org.bukkit.plugin.PluginDescriptionFile;
import org.bukkit.plugin.SimplePluginManager;
import org.bukkit.plugin.java.JavaPlugin;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * A class which deals with the enabling, disabling, loading and unloading.
 * of {@link Module}'s.
 *
 * @author Siris
 */
@SuppressWarnings("unused")
public class ModuleManager {

    JavaPlugin plugin;
    private ConcurrentHashMap<String, Module> disabledModules = new ConcurrentHashMap<>();
    private ConcurrentHashMap<String, Module> enabledModules = new ConcurrentHashMap<>();
    private ConcurrentHashMap<String, Module> modules = new ConcurrentHashMap<>();
    private CommandMap commandMap;
    private File moduleDir;

    public ModuleManager(JavaPlugin plugin) {
        this.plugin = plugin;

        // This needed to reflect the CommandMap from the server into this manager.
        // This is required in order to add commands after the main plugin has loaded.
        try {
            // Get the commandMap field.
            final Field f = SimplePluginManager.class.getDeclaredField("commandMap");

            // Make it accessible.
            f.setAccessible(true);

            // Add it to this class.
            this.commandMap = (CommandMap) f.get(Bukkit.getPluginManager());

            // Make the field inaccessible now to prevent unwanted access.
            f.setAccessible(false);
        } catch (Exception e) {
            // If this happens that means the field no longer is available or was renamed in the API.
            plugin.getLogger().warning("[Module Manager] Could not get command map");
        }

        // Defaults to folder named "Modules"
        createModuleDir("modules");

        // Load Modules in.
        loadModules();
    }

    public ModuleManager(JavaPlugin plugin, String moduleDir) {
        this(plugin);
        createModuleDir(moduleDir);

        // Load Modules in and try to enable them.
        loadModules();
    }

    private void createModuleDir(@NotNull String moduleDir) {
        this.moduleDir = new File(plugin.getDataFolder().getPath() + File.separator + moduleDir);

        // Create module directory.
        if (this.moduleDir.exists()) {
            return;
        }

        if (this.getModuleFolder().mkdirs()) {
            plugin.getLogger().info("Created Module Parent Directory.");
        }
    }

    /**
     * Loads the modules from the given class.
     *
     * @param name The name to be used for the module.
     * @param clazz The class of the Module
     * @param <T> Any object that is, or subclasses Module.
     */
    public <T extends Module> void loadModuleFromClass(@NotNull String name, Class<T> clazz) {
        try {

            // Reflect module constructor and instantiate it during runtime.
            T loadedModule = clazz.getDeclaredConstructor().newInstance();
            loadedModule.init(plugin, name, this);

            // Get description file.
            try {
                PluginDescriptionFile description = new PluginDescriptionFile(loadedModule.getModuleFile());
                // Set the description.
                loadedModule.setDescription(description);
            } catch (Exception ignored) {}

            // Invoke onLoad() in module
            loadedModule.onLoad();

            // Now add it the list.
            disabledModules.put(loadedModule.getName(), loadedModule);

            // Add to global list.
            modules.put(loadedModule.getName(), loadedModule);

        } catch (Exception e) {
            plugin.getLogger().warning("Could not load module: " + name + " with class " + clazz);
            e.printStackTrace();
        }

    }

    private <T extends Module> void loadJarModule(@NotNull String name, Class<T> clazz, PluginDescriptionFile descriptionFile) {
        try {

            // Reflect module constructor and instantiate it during runtime.
            T loadedModule = clazz.getDeclaredConstructor().newInstance();
            loadedModule.init(plugin, name, this);
            loadedModule.setJarLoaded(true);
            loadedModule.setDescription(descriptionFile);

            // Get description file.
            try {
                PluginDescriptionFile description = new PluginDescriptionFile(loadedModule.getModuleFile());
                // Set the description.
                loadedModule.setDescription(description);
            } catch (Exception ignored) {}

            // Invoke onLoad() in module
            loadedModule.onLoad();

            // Now add it the list.
            disabledModules.put(loadedModule.getName(), loadedModule);

            // Add to global list.
            modules.put(loadedModule.getName(), loadedModule);

        } catch (Exception e) {
            plugin.getLogger().warning("Could not load module: " + name + " with class " + clazz);
            e.printStackTrace();
        }
    }

    public void loadModules() {

        if (getModuleFolder() == null) {
            return;
        }

        for (File file : Objects.requireNonNull(getModuleFolder().listFiles((dir, name) -> name.endsWith(".jar")))) {
            loadModuleJar(file);
        }

        enableModules();
    }

    @SuppressWarnings("unchecked")
    protected void loadModuleJar(File file) {
        PluginDescriptionFile moduleDesc = null;

        JarFile jar;
        InputStream stream;

        try {
            jar = new JarFile(file);
            JarEntry entry = jar.getJarEntry("module.yml");

            if (entry == null) {
                throw new InvalidDescriptionException(new FileNotFoundException("Jar does not contain module.yml"));
            }

            stream = jar.getInputStream(entry);

            moduleDesc = new PluginDescriptionFile(stream);

        } catch (InvalidDescriptionException | IOException e) {
            e.printStackTrace();
        }

        if (moduleDesc == null) {
            plugin.getLogger().warning("Could not load module.yml from file: " + file + " skipping this module...");
            return;
        }

        try {

            // Convert the file URL to URI standard.
            URL fileURL = file.toURI().toURL();

            // Load jar from URL.
            ClassLoader jarLoader = new URLClassLoader(new URL[]{fileURL},
                    this.getClass().getClassLoader());

            // Load class based on main path given.
            Class<Module> clazz = (Class<Module>) Class.forName(moduleDesc.getMain(), true, jarLoader);
            loadJarModule(moduleDesc.getName(), clazz, moduleDesc);

        } catch (Exception e) {
            plugin.getLogger().severe("Could not load " + moduleDesc.getMain());
            e.printStackTrace();
        }
    }

    /**
     * Enables all the loaded modules. This should be called after {@link ModuleManager#loadModuleFromClass(String, Class)}
     */
    public void enableModules() {
        for (Module module : disabledModules.values()) {
            enableModule(module);
        }
    }

    /**
     * Disables the given module.
     *
     * @param module The module to disable.
     */
    public void disableModule(@NotNull Module module) {
        try {
            plugin.getLogger().info("Disabling " + module.getName() + "...");
            module.setEnabled(false);

            // Remove the module from enabled map.
            enabledModules.remove(module.getName());

            // Add it back to disabled map.
            disabledModules.put(module.getName(), module);

            // Unregister all stuff that was registered.
            HandlerList.unregisterAll(module);
            plugin.getServer().getServicesManager().unregisterAll(plugin);

            plugin.getLogger().info("Disabled " + module.getName());
        } catch (Exception e) {
            plugin.getLogger().info("Could not disable " + module.getName());
            module.setEnabled(true);
            e.printStackTrace();
        }
    }

    /**
     * Disables a module with the given name.
     *
     * @param name The name of the module to disable.
     */
    public void disableModule(@NotNull String name) {
        Module module = getEnabledModule(name);

        if (module == null) {
            plugin.getLogger().warning("Cannot disable module named: " + name);
            return;
        }

        disableModule(module);
    }

    /**
     * Disables all loaded modules.
     */
    public void disableModules() {
        for (Module module : enabledModules.values()) {
            disableModule(module);
        }
    }

    /**
     * Enables the given module.
     *
     * @param module The module to be enabled.
     */
    public void enableModule(@NotNull Module module) {
        try {
            plugin.getLogger().info("Enabling " + module.getName() + "...");

            // Add module commands.
            List<Command> moduleCommands = ModuleCommandYamlLoader.loadCommands(module);

            if (getCommandMap() != null) {
                plugin.getLogger().info("Registering commands for " + module.getName());
                getCommandMap().registerAll(module.getName(), moduleCommands);
            }

            // Remember this could go wrong if one of the modules onEnable() phases fails.
            // So we need to use a try/catch.
            module.setEnabled(true);

            // Add it to the map.
            enabledModules.put(module.getName(), module);

            // Removed it from the disable map.
            disabledModules.remove(module.getName());

            plugin.getLogger().info("Enabled " + module.getName());
        } catch (Exception e) {
            plugin.getLogger().warning("Could not enable " + module.getName());
            module.setEnabled(false);
            e.printStackTrace();
        }
    }

    /**
     * Enables the module with name.
     *
     * @param name The name of the module to enable.
     */
    public void enableModule(@NotNull String name) {
        Module module = getDisabledModule(name);

        if (module == null) {
            plugin.getLogger().warning("Cannot enable module named: " + name);
            return;
        }

        enableModule(module);
    }

    /**
     * Gets the enabled module with the given name.
     *
     * @param name The name of the module.
     * @return The Module with the given name.
     */
    @Nullable
    public Module getEnabledModule(String name) {
        return enabledModules.get(name);
    }

    /**
     * Gets the module (disabled)
     * @param module The module to get.
     *
     * @return The found module.
     */
    @Nullable
    public Module getDisabledModule(Module module) {
        return disabledModules.get(module.getName());
    }

    /**
     * Gets the module with name (disabled)
     * @param name The module with name to get.
     *
     * @return The found module.
     */
    @Nullable
    public Module getDisabledModule(String name) {
        return disabledModules.get(name);
    }

    /**
     * Checks to see if the given module is disabled.
     *
     * @param module The module to check if disabled.
     * @return true if disabled, false otherwise.
     */
    public boolean hasDisabledModule(Module module) {
        return disabledModules.get(module.getName()) != null;
    }

    /**
     * Checks to see if the given module is disabled.
     *
     * @param name The name of the module to check if disabled.
     * @return true if disabled, false otherwise.
     */
    public boolean hasDisabledModule(String name) {
        return getDisabledModule(name) != null;
    }

    /**
     * Reloads the given module.
     *
     * @param module The module to reload. Note that the module must be enabled
     * in order to reload.
     */
    public void reloadModule(@NotNull Module module) {

        if (getEnabledModule(module.getName()) == null) {
            plugin.getLogger().warning("Cannot reload module named: " + module.getName() + " it is disabled.");
            return;
        }

        // If it is turn it off first.
        disableModule(module);

        // Now turn it back on.
        enableModule(module);
    }

    /**
     * Reloads a given module with the name. Note that the module must be enabled
     * in order to reload.
     *
     * @param name The name of the module to reload.
     */
    @SuppressWarnings("unused")
    public void reloadModule(@NotNull String name) {
        Module module = getEnabledModule(name);

        // Check if the module is enabled.
        if (module == null) {
            plugin.getLogger().warning("Cannot reload module named: " + name + " it is disabled.");
            return;
        }

        reloadModule(module);
    }

    /**
     * Returns whether the given module is loaded.
     *
     * @param module The module to check for.
     * @return true if loaded false, otherwise.
     */
    public boolean hasEnabledModule(Module module) {
        return getEnabledModule(module.getName()) != null;
    }


    /**
     * Returns whether the given module with name is loaded.
     *
     * @param name The name of the module to check for.
     * @return true if loaded false, otherwise.
     */
    public boolean hasEnabledModule(String name) {
        return getEnabledModule(name) != null;
    }

    protected CommandMap getCommandMap() {
        return commandMap;
    }

    /**
     * Gets all modules loaded by the manager.
     *
     * @return A list of all loaded modules.
     */
    @NotNull
    public List<Module> getModules() {
        return Collections.unmodifiableList(new ArrayList<>(modules.values()));
    }

    public File getModuleFolder() {
        return moduleDir;
    }

    /**
     * Gets all the active modules from this manager.
     *
     * @return A list of enabled modules.
     */
    @NotNull
    public List<Module> getEnabledModules() {
        return Collections.unmodifiableList(new ArrayList<>(enabledModules.values()));
    }

    /**
     * Gets all the inactive modules from this manager.
     *
     * @return A list of disabled managers.
     */
    @NotNull
    public List<Module> getDisabledModules() {
        return Collections.unmodifiableList(new ArrayList<>(disabledModules.values()));
    }
}
