/*
 * Decompiled with CFR 0.152.
 */
package okhttp3.internal.http2;

import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import okhttp3.internal.Util;
import okhttp3.internal.http2.ErrorCode;
import okhttp3.internal.http2.Header;
import okhttp3.internal.http2.Hpack;
import okhttp3.internal.http2.Http2;
import okhttp3.internal.http2.Settings;
import okio.Buffer;
import okio.BufferedSource;
import okio.ByteString;
import okio.Source;
import okio.Timeout;

final class Http2Reader
implements Closeable {
    static final Logger logger = Logger.getLogger(Http2.class.getName());
    private final BufferedSource source;
    private final ContinuationSource continuation;
    private final boolean client;
    final Hpack.Reader hpackReader;

    Http2Reader(BufferedSource source, boolean client) {
        this.source = source;
        this.client = client;
        this.continuation = new ContinuationSource(this.source);
        this.hpackReader = new Hpack.Reader(4096, this.continuation);
    }

    public void readConnectionPreface(Handler handler) {
        if (this.client) {
            if (!this.nextFrame(true, handler)) {
                throw Http2.ioException("Required SETTINGS preface not received", new Object[0]);
            }
        } else {
            ByteString byteString = this.source.readByteString(Http2.CONNECTION_PREFACE.size());
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(Util.format("<< CONNECTION %s", byteString.hex()));
            }
            if (!Http2.CONNECTION_PREFACE.equals(byteString)) {
                throw Http2.ioException("Expected a connection header but was %s", byteString.utf8());
            }
        }
    }

    public boolean nextFrame(boolean requireSettings, Handler handler) {
        try {
            this.source.require(9L);
        }
        catch (IOException iOException) {
            return false;
        }
        int n = Http2Reader.readMedium(this.source);
        if (n < 0 || n > 16384) {
            throw Http2.ioException("FRAME_SIZE_ERROR: %s", n);
        }
        byte by = (byte)(this.source.readByte() & 0xFF);
        if (requireSettings && by != 4) {
            throw Http2.ioException("Expected a SETTINGS frame but was %s", by);
        }
        byte by2 = (byte)(this.source.readByte() & 0xFF);
        int n2 = this.source.readInt() & Integer.MAX_VALUE;
        if (logger.isLoggable(Level.FINE)) {
            logger.fine(Http2.frameLog(true, n2, n, by, by2));
        }
        switch (by) {
            case 0: {
                this.readData(handler, n, by2, n2);
                break;
            }
            case 1: {
                this.readHeaders(handler, n, by2, n2);
                break;
            }
            case 2: {
                this.readPriority(handler, n, by2, n2);
                break;
            }
            case 3: {
                this.readRstStream(handler, n, by2, n2);
                break;
            }
            case 4: {
                this.readSettings(handler, n, by2, n2);
                break;
            }
            case 5: {
                this.readPushPromise(handler, n, by2, n2);
                break;
            }
            case 6: {
                this.readPing(handler, n, by2, n2);
                break;
            }
            case 7: {
                this.readGoAway(handler, n, by2, n2);
                break;
            }
            case 8: {
                this.readWindowUpdate(handler, n, by2, n2);
                break;
            }
            default: {
                this.source.skip(n);
            }
        }
        return true;
    }

    private void readHeaders(Handler handler, int length, byte flags, int streamId) {
        short s;
        if (streamId == 0) {
            throw Http2.ioException("PROTOCOL_ERROR: TYPE_HEADERS streamId == 0", new Object[0]);
        }
        boolean bl = (flags & 1) != 0;
        short s2 = s = (flags & 8) != 0 ? (short)(this.source.readByte() & 0xFF) : (short)0;
        if ((flags & 0x20) != 0) {
            this.readPriority(handler, streamId);
            length -= 5;
        }
        length = Http2Reader.lengthWithoutPadding(length, flags, s);
        List<Header> list = this.readHeaderBlock(length, s, flags, streamId);
        handler.headers(bl, streamId, -1, list);
    }

    private List<Header> readHeaderBlock(int length, short padding, byte flags, int streamId) {
        this.continuation.length = this.continuation.left = length;
        this.continuation.padding = padding;
        this.continuation.flags = flags;
        this.continuation.streamId = streamId;
        this.hpackReader.readHeaders();
        return this.hpackReader.getAndResetHeaderList();
    }

    private void readData(Handler handler, int length, byte flags, int streamId) {
        boolean bl;
        if (streamId == 0) {
            throw Http2.ioException("PROTOCOL_ERROR: TYPE_DATA streamId == 0", new Object[0]);
        }
        boolean bl2 = (flags & 1) != 0;
        boolean bl3 = bl = (flags & 0x20) != 0;
        if (bl) {
            throw Http2.ioException("PROTOCOL_ERROR: FLAG_COMPRESSED without SETTINGS_COMPRESS_DATA", new Object[0]);
        }
        short s = (flags & 8) != 0 ? (short)(this.source.readByte() & 0xFF) : (short)0;
        length = Http2Reader.lengthWithoutPadding(length, flags, s);
        handler.data(bl2, streamId, this.source, length);
        this.source.skip(s);
    }

    private void readPriority(Handler handler, int length, byte flags, int streamId) {
        if (length != 5) {
            throw Http2.ioException("TYPE_PRIORITY length: %d != 5", length);
        }
        if (streamId == 0) {
            throw Http2.ioException("TYPE_PRIORITY streamId == 0", new Object[0]);
        }
        this.readPriority(handler, streamId);
    }

    private void readPriority(Handler handler, int streamId) {
        int n = this.source.readInt();
        boolean bl = (n & Integer.MIN_VALUE) != 0;
        int n2 = n & Integer.MAX_VALUE;
        int n3 = (this.source.readByte() & 0xFF) + 1;
        handler.priority(streamId, n2, n3, bl);
    }

    private void readRstStream(Handler handler, int length, byte flags, int streamId) {
        if (length != 4) {
            throw Http2.ioException("TYPE_RST_STREAM length: %d != 4", length);
        }
        if (streamId == 0) {
            throw Http2.ioException("TYPE_RST_STREAM streamId == 0", new Object[0]);
        }
        int n = this.source.readInt();
        ErrorCode errorCode = ErrorCode.fromHttp2(n);
        if (errorCode == null) {
            throw Http2.ioException("TYPE_RST_STREAM unexpected error code: %d", n);
        }
        handler.rstStream(streamId, errorCode);
    }

    private void readSettings(Handler handler, int length, byte flags, int streamId) {
        if (streamId != 0) {
            throw Http2.ioException("TYPE_SETTINGS streamId != 0", new Object[0]);
        }
        if ((flags & 1) != 0) {
            if (length != 0) {
                throw Http2.ioException("FRAME_SIZE_ERROR ack frame should be empty!", new Object[0]);
            }
            handler.ackSettings();
            return;
        }
        if (length % 6 != 0) {
            throw Http2.ioException("TYPE_SETTINGS length %% 6 != 0: %s", length);
        }
        Settings settings = new Settings();
        for (int i = 0; i < length; i += 6) {
            int n = this.source.readShort();
            int n2 = this.source.readInt();
            switch (n) {
                case 1: {
                    break;
                }
                case 2: {
                    if (n2 == 0 || n2 == 1) break;
                    throw Http2.ioException("PROTOCOL_ERROR SETTINGS_ENABLE_PUSH != 0 or 1", new Object[0]);
                }
                case 3: {
                    n = 4;
                    break;
                }
                case 4: {
                    n = 7;
                    if (n2 >= 0) break;
                    throw Http2.ioException("PROTOCOL_ERROR SETTINGS_INITIAL_WINDOW_SIZE > 2^31 - 1", new Object[0]);
                }
                case 5: {
                    if (n2 >= 16384 && n2 <= 0xFFFFFF) break;
                    throw Http2.ioException("PROTOCOL_ERROR SETTINGS_MAX_FRAME_SIZE: %s", n2);
                }
                case 6: {
                    break;
                }
            }
            settings.set(n, n2);
        }
        handler.settings(false, settings);
    }

    private void readPushPromise(Handler handler, int length, byte flags, int streamId) {
        if (streamId == 0) {
            throw Http2.ioException("PROTOCOL_ERROR: TYPE_PUSH_PROMISE streamId == 0", new Object[0]);
        }
        short s = (flags & 8) != 0 ? (short)(this.source.readByte() & 0xFF) : (short)0;
        int n = this.source.readInt() & Integer.MAX_VALUE;
        length -= 4;
        length = Http2Reader.lengthWithoutPadding(length, flags, s);
        List<Header> list = this.readHeaderBlock(length, s, flags, streamId);
        handler.pushPromise(streamId, n, list);
    }

    private void readPing(Handler handler, int length, byte flags, int streamId) {
        if (length != 8) {
            throw Http2.ioException("TYPE_PING length != 8: %s", length);
        }
        if (streamId != 0) {
            throw Http2.ioException("TYPE_PING streamId != 0", new Object[0]);
        }
        int n = this.source.readInt();
        int n2 = this.source.readInt();
        boolean bl = (flags & 1) != 0;
        handler.ping(bl, n, n2);
    }

    private void readGoAway(Handler handler, int length, byte flags, int streamId) {
        if (length < 8) {
            throw Http2.ioException("TYPE_GOAWAY length < 8: %s", length);
        }
        if (streamId != 0) {
            throw Http2.ioException("TYPE_GOAWAY streamId != 0", new Object[0]);
        }
        int n = this.source.readInt();
        int n2 = this.source.readInt();
        int n3 = length - 8;
        ErrorCode errorCode = ErrorCode.fromHttp2(n2);
        if (errorCode == null) {
            throw Http2.ioException("TYPE_GOAWAY unexpected error code: %d", n2);
        }
        ByteString byteString = ByteString.EMPTY;
        if (n3 > 0) {
            byteString = this.source.readByteString(n3);
        }
        handler.goAway(n, errorCode, byteString);
    }

    private void readWindowUpdate(Handler handler, int length, byte flags, int streamId) {
        if (length != 4) {
            throw Http2.ioException("TYPE_WINDOW_UPDATE length !=4: %s", length);
        }
        long l = (long)this.source.readInt() & Integer.MAX_VALUE;
        if (l == 0L) {
            throw Http2.ioException("windowSizeIncrement was 0", l);
        }
        handler.windowUpdate(streamId, l);
    }

    @Override
    public void close() {
        this.source.close();
    }

    static int readMedium(BufferedSource source) {
        return (source.readByte() & 0xFF) << 16 | (source.readByte() & 0xFF) << 8 | source.readByte() & 0xFF;
    }

    static int lengthWithoutPadding(int length, byte flags, short padding) {
        if ((flags & 8) != 0) {
            --length;
        }
        if (padding > length) {
            throw Http2.ioException("PROTOCOL_ERROR padding %s > remaining length %s", padding, length);
        }
        return (short)(length - padding);
    }

    static interface Handler {
        public void data(boolean var1, int var2, BufferedSource var3, int var4);

        public void headers(boolean var1, int var2, int var3, List<Header> var4);

        public void rstStream(int var1, ErrorCode var2);

        public void settings(boolean var1, Settings var2);

        public void ackSettings();

        public void ping(boolean var1, int var2, int var3);

        public void goAway(int var1, ErrorCode var2, ByteString var3);

        public void windowUpdate(int var1, long var2);

        public void priority(int var1, int var2, int var3, boolean var4);

        public void pushPromise(int var1, int var2, List<Header> var3);

        public void alternateService(int var1, String var2, ByteString var3, String var4, int var5, long var6);
    }

    static final class ContinuationSource
    implements Source {
        private final BufferedSource source;
        int length;
        byte flags;
        int streamId;
        int left;
        short padding;

        ContinuationSource(BufferedSource source) {
            this.source = source;
        }

        @Override
        public long read(Buffer sink, long byteCount) {
            while (this.left == 0) {
                this.source.skip(this.padding);
                this.padding = 0;
                if ((this.flags & 4) != 0) {
                    return -1L;
                }
                this.readContinuationHeader();
            }
            long l = this.source.read(sink, Math.min(byteCount, (long)this.left));
            if (l == -1L) {
                return -1L;
            }
            this.left = (int)((long)this.left - l);
            return l;
        }

        @Override
        public Timeout timeout() {
            return this.source.timeout();
        }

        @Override
        public void close() {
        }

        private void readContinuationHeader() {
            int n = this.streamId;
            this.length = this.left = Http2Reader.readMedium(this.source);
            byte by = (byte)(this.source.readByte() & 0xFF);
            this.flags = (byte)(this.source.readByte() & 0xFF);
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(Http2.frameLog(true, this.streamId, this.length, by, this.flags));
            }
            this.streamId = this.source.readInt() & Integer.MAX_VALUE;
            if (by != 9) {
                throw Http2.ioException("%s != TYPE_CONTINUATION", by);
            }
            if (this.streamId != n) {
                throw Http2.ioException("TYPE_CONTINUATION streamId changed", new Object[0]);
            }
        }
    }
}

