/*                 _                                            _ _                           _       _
 *  ___ _ __  _ __(_)_ __   __ _       ___  ___  ___ _   _ _ __(_) |_ _   _       _ __   __ _| |_ ___| |__
 * / __| '_ \| '__| | '_ \ / _` |_____/ __|/ _ \/ __| | | | '__| | __| | | |_____| '_ \ / _` | __/ __| '_ \
 * \__ \ |_) | |  | | | | | (_| |_____\__ \  __/ (__| |_| | |  | | |_| |_| |_____| |_) | (_| | || (__| | | |
 * |___/ .__/|_|  |_|_| |_|\__, |     |___/\___|\___|\__,_|_|  |_|\__|\__, |     | .__/ \__,_|\__\___|_| |_|
 *     |_|                 |___/                                      |___/      |_|
 *
 *  https://github.com/yingzhuo/spring-security-patch
 */
package com.github.yingzhuo.spring.security.simpletoken.core;

import com.github.yingzhuo.spring.security.core.AbstractAuthenticationFilter;
import com.github.yingzhuo.spring.security.exception.UserDetailsNotFoundException;
import com.github.yingzhuo.spring.security.simpletoken.errorhandler.SimpleTokenAuthenticationEntryPoint;
import com.github.yingzhuo.spring.security.simpletoken.resolver.SimpleTokenResolver;
import com.github.yingzhuo.spring.security.token.StringToken;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.web.authentication.NullRememberMeServices;
import org.springframework.security.web.authentication.RememberMeServices;
import org.springframework.security.web.authentication.WebAuthenticationDetailsSource;
import org.springframework.util.Assert;
import org.springframework.web.context.request.ServletWebRequest;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Optional;

/**
 * @author 应卓
 * @since 1.3.1
 */
public class SimpleTokenAuthenticationFilter extends AbstractAuthenticationFilter {

    private final SimpleTokenResolver tokenResolver;
    private final UserDetailsFinder userDetailsFinder;
    private RememberMeServices rememberMeServices = new NullRememberMeServices();
    private SimpleTokenAuthenticationEntryPoint simpleTokenAuthenticationEntryPoint;

    public SimpleTokenAuthenticationFilter(SimpleTokenResolver tokenResolver, UserDetailsFinder userDetailsFinder) {
        this.tokenResolver = tokenResolver;
        this.userDetailsFinder = userDetailsFinder;
        Assert.notNull(tokenResolver, () -> null);
        Assert.notNull(userDetailsFinder, () -> null);
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain chain) throws ServletException, IOException {
        try {
            if (authenticationIsRequired()) {
                Optional<StringToken> tokenOption = tokenResolver.resolve(new ServletWebRequest(request, response));
                if (tokenOption.isPresent()) {
                    StringToken oldToken = tokenOption.get();
                    UserDetails userDetails = userDetailsFinder.find(oldToken);
                    if (userDetails != null) {
                        StringToken newToken = new StringToken(userDetails.getAuthorities(), oldToken.toString());
                        newToken.setPrincipal(userDetails);
                        newToken.setAuthenticated(true);
                        newToken.setDetails(new WebAuthenticationDetailsSource().buildDetails(request));
                        SecurityContextHolder.getContext().setAuthentication(newToken);
                        this.rememberMeServices.loginSuccess(request, response, newToken);
                        this.onSuccessfulAuthentication(request, response, newToken);
                    }
                } else {
                    throw new UserDetailsNotFoundException(null);
                }
            }
        } catch (AuthenticationException authException) {
            SecurityContextHolder.clearContext();
            this.rememberMeServices.loginFail(request, response);
            this.onUnsuccessfulAuthentication(request, response, authException);

            if (simpleTokenAuthenticationEntryPoint != null) {
                simpleTokenAuthenticationEntryPoint.commence(request, response, authException);
                return;
            }
        }

        chain.doFilter(request, response);
    }

    @Override
    public void afterPropertiesSet() throws ServletException {
        super.afterPropertiesSet();
    }

    public void setRememberMeServices(RememberMeServices rememberMeServices) {
        this.rememberMeServices = rememberMeServices;
    }

    public void setSimpleTokenAuthenticationEntryPoint(SimpleTokenAuthenticationEntryPoint simpleTokenAuthenticationEntryPoint) {
        this.simpleTokenAuthenticationEntryPoint = simpleTokenAuthenticationEntryPoint;
    }

    public SimpleTokenResolver getTokenResolver() {
        return tokenResolver;
    }

    public UserDetailsFinder getUserDetailsFinder() {
        return userDetailsFinder;
    }

}
