package com.google.appengine.api.labs.modules;

import java.util.Set;
import java.util.concurrent.Future;

/**
 * ModulesService allows the application to fetch information about its
 * own module and version information.  Additionally, the service has the
 * ability to start, stop and change the number of instances associated with a
 * module version.
 *
 */
public interface ModulesService {

  /**
   * Get the name of the current module.
   *
   * @return the name of the module
   */
  String getCurrentModule();

  /**
   * Get the name of the current version.
   *
   * @return the name of the version
   */
  String getCurrentVersion();

  /**
   * Get the id of the current instance.
   *
   * @return current instance id
   * @throws ModulesException when no instance id exists for the current instance
   */
  String getCurrentInstanceId();

  /**
   * Get the set of modules that are available to the application.
   *
   * @return Set of modules available to the application
   * @throws ModulesException when an error occurs
   */
  Set<String> getModules();

  /**
   * Returns a {@link Future} for an asynchronous call to {@link #getModules}.
   */
  Future<Set<String>> getModulesAsync();

  /**
   * Returns the set of versions that are available to the given module.
   *
   * @param module the name of the module
   * @throws ModulesException when input is invalid
   */
  Set<String> getVersions(String module);

  /**
   * Returns a {@link Future} for an asynchronous call to {@link #getVersions}.
   */
  Future<Set<String>> getVersionsAsync(String module);

  /**
   * Returns the name of the default version for the module.
   *
   * @param module the name of the module
   * @throws ModulesException when an error occurs
   */
  String getDefaultVersion(String module);

  /**
   * Returns a {@link Future} for an asynchronous call to {@link #getDefaultVersion}.
   */
  Future<String> getDefaultVersionAsync(String module);

  /**
   * Returns the number of instances that are available to the given module and version.
   *
   * @param module the name of the module
   * @param version the name of the version
   * @throws ModulesException when input is invalid
   */
  long getNumInstances(String module, String version);

  /**
   * Returns a {@link Future} for an asynchronous call to {@link #getNumInstances}.
   */
  Future<Long> getNumInstancesAsync(String module, String version);

  /**
   * Set the number of instances that are available to the given module and version.
   *
   * @param module the name of the module
   * @param version the name of the version
   * @param instances the number of instances to set
   * @throws ModulesException when input is invalid
   */
  void setNumInstances(String module, String version, long instances);

  /**
   * Returns a {@link Future} for an asynchronous call to {@link #setNumInstances}.
   */
  Future<Void> setNumInstancesAsync(String module, String version, long instances);

  /**
   * Start a given module and version.
   *
   * @param module the name of the module
   * @param version the name of the version
   * @throws ModulesException when input or existing state is invalid
   */
  void startModule(String module, String version);

  /**
   * Returns a {@link Future} for an asynchronous call to {@link #startModule}.
   */
  Future<Void> startModuleAsync(String module, String version);

  /**
   * Stop a given module and version.
   *
   * @param module the name of the module
   * @param version the name of the version
   * @throws ModulesException when input or existing state is invalid
   */
  void stopModule(String module, String version);

  /**
   * Returns a {@link Future} for an asynchronous call to {@link #stopModule}.
   */
  Future<Void> stopModuleAsync(String module, String version);

  /**
   * Returns a hostname to use for the given module and version.
   *
   * @param module the name of the module or null to indicate the current module
   * @param version the name of the version or null to indicate the current version
   * @throws ModulesException when input is invalid
   */
  String getModuleHostname(String module, String version);

  /**
   * Returns a {@link Future} for an asynchronous call to
   * {@link #getModuleHostname(String, String)}.
   */
  Future<String> getModuleHostnameAsync(String module, String version);

  /**
   * Returns a hostname to use for the given module and version.
   *
   * @param module the name of the module or null to indicate the current module
   * @param version the name of the version or null to indicate the current version
   * @param instance the id of a particular instance to address
   * @return the hostname of the given instance
   * @throws ModulesException when input is invalid
   */
  String getModuleHostname(String module, String version, int instance);

  /**
   * Returns a {@link Future} for an asynchronous call to
   * {@link #getModuleHostname(String, String, int)}.
   */
  Future<String> getModuleHostnameAsync(String module, String version, int instance);
}
